/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.shield.crypto;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.regex.Pattern;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.common.Base64;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.env.Environment;
import org.elasticsearch.shield.ShieldPlugin;
import org.elasticsearch.shield.authc.support.CharArrays;
import org.elasticsearch.shield.authc.support.SecuredString;
import org.elasticsearch.shield.crypto.CryptoService;
import org.elasticsearch.watcher.FileChangesListener;
import org.elasticsearch.watcher.FileWatcher;
import org.elasticsearch.watcher.ResourceWatcher;
import org.elasticsearch.watcher.ResourceWatcherService;

public class InternalCryptoService
extends AbstractLifecycleComponent<InternalCryptoService>
implements CryptoService {
    public static final String FILE_SETTING = "shield.system_key.file";
    public static final String KEY_ALGO = "HmacSHA512";
    public static final int KEY_SIZE = 1024;
    static final String FILE_NAME = "system_key";
    static final String HMAC_ALGO = "HmacSHA1";
    static final String DEFAULT_ENCRYPTION_ALGORITHM = "AES/CTR/NoPadding";
    static final String DEFAULT_KEY_ALGORITH = "AES";
    static final String ENCRYPTED_TEXT_PREFIX = "::es_encrypted::";
    static final byte[] ENCRYPTED_BYTE_PREFIX = "::es_encrypted::".getBytes(StandardCharsets.UTF_8);
    static final int DEFAULT_KEY_LENGTH = 128;
    static final int RANDOM_KEY_SIZE = 128;
    private static final Pattern SIG_PATTERN = Pattern.compile("^\\$\\$[0-9]+\\$\\$[^\\$]*\\$\\$.+");
    private static final byte[] HKDF_APP_INFO = "es-shield-crypto-service".getBytes(StandardCharsets.UTF_8);
    private final Environment env;
    private final ResourceWatcherService watcherService;
    private final List<CryptoService.Listener> listeners;
    private final SecureRandom secureRandom = new SecureRandom();
    private final String encryptionAlgorithm;
    private final String keyAlgorithm;
    private final int keyLength;
    private final int ivLength;
    private Path keyFile;
    private SecretKey randomKey;
    private String randomKeyBase64;
    private volatile SecretKey encryptionKey;
    private volatile SecretKey systemKey;
    private volatile SecretKey signingKey;

    @Inject
    public InternalCryptoService(Settings settings, Environment env, ResourceWatcherService watcherService) {
        this(settings, env, watcherService, Collections.emptyList());
    }

    InternalCryptoService(Settings settings, Environment env, ResourceWatcherService watcherService, List<CryptoService.Listener> listeners) {
        super(settings);
        this.env = env;
        this.watcherService = watcherService;
        this.listeners = new CopyOnWriteArrayList<CryptoService.Listener>(listeners);
        this.encryptionAlgorithm = settings.get("shield.encryption.algorithm", DEFAULT_ENCRYPTION_ALGORITHM);
        this.keyLength = settings.getAsInt("shield.encryption_key.length", Integer.valueOf(128));
        this.ivLength = this.keyLength / 8;
        this.keyAlgorithm = settings.get("shield.encryption_key.algorithm", DEFAULT_KEY_ALGORITH);
    }

    protected void doStart() throws ElasticsearchException {
        if (this.keyLength % 8 != 0) {
            throw new IllegalArgumentException("invalid key length [" + this.keyLength + "]. value must be a multiple of 8");
        }
        this.loadKeys();
        FileWatcher watcher = new FileWatcher(this.keyFile.getParent());
        watcher.addListener((Object)new FileListener(this.listeners));
        try {
            this.watcherService.add((ResourceWatcher)watcher, ResourceWatcherService.Frequency.HIGH);
        }
        catch (IOException e) {
            throw new ElasticsearchException("failed to start watching system key file [" + this.keyFile.toAbsolutePath() + "]", (Throwable)e, new Object[0]);
        }
    }

    protected void doStop() throws ElasticsearchException {
    }

    protected void doClose() throws ElasticsearchException {
    }

    private void loadKeys() {
        this.keyFile = InternalCryptoService.resolveSystemKey(this.settings, this.env);
        this.systemKey = InternalCryptoService.readSystemKey(this.keyFile);
        this.randomKey = InternalCryptoService.generateSecretKey(128);
        try {
            this.randomKeyBase64 = Base64.encodeBytes((byte[])this.randomKey.getEncoded(), (int)0, (int)this.randomKey.getEncoded().length, (int)16);
        }
        catch (IOException e) {
            throw new ElasticsearchException("failed to encode key data as base64", (Throwable)e, new Object[0]);
        }
        this.signingKey = InternalCryptoService.createSigningKey(this.systemKey, this.randomKey);
        try {
            this.encryptionKey = InternalCryptoService.encryptionKey(this.systemKey, this.keyLength, this.keyAlgorithm);
        }
        catch (NoSuchAlgorithmException nsae) {
            throw new ElasticsearchException("failed to start crypto service. could not load encryption key", (Throwable)nsae, new Object[0]);
        }
    }

    public static byte[] generateKey() {
        return InternalCryptoService.generateSecretKey(1024).getEncoded();
    }

    static SecretKey generateSecretKey(int keyLength) {
        try {
            KeyGenerator generator = KeyGenerator.getInstance(KEY_ALGO);
            generator.init(keyLength);
            return generator.generateKey();
        }
        catch (NoSuchAlgorithmException e) {
            throw new ElasticsearchException("failed to generate key", (Throwable)e, new Object[0]);
        }
    }

    public static Path resolveSystemKey(Settings settings, Environment env) {
        String location = settings.get(FILE_SETTING);
        if (location == null) {
            return ShieldPlugin.resolveConfigFile(env, FILE_NAME);
        }
        return env.binFile().getParent().resolve(location);
    }

    static SecretKey createSigningKey(@Nullable SecretKey systemKey, SecretKey randomKey) {
        assert (randomKey != null);
        if (systemKey != null) {
            return systemKey;
        }
        byte[] keyBytes = HmacSHA1HKDF.extractAndExpand(null, randomKey.getEncoded(), HKDF_APP_INFO, 128);
        assert (keyBytes.length * 8 == 1024);
        return new SecretKeySpec(keyBytes, KEY_ALGO);
    }

    private static SecretKey readSystemKey(Path file) {
        if (!Files.exists(file, new LinkOption[0])) {
            return null;
        }
        try {
            byte[] bytes = Files.readAllBytes(file);
            return new SecretKeySpec(bytes, KEY_ALGO);
        }
        catch (IOException e) {
            throw new ElasticsearchException("could not read secret key", (Throwable)e, new Object[0]);
        }
    }

    @Override
    public String sign(String text) throws IOException {
        return this.sign(text, this.signingKey, this.systemKey);
    }

    @Override
    public String sign(String text, SecretKey signingKey, @Nullable SecretKey systemKey) throws IOException {
        assert (signingKey != null);
        String sigStr = InternalCryptoService.signInternal(text, signingKey);
        return "$$" + sigStr.length() + "$$" + (systemKey == signingKey ? "" : this.randomKeyBase64) + "$$" + sigStr + text;
    }

    @Override
    public String unsignAndVerify(String signedText) {
        return this.unsignAndVerify(signedText, this.systemKey);
    }

    @Override
    public String unsignAndVerify(String signedText, SecretKey systemKey) {
        SecretKey signingKey;
        String text;
        String receivedSignature;
        String base64RandomKey;
        if (!signedText.startsWith("$$") || signedText.length() < 2) {
            throw new IllegalArgumentException("tampered signed text");
        }
        String[] pieces = signedText.split("\\$\\$");
        if (pieces.length != 4 || !pieces[0].equals("")) {
            this.logger.debug("received signed text [{}] with [{}] parts", new Object[]{signedText, pieces.length});
            throw new IllegalArgumentException("tampered signed text");
        }
        try {
            int length = Integer.parseInt(pieces[1]);
            base64RandomKey = pieces[2];
            receivedSignature = pieces[3].substring(0, length);
            text = pieces[3].substring(length);
        }
        catch (Throwable t) {
            this.logger.error("error occurred while parsing signed text", t, new Object[0]);
            throw new IllegalArgumentException("tampered signed text");
        }
        if (base64RandomKey.isEmpty()) {
            if (systemKey == null) {
                this.logger.debug("received signed text without random key information and no system key is present", new Object[0]);
                throw new IllegalArgumentException("tampered signed text");
            }
            signingKey = systemKey;
        } else {
            if (systemKey != null) {
                this.logger.debug("received signed text with random key information but a system key is present", new Object[0]);
                throw new IllegalArgumentException("tampered signed text");
            }
            try {
                byte[] randomKeyBytes = Base64.decode((String)base64RandomKey, (int)16);
                if (randomKeyBytes.length * 8 != 128) {
                    this.logger.debug("incorrect random key data length. received [{}] bytes", new Object[]{randomKeyBytes.length});
                    throw new IllegalArgumentException("tampered signed text");
                }
                SecretKeySpec randomKey = new SecretKeySpec(randomKeyBytes, KEY_ALGO);
                signingKey = InternalCryptoService.createSigningKey(systemKey, randomKey);
            }
            catch (IOException e) {
                this.logger.error("error occurred while decoding key data", (Throwable)e, new Object[0]);
                throw new IllegalStateException("error while verifying the signed text");
            }
        }
        try {
            String sig = InternalCryptoService.signInternal(text, signingKey);
            if (SecuredString.constantTimeEquals(sig, receivedSignature)) {
                return text;
            }
        }
        catch (Throwable t) {
            this.logger.error("error occurred while verifying signed text", t, new Object[0]);
            throw new IllegalStateException("error while verifying the signed text");
        }
        throw new IllegalArgumentException("tampered signed text");
    }

    @Override
    public boolean signed(String text) {
        return SIG_PATTERN.matcher(text).matches();
    }

    @Override
    public char[] encrypt(char[] chars) {
        SecretKey key = this.encryptionKey;
        if (key == null) {
            this.logger.warn("encrypt called without a key, returning plain text. run syskeygen and copy same key to all nodes to enable encryption", new Object[0]);
            return chars;
        }
        byte[] charBytes = CharArrays.toUtf8Bytes(chars);
        String base64 = Base64.encodeBytes((byte[])this.encryptInternal(charBytes, key));
        return ENCRYPTED_TEXT_PREFIX.concat(base64).toCharArray();
    }

    @Override
    public byte[] encrypt(byte[] bytes) {
        SecretKey key = this.encryptionKey;
        if (key == null) {
            this.logger.warn("encrypt called without a key, returning plain text. run syskeygen and copy same key to all nodes to enable encryption", new Object[0]);
            return bytes;
        }
        byte[] encrypted = this.encryptInternal(bytes, key);
        byte[] prefixed = new byte[ENCRYPTED_BYTE_PREFIX.length + encrypted.length];
        System.arraycopy(ENCRYPTED_BYTE_PREFIX, 0, prefixed, 0, ENCRYPTED_BYTE_PREFIX.length);
        System.arraycopy(encrypted, 0, prefixed, ENCRYPTED_BYTE_PREFIX.length, encrypted.length);
        return prefixed;
    }

    @Override
    public char[] decrypt(char[] chars) {
        return this.decrypt(chars, this.encryptionKey);
    }

    @Override
    public char[] decrypt(char[] chars, SecretKey key) {
        byte[] bytes;
        if (key == null) {
            return chars;
        }
        if (!this.encrypted(chars)) {
            return chars;
        }
        String encrypted = new String(chars, ENCRYPTED_TEXT_PREFIX.length(), chars.length - ENCRYPTED_TEXT_PREFIX.length());
        try {
            bytes = Base64.decode((String)encrypted);
        }
        catch (IOException e) {
            throw new ElasticsearchException("unable to decode encrypted data", (Throwable)e, new Object[0]);
        }
        byte[] decrypted = this.decryptInternal(bytes, key);
        return CharArrays.utf8BytesToChars(decrypted);
    }

    @Override
    public byte[] decrypt(byte[] bytes) {
        return this.decrypt(bytes, this.encryptionKey);
    }

    @Override
    public byte[] decrypt(byte[] bytes, SecretKey key) {
        if (key == null) {
            return bytes;
        }
        if (!this.encrypted(bytes)) {
            return bytes;
        }
        byte[] encrypted = Arrays.copyOfRange(bytes, ENCRYPTED_BYTE_PREFIX.length, bytes.length);
        return this.decryptInternal(encrypted, key);
    }

    @Override
    public boolean encrypted(char[] chars) {
        return CharArrays.charsBeginsWith(ENCRYPTED_TEXT_PREFIX, chars);
    }

    @Override
    public boolean encrypted(byte[] bytes) {
        return InternalCryptoService.bytesBeginsWith(ENCRYPTED_BYTE_PREFIX, bytes);
    }

    @Override
    public void register(CryptoService.Listener listener) {
        this.listeners.add(listener);
    }

    @Override
    public boolean encryptionEnabled() {
        return this.encryptionKey != null;
    }

    private byte[] encryptInternal(byte[] bytes, SecretKey key) {
        byte[] iv = new byte[this.ivLength];
        this.secureRandom.nextBytes(iv);
        Cipher cipher = InternalCryptoService.cipher(1, this.encryptionAlgorithm, key, iv);
        try {
            byte[] encrypted = cipher.doFinal(bytes);
            byte[] output = new byte[iv.length + encrypted.length];
            System.arraycopy(iv, 0, output, 0, iv.length);
            System.arraycopy(encrypted, 0, output, iv.length, encrypted.length);
            return output;
        }
        catch (BadPaddingException | IllegalBlockSizeException e) {
            throw new ElasticsearchException("error encrypting data", (Throwable)e, new Object[0]);
        }
    }

    private byte[] decryptInternal(byte[] bytes, SecretKey key) {
        if (bytes.length < this.ivLength) {
            this.logger.error("received data for decryption with size [{}] that is less than IV length [{}]", new Object[]{bytes.length, this.ivLength});
            throw new IllegalArgumentException("invalid data to decrypt");
        }
        byte[] iv = new byte[this.ivLength];
        System.arraycopy(bytes, 0, iv, 0, this.ivLength);
        byte[] data = new byte[bytes.length - this.ivLength];
        System.arraycopy(bytes, this.ivLength, data, 0, bytes.length - this.ivLength);
        Cipher cipher = InternalCryptoService.cipher(2, this.encryptionAlgorithm, key, iv);
        try {
            return cipher.doFinal(data);
        }
        catch (BadPaddingException | IllegalBlockSizeException e) {
            throw new IllegalStateException("error decrypting data", e);
        }
    }

    static Mac createMac(SecretKey key) {
        try {
            Mac mac = HmacSHA1Provider.hmacSHA1();
            mac.init(key);
            return mac;
        }
        catch (Exception e) {
            throw new ElasticsearchException("could not initialize mac", (Throwable)e, new Object[0]);
        }
    }

    private static String signInternal(String text, SecretKey key) throws IOException {
        Mac mac = InternalCryptoService.createMac(key);
        byte[] sig = mac.doFinal(text.getBytes(StandardCharsets.UTF_8));
        return Base64.encodeBytes((byte[])sig, (int)0, (int)sig.length, (int)16);
    }

    static Cipher cipher(int mode, String encryptionAlgorithm, SecretKey key, byte[] initializationVector) {
        try {
            Cipher cipher = Cipher.getInstance(encryptionAlgorithm);
            cipher.init(mode, (Key)key, new IvParameterSpec(initializationVector));
            return cipher;
        }
        catch (Exception e) {
            throw new ElasticsearchException("error creating cipher", (Throwable)e, new Object[0]);
        }
    }

    static SecretKey encryptionKey(SecretKey systemKey, int keyLength, String algorithm) throws NoSuchAlgorithmException {
        if (systemKey == null) {
            return null;
        }
        byte[] bytes = systemKey.getEncoded();
        if (bytes.length * 8 < keyLength) {
            throw new IllegalArgumentException("at least " + keyLength + " bits should be provided as key data");
        }
        MessageDigest messageDigest = MessageDigest.getInstance("SHA-256");
        byte[] digest = messageDigest.digest(bytes);
        assert (digest.length == 32);
        if (digest.length * 8 < keyLength) {
            throw new IllegalArgumentException("requested key length is too large");
        }
        byte[] truncatedDigest = Arrays.copyOfRange(digest, 0, keyLength / 8);
        return new SecretKeySpec(truncatedDigest, algorithm);
    }

    private static boolean bytesBeginsWith(byte[] prefix, byte[] bytes) {
        if (bytes == null || prefix == null) {
            return false;
        }
        if (prefix.length > bytes.length) {
            return false;
        }
        for (int i = 0; i < prefix.length; ++i) {
            if (bytes[i] == prefix[i]) continue;
            return false;
        }
        return true;
    }

    private static class HmacSHA1HKDF {
        private static final int HMAC_SHA1_BYTE_LENGTH = 20;
        private static final String HMAC_SHA1_ALGORITHM = "HmacSHA1";

        private HmacSHA1HKDF() {
        }

        static byte[] extractAndExpand(@Nullable SecretKey salt, byte[] ikm, @Nullable byte[] info, int outputLength) {
            Objects.requireNonNull(ikm, "the input keying material must not be null");
            if (outputLength < 1) {
                throw new IllegalArgumentException("output length must be positive int >= 1");
            }
            if (outputLength > 5100) {
                throw new IllegalArgumentException("output length must be <= 255*20");
            }
            if (salt == null) {
                salt = new SecretKeySpec(new byte[20], "HmacSHA1");
            }
            if (info == null) {
                info = new byte[]{};
            }
            Mac mac = InternalCryptoService.createMac(salt);
            byte[] keyBytes = mac.doFinal(ikm);
            SecretKeySpec pseudoRandomKey = new SecretKeySpec(keyBytes, "HmacSHA1");
            int n = outputLength % 20 == 0 ? outputLength / 20 : outputLength / 20 + 1;
            byte[] hashRound = new byte[]{};
            ByteBuffer generatedBytes = ByteBuffer.allocate(HmacSHA1HKDF.multiplyExact(n, 20));
            try {
                mac.init(pseudoRandomKey);
            }
            catch (InvalidKeyException e) {
                throw new ElasticsearchException("failed to initialize the mac", (Throwable)e, new Object[0]);
            }
            for (int roundNum = 1; roundNum <= n; ++roundNum) {
                mac.reset();
                mac.update(hashRound);
                mac.update(info);
                mac.update((byte)roundNum);
                hashRound = mac.doFinal();
                generatedBytes.put(hashRound);
            }
            byte[] result = new byte[outputLength];
            generatedBytes.rewind();
            generatedBytes.get(result, 0, outputLength);
            return result;
        }

        private static int multiplyExact(int x, int y) {
            long r = (long)x * (long)y;
            if ((long)((int)r) != r) {
                throw new ArithmeticException("integer overflow");
            }
            return (int)r;
        }
    }

    private static class HmacSHA1Provider {
        private static final Mac mac;

        private HmacSHA1Provider() {
        }

        private static Mac hmacSHA1() {
            try {
                Mac hmac = (Mac)mac.clone();
                hmac.reset();
                return hmac;
            }
            catch (CloneNotSupportedException e) {
                throw new IllegalStateException("could not create [HmacSHA1] MAC", e);
            }
        }

        static {
            try {
                mac = Mac.getInstance(InternalCryptoService.HMAC_ALGO);
            }
            catch (NoSuchAlgorithmException e) {
                throw new IllegalStateException("could not create message authentication code instance with algorithm [HmacSHA1]", e);
            }
        }
    }

    private class FileListener
    extends FileChangesListener {
        private final List<CryptoService.Listener> listeners;

        private FileListener(List<CryptoService.Listener> listeners) {
            this.listeners = listeners;
        }

        public void onFileCreated(Path file) {
            if (file.equals(InternalCryptoService.this.keyFile)) {
                SecretKey oldSystemKey = InternalCryptoService.this.systemKey;
                SecretKey oldEncryptionKey = InternalCryptoService.this.encryptionKey;
                InternalCryptoService.this.systemKey = InternalCryptoService.readSystemKey(file);
                InternalCryptoService.this.signingKey = InternalCryptoService.createSigningKey(InternalCryptoService.this.systemKey, InternalCryptoService.this.randomKey);
                try {
                    InternalCryptoService.this.encryptionKey = InternalCryptoService.encryptionKey(InternalCryptoService.this.signingKey, InternalCryptoService.this.keyLength, InternalCryptoService.this.keyAlgorithm);
                }
                catch (NoSuchAlgorithmException nsae) {
                    InternalCryptoService.this.logger.error("could not load encryption key", (Throwable)nsae, new Object[0]);
                    InternalCryptoService.this.encryptionKey = null;
                }
                InternalCryptoService.this.logger.info("system key [{}] has been loaded", new Object[]{file.toAbsolutePath()});
                this.callListeners(oldSystemKey, oldEncryptionKey);
            }
        }

        public void onFileDeleted(Path file) {
            if (file.equals(InternalCryptoService.this.keyFile)) {
                SecretKey oldSystemKey = InternalCryptoService.this.systemKey;
                SecretKey oldEncryptionKey = InternalCryptoService.this.encryptionKey;
                InternalCryptoService.this.logger.error("system key file was removed! as long as the system key file is missing, elasticsearch won't function as expected for some requests (e.g. scroll/scan)", new Object[0]);
                InternalCryptoService.this.systemKey = null;
                InternalCryptoService.this.encryptionKey = null;
                InternalCryptoService.this.signingKey = InternalCryptoService.createSigningKey(InternalCryptoService.this.systemKey, InternalCryptoService.this.randomKey);
                this.callListeners(oldSystemKey, oldEncryptionKey);
            }
        }

        public void onFileChanged(Path file) {
            if (file.equals(InternalCryptoService.this.keyFile)) {
                SecretKey oldSystemKey = InternalCryptoService.this.systemKey;
                SecretKey oldEncryptionKey = InternalCryptoService.this.encryptionKey;
                InternalCryptoService.this.logger.warn("system key file changed!", new Object[0]);
                SecretKey systemKey = InternalCryptoService.readSystemKey(file);
                InternalCryptoService.this.signingKey = InternalCryptoService.createSigningKey(systemKey, InternalCryptoService.this.randomKey);
                try {
                    InternalCryptoService.this.encryptionKey = InternalCryptoService.encryptionKey(InternalCryptoService.this.signingKey, InternalCryptoService.this.keyLength, InternalCryptoService.this.keyAlgorithm);
                }
                catch (NoSuchAlgorithmException nsae) {
                    InternalCryptoService.this.logger.error("could not load encryption key", (Throwable)nsae, new Object[0]);
                    InternalCryptoService.this.encryptionKey = null;
                }
                this.callListeners(oldSystemKey, oldEncryptionKey);
            }
        }

        private void callListeners(SecretKey oldSystemKey, SecretKey oldEncryptionKey) {
            Throwable th = null;
            for (CryptoService.Listener listener : this.listeners) {
                try {
                    listener.onKeyChange(oldSystemKey, oldEncryptionKey);
                }
                catch (Throwable t) {
                    if (th == null) {
                        th = t;
                        continue;
                    }
                    th.addSuppressed(t);
                }
            }
            if (th != null) {
                InternalCryptoService.this.logger.error("called all key change listeners but one or more exceptions was thrown", th, new Object[0]);
                if (th instanceof RuntimeException) {
                    throw (RuntimeException)th;
                }
                if (th instanceof Error) {
                    throw (Error)th;
                }
                throw new RuntimeException(th);
            }
        }
    }
}

