/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.shield.authc.ldap;

import com.google.common.primitives.Ints;
import com.unboundid.ldap.sdk.BindRequest;
import com.unboundid.ldap.sdk.FailoverServerSet;
import com.unboundid.ldap.sdk.Filter;
import com.unboundid.ldap.sdk.GetEntryLDAPConnectionPoolHealthCheck;
import com.unboundid.ldap.sdk.LDAPConnection;
import com.unboundid.ldap.sdk.LDAPConnectionOptions;
import com.unboundid.ldap.sdk.LDAPConnectionPool;
import com.unboundid.ldap.sdk.LDAPConnectionPoolHealthCheck;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.LDAPInterface;
import com.unboundid.ldap.sdk.SearchRequest;
import com.unboundid.ldap.sdk.SearchResultEntry;
import com.unboundid.ldap.sdk.ServerSet;
import com.unboundid.ldap.sdk.SimpleBindRequest;
import java.io.IOException;
import java.util.Locale;
import javax.net.SocketFactory;
import javax.net.ssl.SSLSocketFactory;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.shield.ShieldSettingsFilter;
import org.elasticsearch.shield.authc.RealmConfig;
import org.elasticsearch.shield.authc.ldap.SearchGroupsResolver;
import org.elasticsearch.shield.authc.ldap.UserAttributeGroupsResolver;
import org.elasticsearch.shield.authc.ldap.support.LdapSearchScope;
import org.elasticsearch.shield.authc.ldap.support.LdapSession;
import org.elasticsearch.shield.authc.ldap.support.LdapUtils;
import org.elasticsearch.shield.authc.ldap.support.SessionFactory;
import org.elasticsearch.shield.authc.support.SecuredString;
import org.elasticsearch.shield.ssl.ClientSSLService;
import org.elasticsearch.shield.support.Exceptions;

public class LdapUserSearchSessionFactory
extends SessionFactory {
    static final int DEFAULT_CONNECTION_POOL_SIZE = 20;
    static final int DEFAULT_CONNECTION_POOL_INITIAL_SIZE = 5;
    static final String DEFAULT_USERNAME_ATTRIBUTE = "uid";
    static final TimeValue DEFAULT_HEALTH_CHECK_INTERVAL = TimeValue.timeValueSeconds((long)60L);
    private final LdapSession.GroupsResolver groupResolver;
    private final String userSearchBaseDn;
    private final LdapSearchScope scope;
    private final String userAttribute;
    private final ServerSet serverSet;
    private LDAPConnectionPool connectionPool;

    public LdapUserSearchSessionFactory(RealmConfig config, ClientSSLService sslService) {
        super(config);
        Settings settings = config.settings();
        this.userSearchBaseDn = settings.get("user_search.base_dn");
        if (this.userSearchBaseDn == null) {
            throw new IllegalArgumentException("user_search base_dn must be specified");
        }
        this.scope = LdapSearchScope.resolve(settings.get("user_search.scope"), LdapSearchScope.SUB_TREE);
        this.userAttribute = settings.get("user_search.attribute", DEFAULT_USERNAME_ATTRIBUTE);
        this.serverSet = this.serverSet(settings, sslService);
        this.connectionPool = LdapUserSearchSessionFactory.createConnectionPool(config, this.serverSet, this.timeout, this.logger);
        this.groupResolver = LdapUserSearchSessionFactory.groupResolver(settings);
    }

    private synchronized LDAPConnectionPool connectionPool() throws IOException {
        if (this.connectionPool == null) {
            this.connectionPool = LdapUserSearchSessionFactory.createConnectionPool(this.config, this.serverSet, this.timeout, this.logger);
            if (this.connectionPool == null) {
                throw new IOException("failed to create a connection pool for realm [" + this.config.name() + "] as no LDAP servers are available");
            }
        }
        return this.connectionPool;
    }

    static void filterOutSensitiveSettings(String realmName, ShieldSettingsFilter filter) {
        filter.filterOut("shield.authc.realms." + realmName + ".bind_dn");
        filter.filterOut("shield.authc.realms." + realmName + ".bind_password");
        filter.filterOut("shield.authc.realms." + realmName + "." + "hostname_verification");
    }

    static LDAPConnectionPool createConnectionPool(RealmConfig config, ServerSet serverSet, TimeValue timeout, ESLogger logger) {
        Settings settings = config.settings();
        SimpleBindRequest bindRequest = LdapUserSearchSessionFactory.bindRequest(settings);
        int initialSize = settings.getAsInt("user_search.pool.initial_size", Integer.valueOf(5));
        int size = settings.getAsInt("user_search.pool.size", Integer.valueOf(20));
        try {
            LDAPConnectionPool pool = new LDAPConnectionPool(serverSet, (BindRequest)bindRequest, initialSize, size);
            pool.setRetryFailedOperationsDueToInvalidConnections(true);
            if (settings.getAsBoolean("user_search.pool.health_check.enabled", Boolean.valueOf(true)).booleanValue()) {
                String entryDn = settings.get("user_search.pool.health_check.dn", bindRequest == null ? null : bindRequest.getBindDN());
                if (entryDn == null) {
                    pool.close();
                    throw new IllegalArgumentException("[bind_dn] has not been specified so a value must be specified for [user_search.pool.health_check.dn] or [user_search.pool.health_check.enabled] must be set to false");
                }
                long healthCheckInterval = settings.getAsTime("user_search.pool.health_check.interval", DEFAULT_HEALTH_CHECK_INTERVAL).millis();
                GetEntryLDAPConnectionPoolHealthCheck healthCheck = new GetEntryLDAPConnectionPoolHealthCheck(entryDn, timeout.millis(), false, false, false, true, false);
                pool.setHealthCheck((LDAPConnectionPoolHealthCheck)healthCheck);
                pool.setHealthCheckIntervalMillis(healthCheckInterval);
            }
            return pool;
        }
        catch (LDAPException e) {
            if (logger.isDebugEnabled()) {
                logger.debug("unable to create connection pool for realm [{}]", (Throwable)e, new Object[]{config.name()});
            } else {
                logger.error("unable to create connection pool for realm [{}]: {}", new Object[]{config.name(), e.getMessage()});
            }
            return null;
        }
    }

    static SimpleBindRequest bindRequest(Settings settings) {
        SimpleBindRequest request = null;
        String bindDn = settings.get("bind_dn");
        if (bindDn != null) {
            request = new SimpleBindRequest(bindDn, settings.get("bind_password"));
        }
        return request;
    }

    ServerSet serverSet(Settings settings, ClientSSLService clientSSLService) {
        SSLSocketFactory socketFactory;
        String[] ldapUrls = settings.getAsArray("url");
        if (ldapUrls == null || ldapUrls.length == 0) {
            throw new IllegalArgumentException("missing required LDAP setting [url]");
        }
        SessionFactory.LDAPServers servers = new SessionFactory.LDAPServers(ldapUrls);
        LDAPConnectionOptions options = LdapUserSearchSessionFactory.connectionOptions(settings);
        if (servers.ssl()) {
            socketFactory = clientSSLService.sslSocketFactory();
            if (settings.getAsBoolean("hostname_verification", Boolean.valueOf(true)).booleanValue()) {
                this.logger.debug("using encryption for LDAP connections with hostname verification", new Object[0]);
            } else {
                this.logger.debug("using encryption for LDAP connections without hostname verification", new Object[0]);
            }
        } else {
            socketFactory = null;
        }
        FailoverServerSet serverSet = new FailoverServerSet(servers.addresses(), servers.ports(), (SocketFactory)socketFactory, options);
        serverSet.setReOrderOnFailover(true);
        return serverSet;
    }

    @Override
    public LdapSession session(String user, SecuredString password) throws Exception {
        try {
            String dn = this.findUserDN(user);
            this.tryBind(dn, password);
            return new LdapSession(this.logger, (LDAPInterface)this.connectionPool, dn, this.groupResolver, this.timeout);
        }
        catch (LDAPException e) {
            throw Exceptions.authenticationError("failed to authenticate user [{}]", e, user);
        }
    }

    @Override
    public boolean supportsUnauthenticatedSession() {
        return true;
    }

    @Override
    public LdapSession unauthenticatedSession(String user) throws Exception {
        try {
            String dn = this.findUserDN(user);
            return new LdapSession(this.logger, (LDAPInterface)this.connectionPool, dn, this.groupResolver, this.timeout);
        }
        catch (LDAPException e) {
            throw Exceptions.authenticationError("failed to lookup user [{}]", e, user);
        }
    }

    private String findUserDN(String user) throws Exception {
        SearchRequest request = new SearchRequest(this.userSearchBaseDn, this.scope.scope(), Filter.createEqualityFilter((String)this.userAttribute, (String)Filter.encodeValue((String)user)), Strings.EMPTY_ARRAY);
        request.setTimeLimitSeconds(Ints.checkedCast((long)this.timeout.seconds()));
        LDAPConnectionPool connectionPool = this.connectionPool();
        SearchResultEntry entry = LdapUtils.searchForEntry((LDAPInterface)connectionPool, request, this.logger);
        if (entry == null) {
            throw Exceptions.authenticationError("failed to find user [{}] with search base [{}] scope [{}]", user, this.userSearchBaseDn, this.scope.toString().toLowerCase(Locale.ENGLISH));
        }
        return entry.getDN();
    }

    private void tryBind(String dn, SecuredString password) throws IOException {
        LDAPConnection bindConnection;
        try {
            bindConnection = this.serverSet.getConnection();
        }
        catch (LDAPException e) {
            throw new IOException("unable to connect to any LDAP servers for bind", e);
        }
        try {
            bindConnection.bind(dn, new String(password.internalChars()));
        }
        catch (LDAPException e) {
            throw Exceptions.authenticationError("failed LDAP authentication for DN [{}]", e, dn);
        }
        finally {
            bindConnection.close();
        }
    }

    void shutdown() {
        if (this.connectionPool != null) {
            this.connectionPool.close();
        }
    }

    static LdapSession.GroupsResolver groupResolver(Settings settings) {
        Settings searchSettings = settings.getAsSettings("group_search");
        if (!searchSettings.names().isEmpty()) {
            return new SearchGroupsResolver(searchSettings);
        }
        return new UserAttributeGroupsResolver(settings);
    }
}

