/*
 * Decompiled with CFR 0.152.
 */
package org.logstash.ackedqueue.io;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.zip.CRC32;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.logstash.ackedqueue.SequencedList;
import org.logstash.ackedqueue.io.ByteBufferCleaner;
import org.logstash.ackedqueue.io.ByteBufferCleanerImpl;
import org.logstash.ackedqueue.io.IntVector;
import org.logstash.ackedqueue.io.LongVector;
import org.logstash.ackedqueue.io.PageIO;

public final class MmapPageIOV2
implements PageIO {
    public static final byte VERSION_TWO = 2;
    public static final int VERSION_SIZE = 1;
    public static final int CHECKSUM_SIZE = 4;
    public static final int LENGTH_SIZE = 4;
    public static final int SEQNUM_SIZE = 8;
    public static final int MIN_CAPACITY = 18;
    public static final int HEADER_SIZE = 1;
    public static final boolean VERIFY_CHECKSUM = true;
    private static final Logger LOGGER = LogManager.getLogger(MmapPageIOV2.class);
    private static final ByteBufferCleaner BUFFER_CLEANER = new ByteBufferCleanerImpl();
    private final File file;
    private final CRC32 checkSummer;
    private final IntVector offsetMap;
    private int capacity;
    private long minSeqNum = 0L;
    private int elementCount = 0;
    private int head = 0;
    private byte version = 0;
    private MappedByteBuffer buffer;

    public MmapPageIOV2(int pageNum, int capacity, Path dirPath) {
        this.capacity = capacity;
        this.offsetMap = new IntVector();
        this.checkSummer = new CRC32();
        this.file = dirPath.resolve("page." + pageNum).toFile();
    }

    @Override
    public void open(long minSeqNum, int elementCount) throws IOException {
        this.mapFile();
        this.buffer.position(0);
        this.version = this.buffer.get();
        MmapPageIOV2.validateVersion(this.version);
        this.head = 1;
        this.minSeqNum = minSeqNum;
        this.elementCount = elementCount;
        if (this.elementCount > 0) {
            long seqNum = this.buffer.getLong();
            if (seqNum != this.minSeqNum) {
                throw new IOException(String.format("first seqNum=%d is different than minSeqNum=%d", seqNum, this.minSeqNum));
            }
            this.buffer.position(this.head);
            for (int i = 0; i < this.elementCount; ++i) {
                this.readNextElement(this.minSeqNum + (long)i, false);
            }
        }
    }

    @Override
    public SequencedList<byte[]> read(long seqNum, int limit) throws IOException {
        assert (seqNum >= this.minSeqNum) : String.format("seqNum=%d < minSeqNum=%d", seqNum, this.minSeqNum);
        assert (seqNum <= this.maxSeqNum()) : String.format("seqNum=%d is > maxSeqNum=%d", seqNum, this.maxSeqNum());
        ArrayList<byte[]> elements = new ArrayList<byte[]>();
        LongVector seqNums = new LongVector(limit);
        int offset = this.offsetMap.get((int)(seqNum - this.minSeqNum));
        this.buffer.position(offset);
        for (int i = 0; i < limit; ++i) {
            long readSeqNum = this.buffer.getLong();
            assert (readSeqNum == seqNum + (long)i) : String.format("unmatched seqNum=%d to readSeqNum=%d", seqNum + (long)i, readSeqNum);
            int readLength = this.buffer.getInt();
            byte[] readBytes = new byte[readLength];
            this.buffer.get(readBytes);
            int checksum = this.buffer.getInt();
            int computedChecksum = this.checksum(readBytes);
            if (computedChecksum != checksum) {
                throw new IOException(String.format("computed checksum=%d != checksum for file=%d", computedChecksum, checksum));
            }
            elements.add(readBytes);
            seqNums.add(readSeqNum);
            if (seqNum + (long)i >= this.maxSeqNum()) break;
        }
        return new SequencedList<byte[]>(elements, seqNums);
    }

    @Override
    public void recover() throws IOException {
        this.mapFile();
        this.buffer.position(0);
        this.version = this.buffer.get();
        MmapPageIOV2.validateVersion(this.version);
        this.head = 1;
        this.minSeqNum = this.buffer.getLong();
        this.buffer.position(this.head);
        this.elementCount = 0;
        int i = 0;
        while (true) {
            try {
                this.readNextElement(this.minSeqNum + (long)i, true);
                ++this.elementCount;
            }
            catch (PageIOInvalidElementException e) {
                LOGGER.debug("PageIO recovery for '{}' element index:{}, readNextElement exception: {}", (Object)this.file, (Object)i, (Object)e.getMessage());
                break;
            }
            ++i;
        }
        if (this.elementCount <= 0) {
            this.minSeqNum = 0L;
        }
    }

    @Override
    public void create() throws IOException {
        try (RandomAccessFile raf = new RandomAccessFile(this.file, "rw");){
            this.buffer = raf.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, this.capacity);
        }
        this.buffer.position(0);
        this.buffer.put((byte)2);
        this.buffer.force();
        this.head = 1;
        this.minSeqNum = 0L;
        this.elementCount = 0;
    }

    @Override
    public void deactivate() {
        this.close();
    }

    @Override
    public void activate() throws IOException {
        if (this.buffer == null) {
            try (RandomAccessFile raf = new RandomAccessFile(this.file, "rw");){
                this.buffer = raf.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, this.capacity);
            }
            this.buffer.load();
        }
    }

    @Override
    public void ensurePersisted() {
        this.buffer.force();
    }

    @Override
    public void purge() throws IOException {
        this.close();
        this.head = 0;
        LOGGER.debug("PageIO deleting '{}'", (Object)this.file);
        Files.delete(this.file.toPath());
    }

    @Override
    public void write(byte[] bytes, long seqNum) {
        this.write(bytes, seqNum, bytes.length, this.checksum(bytes));
    }

    @Override
    public void close() {
        if (this.buffer != null) {
            this.buffer.force();
            BUFFER_CLEANER.clean(this.buffer);
        }
        this.buffer = null;
    }

    @Override
    public int getCapacity() {
        return this.capacity;
    }

    @Override
    public long getMinSeqNum() {
        return this.minSeqNum;
    }

    @Override
    public int getElementCount() {
        return this.elementCount;
    }

    @Override
    public boolean hasSpace(int bytes) {
        int bytesLeft = this.capacity - this.head;
        return this.persistedByteCount(bytes) <= bytesLeft;
    }

    @Override
    public int persistedByteCount(int byteCount) {
        return 12 + byteCount + 4;
    }

    @Override
    public int getHead() {
        return this.head;
    }

    @Override
    public boolean isCorruptedPage() throws IOException {
        try (RandomAccessFile raf = new RandomAccessFile(this.file, "rw");){
            boolean bl = raf.length() < 18L;
            return bl;
        }
    }

    private int checksum(byte[] bytes) {
        this.checkSummer.reset();
        this.checkSummer.update(bytes, 0, bytes.length);
        return (int)this.checkSummer.getValue();
    }

    private long maxSeqNum() {
        return this.minSeqNum + (long)this.elementCount - 1L;
    }

    private void mapFile() throws IOException {
        try (RandomAccessFile raf = new RandomAccessFile(this.file, "rw");){
            int pageFileCapacity;
            if (raf.length() > Integer.MAX_VALUE) {
                throw new IOException("Page file too large " + this.file);
            }
            this.capacity = pageFileCapacity = (int)raf.length();
            if (this.capacity < 18) {
                throw new IOException("Page file size is too small to hold elements. This is potentially a queue corruption problem. Run `pqcheck` and `pqrepair` to repair the queue.");
            }
            this.buffer = raf.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, this.capacity);
        }
        this.buffer.load();
    }

    private void readNextElement(long expectedSeqNum, boolean verifyChecksum) throws PageIOInvalidElementException {
        if (this.head + 8 + 4 > this.capacity) {
            throw new PageIOInvalidElementException("cannot read seqNum and length bytes past buffer capacity");
        }
        int elementOffset = this.head;
        int newHead = this.head;
        long seqNum = this.buffer.getLong();
        newHead += 8;
        if (seqNum != expectedSeqNum) {
            throw new PageIOInvalidElementException(String.format("Element seqNum %d is expected to be %d", seqNum, expectedSeqNum));
        }
        int length = this.buffer.getInt();
        newHead += 4;
        if (length <= 0) {
            throw new PageIOInvalidElementException("Element invalid length");
        }
        if (newHead + length + 4 > this.capacity) {
            throw new PageIOInvalidElementException("cannot read element payload and checksum past buffer capacity");
        }
        if (verifyChecksum) {
            this.checkSummer.reset();
            int prevLimit = this.buffer.limit();
            this.buffer.limit(this.buffer.position() + length);
            this.checkSummer.update(this.buffer);
            this.buffer.limit(prevLimit);
            int checksum = this.buffer.getInt();
            int computedChecksum = (int)this.checkSummer.getValue();
            if (computedChecksum != checksum) {
                throw new PageIOInvalidElementException("Element invalid checksum");
            }
        }
        this.offsetMap.add(elementOffset);
        this.head = newHead + length + 4;
        this.buffer.position(this.head);
    }

    private int write(byte[] bytes, long seqNum, int length, int checksum) {
        assert (this.offsetMap.size() == this.elementCount) : String.format("offsetMap size=%d != elementCount=%d", this.offsetMap.size(), this.elementCount);
        int initialHead = this.head;
        this.buffer.position(this.head);
        this.buffer.putLong(seqNum);
        this.buffer.putInt(length);
        this.buffer.put(bytes);
        this.buffer.putInt(checksum);
        this.head += this.persistedByteCount(bytes.length);
        assert (this.head == this.buffer.position()) : String.format("head=%d != buffer position=%d", this.head, this.buffer.position());
        if (this.elementCount <= 0) {
            this.minSeqNum = seqNum;
        }
        this.offsetMap.add(initialHead);
        ++this.elementCount;
        return initialHead;
    }

    private static void validateVersion(byte version) throws PageIOInvalidVersionException {
        if (version != 2) {
            throw new PageIOInvalidVersionException(String.format("Expected page version=%d but found version=%d", (byte)2, version));
        }
    }

    public String toString() {
        return "MmapPageIOV2{file=" + this.file + ", capacity=" + this.capacity + ", minSeqNum=" + this.minSeqNum + ", elementCount=" + this.elementCount + ", head=" + this.head + "}";
    }

    public static final class PageIOInvalidVersionException
    extends IOException {
        private static final long serialVersionUID = 1L;

        public PageIOInvalidVersionException(String message) {
            super(message);
        }
    }

    public static final class PageIOInvalidElementException
    extends IOException {
        private static final long serialVersionUID = 1L;

        public PageIOInvalidElementException(String message) {
            super(message);
        }
    }
}

