/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.ingest.geoip;

import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.GZIPInputStream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.client.internal.OriginSettingClient;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ProjectState;
import org.elasticsearch.cluster.metadata.IndexAbstraction;
import org.elasticsearch.cluster.metadata.ProjectId;
import org.elasticsearch.cluster.metadata.ProjectMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.project.ProjectResolver;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.logging.HeaderWarning;
import org.elasticsearch.core.CheckedConsumer;
import org.elasticsearch.core.CheckedRunnable;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.env.Environment;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.query.TermQueryBuilder;
import org.elasticsearch.ingest.IngestService;
import org.elasticsearch.ingest.geoip.ConfigDatabases;
import org.elasticsearch.ingest.geoip.DatabaseReaderLazyLoader;
import org.elasticsearch.ingest.geoip.EnterpriseGeoIpTaskState;
import org.elasticsearch.ingest.geoip.GeoIpCache;
import org.elasticsearch.ingest.geoip.GeoIpDownloaderTaskExecutor;
import org.elasticsearch.ingest.geoip.GeoIpProcessor;
import org.elasticsearch.ingest.geoip.GeoIpTaskState;
import org.elasticsearch.ingest.geoip.IpDatabase;
import org.elasticsearch.ingest.geoip.IpDatabaseProvider;
import org.elasticsearch.ingest.geoip.MMDBUtil;
import org.elasticsearch.ingest.geoip.TarInputStream;
import org.elasticsearch.ingest.geoip.stats.CacheStats;
import org.elasticsearch.persistent.PersistentTaskState;
import org.elasticsearch.persistent.PersistentTasksCustomMetadata;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.watcher.ResourceWatcherService;

public final class DatabaseNodeService
implements IpDatabaseProvider {
    private static final Logger logger = LogManager.getLogger(DatabaseNodeService.class);
    private final Client client;
    private final GeoIpCache cache;
    private final Path geoipTmpBaseDirectory;
    private Path geoipTmpDirectory;
    private final ConfigDatabases configDatabases;
    private final Consumer<Runnable> genericExecutor;
    private final ClusterService clusterService;
    private final IngestService ingestService;
    private final ProjectResolver projectResolver;
    private final ConcurrentMap<ProjectId, ConcurrentMap<String, DatabaseReaderLazyLoader>> databases = new ConcurrentHashMap<ProjectId, ConcurrentMap<String, DatabaseReaderLazyLoader>>();

    DatabaseNodeService(Environment environment, Client client, GeoIpCache cache, Consumer<Runnable> genericExecutor, ClusterService clusterService, IngestService ingestService, ProjectResolver projectResolver) {
        this(environment.tmpDir(), new OriginSettingClient(client, "ingest"), cache, new ConfigDatabases(environment, cache), genericExecutor, clusterService, ingestService, projectResolver);
    }

    DatabaseNodeService(Path tmpDir, Client client, GeoIpCache cache, ConfigDatabases configDatabases, Consumer<Runnable> genericExecutor, ClusterService clusterService, IngestService ingestService, ProjectResolver projectResolver) {
        this.client = client;
        this.cache = cache;
        this.geoipTmpBaseDirectory = tmpDir.resolve("geoip-databases");
        this.configDatabases = configDatabases;
        this.genericExecutor = genericExecutor;
        this.clusterService = clusterService;
        this.ingestService = ingestService;
        this.projectResolver = projectResolver;
    }

    public void initialize(String nodeId, ResourceWatcherService resourceWatcher) throws IOException {
        this.configDatabases.initialize(resourceWatcher);
        this.geoipTmpDirectory = this.geoipTmpBaseDirectory.resolve(nodeId);
        Files.walkFileTree(this.geoipTmpDirectory, (FileVisitor<? super Path>)new FileVisitor<Path>(this){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                try {
                    logger.info("deleting stale file [{}]", (Object)file);
                    Files.deleteIfExists(file);
                }
                catch (IOException e) {
                    logger.warn("can't delete stale file [" + String.valueOf(file) + "]", (Throwable)e);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException e) {
                if (!(e instanceof NoSuchFileException)) {
                    logger.warn("can't delete stale file [" + String.valueOf(file) + "]", (Throwable)e);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                return FileVisitResult.CONTINUE;
            }
        });
        if (!Files.exists(this.geoipTmpDirectory, new LinkOption[0])) {
            Files.createDirectories(this.geoipTmpDirectory, new FileAttribute[0]);
        }
        logger.debug("initialized database node service, using geoip-databases directory [{}]", (Object)this.geoipTmpDirectory);
        this.clusterService.addListener(event -> this.checkDatabases(event.state()));
    }

    @Override
    public Boolean isValid(ProjectId projectId, String databaseFile) {
        ClusterState currentState = this.clusterService.state();
        ProjectMetadata projectMetadata = currentState.metadata().getProject(projectId);
        GeoIpTaskState state = GeoIpTaskState.getGeoIpTaskState(projectMetadata, GeoIpDownloaderTaskExecutor.getTaskId(projectId, this.projectResolver.supportsMultipleProjects()));
        if (state == null) {
            return true;
        }
        GeoIpTaskState.Metadata metadata = state.getDatabases().get(databaseFile);
        if (metadata == null) {
            return true;
        }
        boolean valid = metadata.isNewEnough(currentState.metadata().settings());
        if (valid && metadata.isCloseToExpiration()) {
            HeaderWarning.addWarning("database [{}] was not updated for over 25 days, geoip processor will stop working if there is no update for 30 days", databaseFile);
        }
        return valid;
    }

    DatabaseReaderLazyLoader getDatabaseReaderLazyLoader(ProjectId projectId, String name) {
        DatabaseReaderLazyLoader instance;
        do {
            if ((instance = this.getProjectLazyLoader(projectId, name)) != null) continue;
            instance = this.configDatabases.getDatabase(name);
        } while (instance != null && !instance.preLookup());
        return instance;
    }

    @Override
    public IpDatabase getDatabase(ProjectId projectId, String name) {
        return this.getDatabaseReaderLazyLoader(projectId, name);
    }

    List<DatabaseReaderLazyLoader> getAllDatabases() {
        ArrayList<DatabaseReaderLazyLoader> all = new ArrayList<DatabaseReaderLazyLoader>(this.configDatabases.getConfigDatabases().values());
        this.databases.forEach((key, value) -> all.addAll(value.values()));
        return all;
    }

    DatabaseReaderLazyLoader get(ProjectId projectId, String key) {
        return (DatabaseReaderLazyLoader)this.databases.computeIfAbsent(projectId, k -> new ConcurrentHashMap()).get(key);
    }

    public void shutdown() throws IOException {
        List<ShutdownCloseable> loadersToShutdown = this.databases.values().stream().flatMap(map -> map.values().stream()).map(ShutdownCloseable::new).toList();
        this.databases.clear();
        IOUtils.close(loadersToShutdown);
    }

    void checkDatabases(ClusterState state) {
        if (state.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            return;
        }
        DiscoveryNode localNode = state.nodes().getLocalNode();
        if (!localNode.isIngestNode()) {
            logger.trace("Not checking databases because local node is not ingest node");
            return;
        }
        state.forEachProject(this::checkDatabases);
    }

    void checkDatabases(ProjectState projectState) {
        ProjectId projectId = projectState.projectId();
        ProjectMetadata projectMetadata = projectState.metadata();
        PersistentTasksCustomMetadata persistentTasks = (PersistentTasksCustomMetadata)projectMetadata.custom("persistent_tasks");
        if (persistentTasks == null) {
            logger.trace("Not checking databases for project [{}] because persistent tasks are null", (Object)projectId);
            return;
        }
        IndexAbstraction databasesAbstraction = (IndexAbstraction)projectMetadata.getIndicesLookup().get(".geoip_databases");
        if (databasesAbstraction == null) {
            logger.trace("Not checking databases because geoip databases index does not exist for project [{}]", (Object)projectId);
            return;
        }
        Index databasesIndex = databasesAbstraction.getWriteIndex();
        IndexRoutingTable databasesIndexRT = projectState.routingTable().index(databasesIndex);
        if (databasesIndexRT == null || !databasesIndexRT.allPrimaryShardsActive()) {
            logger.trace("Not checking databases because geoip databases index does not have all active primary shards for project [{}]", (Object)projectId);
            return;
        }
        ArrayList<Tuple> validMetadatas = new ArrayList<Tuple>();
        PersistentTaskState taskState = GeoIpTaskState.getGeoIpTaskState(projectMetadata, GeoIpDownloaderTaskExecutor.getTaskId(projectId, this.projectResolver.supportsMultipleProjects()));
        if (taskState == null) {
            taskState = GeoIpTaskState.EMPTY;
        }
        validMetadatas.addAll(((GeoIpTaskState)taskState).getDatabases().entrySet().stream().filter(e -> ((GeoIpTaskState.Metadata)e.getValue()).isNewEnough(projectState.cluster().metadata().settings())).map(entry -> Tuple.tuple((String)entry.getKey(), (GeoIpTaskState.Metadata)entry.getValue())).toList());
        taskState = EnterpriseGeoIpTaskState.getEnterpriseGeoIpTaskState(projectState.metadata());
        if (taskState == null) {
            taskState = EnterpriseGeoIpTaskState.EMPTY;
        }
        validMetadatas.addAll(((EnterpriseGeoIpTaskState)taskState).getDatabases().entrySet().stream().filter(e -> ((GeoIpTaskState.Metadata)e.getValue()).isNewEnough(projectState.cluster().metadata().settings())).map(entry -> Tuple.tuple((String)entry.getKey(), (GeoIpTaskState.Metadata)entry.getValue())).toList());
        validMetadatas.forEach(e -> {
            String localMd5;
            String name = (String)e.v1();
            GeoIpTaskState.Metadata metadata = (GeoIpTaskState.Metadata)e.v2();
            DatabaseReaderLazyLoader reference = this.getProjectLazyLoader(projectId, name);
            String remoteMd5 = metadata.md5();
            String string = localMd5 = reference != null ? reference.getMd5() : null;
            if (Objects.equals(localMd5, remoteMd5)) {
                logger.debug("[{}] is up to date [{}] with cluster state [{}]", (Object)name, (Object)localMd5, (Object)remoteMd5);
                return;
            }
            try {
                this.retrieveAndUpdateDatabase(projectId, name, metadata);
            }
            catch (Exception ex) {
                logger.warn(() -> "failed to retrieve database [" + name + "]", (Throwable)ex);
            }
        });
        if (this.databases.containsKey(projectId)) {
            HashSet<String> staleDatabases = new HashSet<String>(((ConcurrentMap)this.databases.get(projectId)).keySet());
            staleDatabases.removeAll(validMetadatas.stream().map(Tuple::v1).collect(Collectors.toSet()));
            this.removeStaleEntries(projectId, staleDatabases);
        }
    }

    void retrieveAndUpdateDatabase(ProjectId projectId, String databaseName, GeoIpTaskState.Metadata metadata) throws IOException {
        Path retrievedFile;
        logger.trace("retrieving database [{}]", (Object)databaseName);
        String recordedMd5 = metadata.md5();
        Path databaseTmpDirectory = this.getDatabaseTmpDirectory(projectId);
        try {
            retrievedFile = Files.createFile(databaseTmpDirectory.resolve(databaseName + ".tmp.retrieved"), new FileAttribute[0]);
        }
        catch (FileAlreadyExistsException e) {
            logger.debug("database update [{}] already in progress, skipping...", (Object)databaseName);
            return;
        }
        DatabaseReaderLazyLoader lazyLoader = this.getProjectLazyLoader(projectId, databaseName);
        if (lazyLoader != null && recordedMd5.equals(lazyLoader.getMd5())) {
            logger.debug("deleting tmp file because database [{}] has already been updated.", (Object)databaseName);
            Files.delete(retrievedFile);
            return;
        }
        Path databaseTmpFile = Files.createFile(databaseTmpDirectory.resolve(databaseName + ".tmp"), new FileAttribute[0]);
        logger.debug("retrieving database [{}] from [{}] to [{}]", (Object)databaseName, (Object)".geoip_databases", (Object)retrievedFile);
        this.retrieveDatabase(projectId, databaseName, recordedMd5, metadata, bytes -> Files.write(retrievedFile, bytes, StandardOpenOption.APPEND), () -> {
            Path databaseFile = databaseTmpDirectory.resolve(databaseName);
            boolean isTarGz = MMDBUtil.isGzip(retrievedFile);
            if (isTarGz) {
                logger.debug("decompressing [{}]", (Object)retrievedFile.getFileName());
                try (TarInputStream is = new TarInputStream(new GZIPInputStream(Files.newInputStream(retrievedFile, new OpenOption[0]), 8192));){
                    TarInputStream.TarEntry entry;
                    while ((entry = is.getNextEntry()) != null) {
                        if (entry.notFile()) continue;
                        String name = entry.name().substring(entry.name().lastIndexOf(47) + 1);
                        if (name.startsWith(databaseName)) {
                            Files.copy(is, databaseTmpFile, StandardCopyOption.REPLACE_EXISTING);
                            continue;
                        }
                        Files.copy(is, databaseTmpDirectory.resolve(databaseName + "_" + name), StandardCopyOption.REPLACE_EXISTING);
                    }
                }
            } else {
                Files.copy(retrievedFile, databaseTmpFile, StandardCopyOption.REPLACE_EXISTING);
            }
            logger.debug("moving database from [{}] to [{}]", (Object)databaseTmpFile, (Object)databaseFile);
            Files.move(databaseTmpFile, databaseFile, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
            this.updateDatabase(projectId, databaseName, recordedMd5, databaseFile);
            Files.delete(retrievedFile);
        }, failure -> {
            logger.warn(() -> "failed to retrieve database [" + databaseName + "]", (Throwable)failure);
            try {
                Files.deleteIfExists(databaseTmpFile);
                Files.deleteIfExists(retrievedFile);
            }
            catch (IOException ioe) {
                ioe.addSuppressed((Throwable)failure);
                logger.warn("unable to delete tmp database file after failure", (Throwable)ioe);
            }
        });
    }

    void updateDatabase(ProjectId projectId, String databaseFileName, String recordedMd5, Path file) {
        try {
            logger.debug("starting reload of changed database file [{}]", (Object)file);
            DatabaseReaderLazyLoader loader = new DatabaseReaderLazyLoader(projectId, this.cache, file, recordedMd5);
            DatabaseReaderLazyLoader existing = this.databases.computeIfAbsent(projectId, k -> new ConcurrentHashMap()).put(databaseFileName, loader);
            if (existing != null) {
                existing.shutdown();
            } else {
                Predicate<GeoIpProcessor.DatabaseUnavailableProcessor> predicate = p -> databaseFileName.equals(p.getDatabaseName());
                Collection<String> ids = this.ingestService.getPipelineWithProcessorType(projectId, GeoIpProcessor.DatabaseUnavailableProcessor.class, predicate);
                if (!ids.isEmpty()) {
                    logger.debug("pipelines [{}] found to reload", ids);
                    for (String id : ids) {
                        try {
                            this.ingestService.reloadPipeline(projectId, id);
                            logger.trace("successfully reloaded pipeline [{}] after downloading of database [{}] for the first time", (Object)id, (Object)databaseFileName);
                        }
                        catch (Exception e) {
                            logger.debug(() -> Strings.format("failed to reload pipeline [%s] after downloading of database [%s]", id, databaseFileName), (Throwable)e);
                        }
                    }
                } else {
                    logger.debug("no pipelines found to reload");
                }
            }
            logger.info("successfully loaded database file [{}]", (Object)file.getFileName());
        }
        catch (Exception e) {
            logger.warn(() -> "failed to update database [" + databaseFileName + "]", (Throwable)e);
        }
    }

    void removeStaleEntries(ProjectId projectId, Collection<String> staleEntries) {
        ConcurrentMap projectLoaders = (ConcurrentMap)this.databases.get(projectId);
        assert (projectLoaders != null);
        for (String staleEntry : staleEntries) {
            try {
                logger.debug("database [{}] for project [{}] no longer exists, cleaning up...", (Object)staleEntry, (Object)projectId);
                DatabaseReaderLazyLoader existing = (DatabaseReaderLazyLoader)projectLoaders.remove(staleEntry);
                assert (existing != null);
                existing.shutdown(true);
            }
            catch (Exception e) {
                logger.warn(() -> "failed to clean database [" + staleEntry + "] for project [" + String.valueOf(projectId) + "]", (Throwable)e);
            }
        }
    }

    void retrieveDatabase(ProjectId projectId, String databaseName, String expectedMd5, GeoIpTaskState.Metadata metadata, CheckedConsumer<byte[], IOException> chunkConsumer, CheckedRunnable<Exception> completedHandler, Consumer<Exception> failureHandler) {
        this.genericExecutor.accept(() -> {
            MessageDigest md = MessageDigests.md5();
            int firstChunk = metadata.firstChunk();
            int lastChunk = metadata.lastChunk();
            try {
                for (int chunk = firstChunk; chunk <= lastChunk; ++chunk) {
                    SearchRequest searchRequest = new SearchRequest(".geoip_databases");
                    String id = String.format(Locale.ROOT, "%s_%d_%d", databaseName, chunk, metadata.lastUpdate());
                    searchRequest.source().query(new TermQueryBuilder("_id", id));
                    SearchResponse searchResponse = this.client.projectClient(projectId).search(searchRequest).actionGet();
                    try {
                        SearchHit[] hits = searchResponse.getHits().getHits();
                        if (searchResponse.getHits().getHits().length == 0) {
                            failureHandler.accept(new ResourceNotFoundException("chunk document with id [" + id + "] not found", new Object[0]));
                            return;
                        }
                        byte[] data = (byte[])hits[0].getSourceAsMap().get("data");
                        md.update(data);
                        chunkConsumer.accept(data);
                        continue;
                    }
                    finally {
                        searchResponse.decRef();
                    }
                }
                String actualMd5 = MessageDigests.toHexString(md.digest());
                if (Objects.equals(expectedMd5, actualMd5)) {
                    completedHandler.run();
                } else {
                    failureHandler.accept(new RuntimeException("expected md5 hash [" + expectedMd5 + "], but got md5 hash [" + actualMd5 + "]"));
                }
            }
            catch (Exception e) {
                failureHandler.accept(e);
            }
        });
    }

    public Set<String> getAvailableDatabases(ProjectId projectId) {
        ConcurrentMap loaders = (ConcurrentMap)this.databases.get(projectId);
        return loaders == null ? Set.of() : Set.copyOf(loaders.keySet());
    }

    public Set<String> getConfigDatabases() {
        return this.configDatabases.getConfigDatabases().keySet();
    }

    public Map<String, ConfigDatabaseDetail> getConfigDatabasesDetail() {
        HashMap<String, ConfigDatabaseDetail> allDatabases = new HashMap<String, ConfigDatabaseDetail>();
        for (Map.Entry<String, DatabaseReaderLazyLoader> entry : this.configDatabases.getConfigDatabases().entrySet()) {
            DatabaseReaderLazyLoader databaseReaderLazyLoader = entry.getValue();
            try {
                allDatabases.put(entry.getKey(), new ConfigDatabaseDetail(entry.getKey(), databaseReaderLazyLoader.getMd5(), databaseReaderLazyLoader.getBuildDateMillis(), databaseReaderLazyLoader.getDatabaseType()));
            }
            catch (FileNotFoundException e) {
                logger.trace(org.elasticsearch.common.Strings.format("Unable to get metadata for config database %s", entry.getKey()), (Throwable)e);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        return allDatabases;
    }

    public Set<String> getFilesInTemp(ProjectId projectId) {
        Set<String> set;
        block8: {
            Stream<Path> files = Files.list(this.getDatabaseTmpDirectory(projectId));
            try {
                set = files.map(Path::getFileName).map(Path::toString).collect(Collectors.toSet());
                if (files == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (files != null) {
                        try {
                            files.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
            files.close();
        }
        return set;
    }

    public CacheStats getCacheStats() {
        return this.cache.getCacheStats();
    }

    private DatabaseReaderLazyLoader getProjectLazyLoader(ProjectId projectId, String databaseName) {
        return (DatabaseReaderLazyLoader)this.databases.computeIfAbsent(projectId, k -> new ConcurrentHashMap()).get(databaseName);
    }

    private Path getDatabaseTmpDirectory(ProjectId projectId) {
        Path path = this.projectResolver.supportsMultipleProjects() ? this.geoipTmpDirectory.resolve(projectId.toString()) : this.geoipTmpDirectory;
        try {
            if (!Files.exists(path, new LinkOption[0])) {
                Files.createDirectories(path, new FileAttribute[0]);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to create geoip tmp directory for project [" + String.valueOf(projectId) + "]", e);
        }
        return path;
    }

    public record ConfigDatabaseDetail(String name, @Nullable String md5, @Nullable Long buildDateInMillis, @Nullable String type) {
    }

    private record ShutdownCloseable(DatabaseReaderLazyLoader loader) implements Closeable
    {
        @Override
        public void close() throws IOException {
            if (this.loader != null) {
                this.loader.shutdown();
            }
        }
    }
}

