/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.lucene.index.ElasticsearchDirectoryReader;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.fielddata.FieldDataContext;
import org.elasticsearch.index.fielddata.IndexFieldData;
import org.elasticsearch.index.fielddata.IndexFieldDataService;
import org.elasticsearch.index.mapper.MappedFieldType;
import org.elasticsearch.index.mapper.MapperService;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.threadpool.ThreadPool;

public final class IndexWarmer {
    private static final Logger logger = LogManager.getLogger(IndexWarmer.class);
    private final List<Listener> listeners;

    IndexWarmer(ThreadPool threadPool, IndexFieldDataService indexFieldDataService, Listener ... listeners) {
        ArrayList<FieldDataWarmer> list = new ArrayList<FieldDataWarmer>();
        ExecutorService executor = threadPool.executor("warmer");
        list.add(new FieldDataWarmer(executor, indexFieldDataService));
        Collections.addAll(list, listeners);
        this.listeners = Collections.unmodifiableList(list);
    }

    void warm(ElasticsearchDirectoryReader reader, IndexShard shard, IndexSettings settings) {
        if (shard.state() == IndexShardState.CLOSED) {
            return;
        }
        if (!settings.isWarmerEnabled()) {
            return;
        }
        if (logger.isTraceEnabled()) {
            logger.trace("{} top warming [{}]", (Object)shard.shardId(), (Object)reader);
        }
        shard.warmerService().onPreWarm();
        long time = System.nanoTime();
        ArrayList<TerminationHandle> terminationHandles = new ArrayList<TerminationHandle>();
        for (Listener listener : this.listeners) {
            terminationHandles.add(listener.warmReader(shard, reader));
        }
        for (TerminationHandle terminationHandle : terminationHandles) {
            try {
                terminationHandle.awaitTermination();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                logger.warn("top warming has been interrupted", (Throwable)e);
                break;
            }
        }
        long took = System.nanoTime() - time;
        shard.warmerService().onPostWarm(took);
        if (shard.warmerService().logger().isTraceEnabled()) {
            shard.warmerService().logger().trace("top warming took [{}]", (Object)new TimeValue(took, TimeUnit.NANOSECONDS));
        }
    }

    private static class FieldDataWarmer
    implements Listener {
        private final Executor executor;
        private final IndexFieldDataService indexFieldDataService;

        FieldDataWarmer(Executor executor, IndexFieldDataService indexFieldDataService) {
            this.executor = executor;
            this.indexFieldDataService = indexFieldDataService;
        }

        @Override
        public TerminationHandle warmReader(IndexShard indexShard, ElasticsearchDirectoryReader reader) {
            MapperService mapperService = indexShard.mapperService();
            HashMap<String, MappedFieldType> warmUpGlobalOrdinals = new HashMap<String, MappedFieldType>();
            for (MappedFieldType fieldType : mapperService.getEagerGlobalOrdinalsFields()) {
                String indexName = fieldType.name();
                warmUpGlobalOrdinals.put(indexName, fieldType);
            }
            CountDownLatch latch = new CountDownLatch(warmUpGlobalOrdinals.size());
            for (MappedFieldType fieldType : warmUpGlobalOrdinals.values()) {
                this.executor.execute(() -> {
                    try {
                        long start = System.nanoTime();
                        IndexFieldData.Global ifd = (IndexFieldData.Global)this.indexFieldDataService.getForField(fieldType, FieldDataContext.noRuntimeFields(mapperService.index().getName(), "index warming"));
                        IndexFieldData global = ifd.loadGlobal(reader);
                        if (!reader.leaves().isEmpty()) {
                            global.load(reader.leaves().get(0));
                        }
                        if (indexShard.warmerService().logger().isTraceEnabled()) {
                            indexShard.warmerService().logger().trace("warmed global ordinals for [{}], took [{}]", (Object)fieldType.name(), (Object)TimeValue.timeValueNanos(System.nanoTime() - start));
                        }
                    }
                    catch (Exception e) {
                        indexShard.warmerService().logger().warn(() -> "failed to warm-up global ordinals for [" + fieldType.name() + "]", (Throwable)e);
                    }
                    finally {
                        latch.countDown();
                    }
                });
            }
            return () -> latch.await();
        }
    }

    public static interface Listener {
        public TerminationHandle warmReader(IndexShard var1, ElasticsearchDirectoryReader var2);
    }

    public static interface TerminationHandle {
        public static final TerminationHandle NO_WAIT = () -> {};

        public void awaitTermination() throws InterruptedException;
    }
}

