/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.network;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Locale;
import org.elasticsearch.common.network.CIDRUtils;
import org.elasticsearch.common.network.NetworkAddress;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.xcontent.XContentString;

public class InetAddresses {
    private static final int IPV4_PART_COUNT = 4;
    private static final int IPV6_PART_COUNT = 8;
    private static final char[] HEX_DIGITS = "0123456789abcdef".toCharArray();

    public static boolean isInetAddress(String ipString) {
        byte[] utf8Bytes = ipString.getBytes(StandardCharsets.UTF_8);
        return InetAddresses.ipStringToBytes(utf8Bytes, 0, utf8Bytes.length, false) != null;
    }

    public static String getIpOrHost(String ipString) {
        byte[] utf8Bytes = ipString.getBytes(StandardCharsets.UTF_8);
        byte[] bytes = InetAddresses.ipStringToBytes(utf8Bytes, 0, utf8Bytes.length, false);
        if (bytes == null) {
            return ipString;
        }
        return NetworkAddress.format(InetAddresses.bytesToInetAddress(bytes));
    }

    public static byte[] encodeAsIpv6(XContentString ipString) {
        XContentString.UTF8Bytes uft8Bytes = ipString.bytes();
        byte[] address = InetAddresses.ipStringToBytes(uft8Bytes.bytes(), uft8Bytes.offset(), uft8Bytes.length(), true);
        if (address == null) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "'%s' is not an IP string literal.", ipString.string()));
        }
        return address;
    }

    private static byte[] ipStringToBytes(byte[] ipUtf8, int offset, int length, boolean asIpv6) {
        int indexOfLastColon = -1;
        boolean hasDot = false;
        for (int i = offset; i < offset + length; ++i) {
            byte c = ipUtf8[i];
            if ((c & 0x80) != 0) {
                return null;
            }
            if (c == 46) {
                hasDot = true;
                continue;
            }
            if (c == 58) {
                if (hasDot) {
                    return null;
                }
                indexOfLastColon = i;
                continue;
            }
            if (c != 37) continue;
            if (i == offset + length - 1) {
                return null;
            }
            length = i - offset;
            break;
        }
        if (indexOfLastColon >= 0) {
            if (hasDot) {
                if ((ipUtf8 = InetAddresses.convertDottedQuadToHex(ipUtf8, offset, length, indexOfLastColon)) == null) {
                    return null;
                }
                offset = 0;
                length = ipUtf8.length;
            }
            return InetAddresses.textToNumericFormatV6(ipUtf8, offset, length);
        }
        if (hasDot) {
            return InetAddresses.textToNumericFormatV4(ipUtf8, offset, length, asIpv6);
        }
        return null;
    }

    private static byte[] convertDottedQuadToHex(byte[] ipUtf8, int offset, int length, int indexOfLastColon) {
        int quadOffset = indexOfLastColon - offset + 1;
        assert (quadOffset >= 0) : "Expected at least one colon in dotted quad IPv6 address";
        byte[] quad = InetAddresses.textToNumericFormatV4(ipUtf8, offset + quadOffset, length - quadOffset, false);
        if (quad == null) {
            return null;
        }
        byte[] result = new byte[quadOffset + 9];
        System.arraycopy(ipUtf8, offset, result, 0, quadOffset);
        InetAddresses.appendHexBytes(result, quadOffset, quad[0], quad[1]);
        result[quadOffset + 4] = 58;
        InetAddresses.appendHexBytes(result, quadOffset + 5, quad[2], quad[3]);
        return result;
    }

    static void appendHexBytes(byte[] result, int offset, byte b1, byte b2) {
        result[offset] = (byte)HEX_DIGITS[(b1 & 0xF0) >> 4];
        result[offset + 1] = (byte)HEX_DIGITS[b1 & 0xF];
        result[offset + 2] = (byte)HEX_DIGITS[(b2 & 0xF0) >> 4];
        result[offset + 3] = (byte)HEX_DIGITS[b2 & 0xF];
    }

    private static byte[] textToNumericFormatV4(byte[] ipUtf8, int offset, int length, boolean asIpv6) {
        byte octet;
        byte[] bytes;
        if (asIpv6) {
            bytes = new byte[16];
            System.arraycopy(CIDRUtils.IPV4_PREFIX, 0, bytes, 0, CIDRUtils.IPV4_PREFIX.length);
            octet = (byte)CIDRUtils.IPV4_PREFIX.length;
        } else {
            bytes = new byte[4];
            octet = 0;
        }
        int digits = 0;
        int current = 0;
        for (int i = offset; i < offset + length; ++i) {
            byte c = ipUtf8[i];
            if (c == 46) {
                if (octet >= bytes.length || digits == 0 || current > 255) {
                    return null;
                }
                byte by = octet;
                octet = (byte)(octet + 1);
                bytes[by] = (byte)current;
                current = 0;
                digits = 0;
                continue;
            }
            if (c >= 48 && c <= 57) {
                if (digits != 0 && current == 0) {
                    return null;
                }
                current = current * 10 + (c - 48);
                digits = (byte)(digits + 1);
                continue;
            }
            return null;
        }
        if (octet != bytes.length - 1 || digits == 0 || current > 255) {
            return null;
        }
        bytes[octet] = (byte)current;
        return bytes;
    }

    private static byte[] textToNumericFormatV6(byte[] ipUtf8, int offset, int length) {
        if (length < 2) {
            return null;
        }
        if (ipUtf8[offset] == 58 && ipUtf8[offset + 1] != 58) {
            return null;
        }
        if (ipUtf8[offset + length - 1] == 58 && ipUtf8[offset + length - 2] != 58) {
            return null;
        }
        ByteBuffer bytes = ByteBuffer.allocate(16);
        int compressedHextetIndex = -1;
        int hextetIndex = 0;
        int currentHextetStart = offset;
        int currentHextet = 0;
        for (int i = offset; i < offset + length; ++i) {
            byte c = ipUtf8[i];
            if (c == 58) {
                if (currentHextetStart == i) {
                    if (compressedHextetIndex >= 0 && i != offset + 1) {
                        return null;
                    }
                    compressedHextetIndex = hextetIndex;
                } else {
                    if (!InetAddresses.putHextet(bytes, currentHextet)) {
                        return null;
                    }
                    currentHextet = 0;
                    ++hextetIndex;
                }
                currentHextetStart = i + 1;
                continue;
            }
            if (c >= 48 && c <= 57) {
                currentHextet = currentHextet * 16 + (c - 48);
                continue;
            }
            if (c >= 97 && c <= 102) {
                currentHextet = currentHextet * 16 + (c - 97 + 10);
                continue;
            }
            if (c >= 65 && c <= 70) {
                currentHextet = currentHextet * 16 + (c - 65 + 10);
                continue;
            }
            return null;
        }
        if (currentHextetStart < offset + length) {
            if (!InetAddresses.putHextet(bytes, currentHextet)) {
                return null;
            }
            ++hextetIndex;
        }
        if (compressedHextetIndex >= 0) {
            if (hextetIndex >= 8) {
                return null;
            }
            InetAddresses.shiftHextetsRight(bytes, compressedHextetIndex, hextetIndex);
        } else if (hextetIndex != 8) {
            return null;
        }
        return bytes.array();
    }

    private static void shiftHextetsRight(ByteBuffer bytes, int start, int end) {
        int shift = 8 - end;
        for (int hextetIndexToShift = end - 1; hextetIndexToShift >= start; --hextetIndexToShift) {
            int bytesIndexBeforeShift = hextetIndexToShift * 2;
            short hextetToShift = bytes.getShort(bytesIndexBeforeShift);
            bytes.putShort(bytesIndexBeforeShift, (short)0);
            bytes.putShort(bytesIndexBeforeShift + shift * 2, hextetToShift);
        }
    }

    private static boolean putHextet(ByteBuffer buf, int hextet) {
        if (buf.remaining() < 2) {
            return false;
        }
        if (hextet > 65535) {
            return false;
        }
        buf.putShort((short)hextet);
        return true;
    }

    public static String toUriString(InetAddress ip) {
        if (ip instanceof Inet6Address) {
            return "[" + InetAddresses.toAddrString(ip) + "]";
        }
        return InetAddresses.toAddrString(ip);
    }

    @SuppressForbidden(reason="java.net.Inet4Address#getHostAddress() is fine no need to duplicate its code")
    public static String toAddrString(InetAddress ip) {
        if (ip == null) {
            throw new NullPointerException("ip");
        }
        if (ip instanceof Inet4Address) {
            Inet4Address inet4Address = (Inet4Address)ip;
            return inet4Address.getHostAddress();
        }
        if (!(ip instanceof Inet6Address)) {
            throw new IllegalArgumentException("ip");
        }
        byte[] bytes = ip.getAddress();
        int[] hextets = new int[8];
        for (int i = 0; i < hextets.length; ++i) {
            hextets[i] = (bytes[2 * i] & 0xFF) << 8 | bytes[2 * i + 1] & 0xFF;
        }
        InetAddresses.compressLongestRunOfZeroes(hextets);
        return InetAddresses.hextetsToIPv6String(hextets);
    }

    private static void compressLongestRunOfZeroes(int[] hextets) {
        int bestRunStart = -1;
        int bestRunLength = -1;
        int runStart = -1;
        for (int i = 0; i < hextets.length + 1; ++i) {
            if (i < hextets.length && hextets[i] == 0) {
                if (runStart >= 0) continue;
                runStart = i;
                continue;
            }
            if (runStart < 0) continue;
            int runLength = i - runStart;
            if (runLength > bestRunLength) {
                bestRunStart = runStart;
                bestRunLength = runLength;
            }
            runStart = -1;
        }
        if (bestRunLength >= 2) {
            Arrays.fill(hextets, bestRunStart, bestRunStart + bestRunLength, -1);
        }
    }

    private static String hextetsToIPv6String(int[] hextets) {
        StringBuilder buf = new StringBuilder(39);
        boolean lastWasNumber = false;
        for (int i = 0; i < hextets.length; ++i) {
            boolean thisIsNumber;
            boolean bl = thisIsNumber = hextets[i] >= 0;
            if (thisIsNumber) {
                if (lastWasNumber) {
                    buf.append(':');
                }
                buf.append(Integer.toHexString(hextets[i]));
            } else if (i == 0 || lastWasNumber) {
                buf.append("::");
            }
            lastWasNumber = thisIsNumber;
        }
        return buf.toString();
    }

    public static InetAddress forString(String ipString) {
        byte[] utf8Bytes = ipString.getBytes(StandardCharsets.UTF_8);
        return InetAddresses.forString(utf8Bytes, 0, utf8Bytes.length);
    }

    public static InetAddress forString(XContentString.UTF8Bytes bytes) {
        return InetAddresses.forString(bytes.bytes(), bytes.offset(), bytes.length());
    }

    public static InetAddress forString(byte[] ipUtf8, int offset, int length) {
        byte[] addr = InetAddresses.ipStringToBytes(ipUtf8, offset, length, false);
        if (addr == null) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "'%s' is not an IP string literal.", new String(ipUtf8, offset, length, StandardCharsets.UTF_8)));
        }
        return InetAddresses.bytesToInetAddress(addr);
    }

    private static InetAddress bytesToInetAddress(byte[] addr) {
        try {
            return InetAddress.getByAddress(addr);
        }
        catch (UnknownHostException e) {
            throw new AssertionError((Object)e);
        }
    }

    public static Tuple<InetAddress, Integer> parseCidr(String maskedAddress) {
        String[] fields = maskedAddress.split("/");
        if (fields.length == 2) {
            String addressString = fields[0];
            InetAddress address = InetAddresses.forString(addressString);
            if (addressString.contains(":") && address.getAddress().length == 4) {
                throw new IllegalArgumentException("CIDR notation is not allowed with IPv6-mapped IPv4 address [" + addressString + " as it introduces ambiguity as to whether the prefix length should be interpreted as a v4 prefix length or a v6 prefix length");
            }
            int prefixLength = Integer.parseInt(fields[1]);
            if (prefixLength < 0 || prefixLength > 8 * address.getAddress().length) {
                throw new IllegalArgumentException("Illegal prefix length [" + prefixLength + "] in [" + maskedAddress + "]. Must be 0-32 for IPv4 ranges, 0-128 for IPv6 ranges");
            }
            return new Tuple<InetAddress, Integer>(address, prefixLength);
        }
        throw new IllegalArgumentException("Expected [ip/prefix] but was [" + maskedAddress + "]");
    }

    public static String toCidrString(InetAddress address, int prefixLength) {
        return InetAddresses.toAddrString(address) + "/" + prefixLength;
    }

    public static IpRange parseIpRangeFromCidr(String maskedAddress) {
        Tuple<InetAddress, Integer> cidr = InetAddresses.parseCidr(maskedAddress);
        byte[] lower = cidr.v1().getAddress();
        byte[] upper = (byte[])lower.clone();
        for (int i = cidr.v2().intValue(); i < 8 * lower.length; ++i) {
            int m = 1 << 7 - (i & 7);
            int n = i >> 3;
            lower[n] = (byte)(lower[n] & (byte)(~m));
            int n2 = i >> 3;
            upper[n2] = (byte)(upper[n2] | (byte)m);
        }
        try {
            return new IpRange(InetAddress.getByAddress(lower), InetAddress.getByAddress(upper));
        }
        catch (UnknownHostException bogus) {
            throw new AssertionError((Object)bogus);
        }
    }

    public record IpRange(InetAddress lowerBound, InetAddress upperBound) {
    }
}

