/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.io.stream;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Objects;
import org.apache.lucene.util.BytesRef;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.bytes.CompositeBytesReference;
import org.elasticsearch.common.bytes.ReleasableBytesReference;
import org.elasticsearch.common.io.stream.BytesStream;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.recycler.Recycler;
import org.elasticsearch.common.util.ByteUtils;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Releasables;

public class RecyclerBytesStreamOutput
extends BytesStream
implements Releasable {
    private ArrayList<Recycler.V<BytesRef>> pages = new ArrayList(8);
    private final Recycler<BytesRef> recycler;
    private final int pageSize;
    private int pageIndex = -1;
    private int currentCapacity = 0;
    private BytesRef currentBytesRef;
    private int currentPageOffset;

    public RecyclerBytesStreamOutput(Recycler<BytesRef> recycler) {
        this.recycler = recycler;
        this.currentPageOffset = this.pageSize = recycler.pageSize();
        this.ensureCapacityFromPosition(1L);
        this.nextPage();
    }

    @Override
    public long position() {
        return (long)this.pageSize * (long)this.pageIndex + (long)this.currentPageOffset;
    }

    @Override
    public void writeByte(byte b) {
        int currentPageOffset = this.currentPageOffset;
        if (1 > this.pageSize - currentPageOffset) {
            this.ensureCapacity(1);
            this.nextPage();
            currentPageOffset = 0;
        }
        BytesRef currentPage = this.currentBytesRef;
        int destOffset = currentPage.offset + currentPageOffset;
        currentPage.bytes[destOffset] = b;
        this.currentPageOffset = currentPageOffset + 1;
    }

    @Override
    public void write(byte[] b) throws IOException {
        this.writeBytes(b, 0, b.length);
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        this.writeBytes(b, off, len);
    }

    @Override
    public void writeBytes(byte[] b, int offset, int length) {
        int toCopyThisLoop;
        if (length == 0) {
            return;
        }
        Objects.checkFromIndexSize(offset, length, b.length);
        int currentPageOffset = this.currentPageOffset;
        BytesRef currentPage = this.currentBytesRef;
        if (length > this.pageSize - currentPageOffset) {
            this.ensureCapacity(length);
        }
        int bytesToCopy = length;
        int srcOff = offset;
        while (true) {
            toCopyThisLoop = Math.min(this.pageSize - currentPageOffset, bytesToCopy);
            int destOffset = currentPage.offset + currentPageOffset;
            System.arraycopy(b, srcOff, currentPage.bytes, destOffset, toCopyThisLoop);
            srcOff += toCopyThisLoop;
            if ((bytesToCopy -= toCopyThisLoop) <= 0) break;
            currentPageOffset = 0;
            currentPage = this.pages.get(++this.pageIndex).v();
        }
        this.currentPageOffset = currentPageOffset += toCopyThisLoop;
        this.currentBytesRef = currentPage;
    }

    @Override
    public void writeVInt(int i) throws IOException {
        int currentPageOffset = this.currentPageOffset;
        int remainingBytesInPage = this.pageSize - currentPageOffset;
        if ((i & 0xFFFFFF80) == 0) {
            if (1 > remainingBytesInPage) {
                super.writeVInt(i);
            } else {
                BytesRef currentPage = this.currentBytesRef;
                currentPage.bytes[currentPage.offset + currentPageOffset] = (byte)i;
                this.currentPageOffset = currentPageOffset + 1;
            }
            return;
        }
        int bytesNeeded = RecyclerBytesStreamOutput.vIntLength(i);
        if (bytesNeeded > remainingBytesInPage) {
            super.writeVInt(i);
        } else {
            BytesRef currentPage = this.currentBytesRef;
            this.putVInt(i, bytesNeeded, currentPage.bytes, currentPage.offset + currentPageOffset);
            this.currentPageOffset = currentPageOffset + bytesNeeded;
        }
    }

    public static int vIntLength(int value) {
        int leadingZeros = Integer.numberOfLeadingZeros(value);
        if (leadingZeros >= 25) {
            return 1;
        }
        if (leadingZeros >= 18) {
            return 2;
        }
        if (leadingZeros >= 11) {
            return 3;
        }
        if (leadingZeros >= 4) {
            return 4;
        }
        return 5;
    }

    private void putVInt(int i, int bytesNeeded, byte[] page, int offset) {
        if (bytesNeeded == 1) {
            page[offset] = (byte)i;
        } else {
            RecyclerBytesStreamOutput.putMultiByteVInt(page, i, offset);
        }
    }

    @Override
    public void writeInt(int i) throws IOException {
        int currentPageOffset = this.currentPageOffset;
        if (4 > this.pageSize - currentPageOffset) {
            super.writeInt(i);
        } else {
            BytesRef currentPage = this.currentBytesRef;
            ByteUtils.writeIntBE(i, currentPage.bytes, currentPage.offset + currentPageOffset);
            this.currentPageOffset = currentPageOffset + 4;
        }
    }

    @Override
    public void writeIntLE(int i) throws IOException {
        int currentPageOffset = this.currentPageOffset;
        if (4 > this.pageSize - currentPageOffset) {
            super.writeIntLE(i);
        } else {
            BytesRef currentPage = this.currentBytesRef;
            ByteUtils.writeIntLE(i, currentPage.bytes, currentPage.offset + currentPageOffset);
            this.currentPageOffset = currentPageOffset + 4;
        }
    }

    @Override
    public void writeLong(long i) throws IOException {
        int currentPageOffset = this.currentPageOffset;
        if (8 > this.pageSize - currentPageOffset) {
            super.writeLong(i);
        } else {
            BytesRef currentPage = this.currentBytesRef;
            ByteUtils.writeLongBE(i, currentPage.bytes, currentPage.offset + currentPageOffset);
            this.currentPageOffset = currentPageOffset + 8;
        }
    }

    @Override
    public void writeLongLE(long i) throws IOException {
        int currentPageOffset = this.currentPageOffset;
        if (8 > this.pageSize - currentPageOffset) {
            super.writeLongLE(i);
        } else {
            BytesRef currentPage = this.currentBytesRef;
            ByteUtils.writeLongLE(i, currentPage.bytes, currentPage.offset + currentPageOffset);
            this.currentPageOffset = currentPageOffset + 8;
        }
    }

    @Override
    public void legacyWriteWithSizePrefix(Writeable writeable) throws IOException {
        try (RecyclerBytesStreamOutput tmp = new RecyclerBytesStreamOutput(this.recycler);){
            tmp.setTransportVersion(this.getTransportVersion());
            writeable.writeTo(tmp);
            int size = tmp.size();
            this.writeVInt(size);
            int tmpPage = 0;
            while (size > 0) {
                Recycler.V<BytesRef> p = tmp.pages.get(tmpPage);
                BytesRef b = p.v();
                int writeSize = Math.min(size, b.length);
                this.writeBytes(b.bytes, b.offset, writeSize);
                ((Recycler.V)tmp.pages.set(tmpPage, null)).close();
                size -= writeSize;
                ++tmpPage;
            }
        }
    }

    public BytesRef tryGetPageForWrite(int bytes) {
        int beforePageOffset = this.currentPageOffset;
        if (bytes <= this.pageSize - beforePageOffset) {
            BytesRef currentPage = this.currentBytesRef;
            BytesRef bytesRef = new BytesRef(currentPage.bytes, currentPage.offset + beforePageOffset, bytes);
            this.currentPageOffset = beforePageOffset + bytes;
            return bytesRef;
        }
        return null;
    }

    @Override
    public void writeString(String str) throws IOException {
        int bytesNeededForVInt;
        int currentPageOffset = this.currentPageOffset;
        int charCount = str.length();
        if (charCount * 3 + (bytesNeededForVInt = RecyclerBytesStreamOutput.vIntLength(charCount)) > this.pageSize - currentPageOffset) {
            super.writeString(str);
            return;
        }
        BytesRef currentPage = this.currentBytesRef;
        int offset = currentPage.offset + currentPageOffset;
        byte[] buffer = currentPage.bytes;
        this.putVInt(charCount, bytesNeededForVInt, currentPage.bytes, offset);
        offset += bytesNeededForVInt;
        for (int i = 0; i < charCount; ++i) {
            char c = str.charAt(i);
            if (c <= '\u007f') {
                buffer[offset++] = (byte)c;
                continue;
            }
            if (c > '\u07ff') {
                buffer[offset++] = (byte)(0xE0 | c >> 12 & 0xF);
                buffer[offset++] = (byte)(0x80 | c >> 6 & 0x3F);
                buffer[offset++] = (byte)(0x80 | c >> 0 & 0x3F);
                continue;
            }
            buffer[offset++] = (byte)(0xC0 | c >> 6 & 0x1F);
            buffer[offset++] = (byte)(0x80 | c >> 0 & 0x3F);
        }
        this.currentPageOffset = offset - currentPage.offset;
    }

    @Override
    public void flush() {
    }

    @Override
    public void seek(long position) {
        this.ensureCapacityFromPosition(position);
        if (position > 0L) {
            int offsetInPage = (int)(position % (long)this.pageSize);
            int pageIndex = (int)position / this.pageSize;
            if (offsetInPage == 0) {
                this.pageIndex = pageIndex - 1;
                this.currentPageOffset = this.pageSize;
            } else {
                this.pageIndex = pageIndex;
                this.currentPageOffset = offsetInPage;
            }
        } else {
            assert (position == 0L);
            this.pageIndex = 0;
            this.currentPageOffset = 0;
        }
        this.currentBytesRef = this.pages.get(this.pageIndex).v();
    }

    public void skip(int length) {
        this.seek(this.position() + (long)length);
    }

    @Override
    public void close() {
        ArrayList<Recycler.V<BytesRef>> pages = this.pages;
        if (pages != null) {
            this.closeFields();
            Releasables.close(pages);
        }
    }

    public ReleasableBytesReference moveToBytesReference() {
        BytesReference bytes = this.bytes();
        ArrayList<Recycler.V<BytesRef>> pages = this.pages;
        this.closeFields();
        return new ReleasableBytesReference(bytes, () -> Releasables.close(pages));
    }

    private void closeFields() {
        this.pages = null;
        this.currentBytesRef = null;
        this.pageIndex = -1;
        this.currentPageOffset = this.pageSize;
        this.currentCapacity = 0;
    }

    public int size() {
        return Math.toIntExact(this.position());
    }

    @Override
    public BytesReference bytes() {
        int bytesInLastPage;
        int adjustment;
        int position = (int)this.position();
        if (position == 0) {
            return BytesArray.EMPTY;
        }
        int remainder = position % this.pageSize;
        if (remainder != 0) {
            adjustment = 1;
            bytesInLastPage = remainder;
        } else {
            adjustment = 0;
            bytesInLastPage = this.pageSize;
        }
        int pageCount = position / this.pageSize + adjustment;
        if (pageCount == 1) {
            BytesRef page = this.pages.get(0).v();
            return new BytesArray(page.bytes, page.offset, bytesInLastPage);
        }
        BytesReference[] references = new BytesReference[pageCount];
        for (int i = 0; i < pageCount - 1; ++i) {
            references[i] = new BytesArray(this.pages.get(i).v());
        }
        BytesRef last = this.pages.get(pageCount - 1).v();
        references[pageCount - 1] = new BytesArray(last.bytes, last.offset, bytesInLastPage);
        return CompositeBytesReference.of(references);
    }

    private void ensureCapacity(int bytesNeeded) {
        assert (bytesNeeded > this.pageSize - this.currentPageOffset);
        this.ensureCapacityFromPosition(this.position() + (long)bytesNeeded);
    }

    private void ensureCapacityFromPosition(long newPosition) {
        if (newPosition > (long)(Integer.MAX_VALUE - Integer.MAX_VALUE % this.pageSize)) {
            throw new IllegalArgumentException(this.getClass().getSimpleName() + " cannot hold more than 2GB of data");
        }
        if (this.pages == null) {
            throw new IllegalStateException("Cannot use " + this.getClass().getSimpleName() + " after it has been closed");
        }
        long additionalCapacityNeeded = newPosition - (long)this.currentCapacity;
        if (additionalCapacityNeeded > 0L) {
            int additionalPagesNeeded = (int)((additionalCapacityNeeded + (long)this.pageSize - 1L) / (long)this.pageSize);
            this.pages.ensureCapacity(this.pages.size() + additionalPagesNeeded);
            for (int i = 0; i < additionalPagesNeeded; ++i) {
                Recycler.V<BytesRef> newPage = this.recycler.obtain();
                assert (this.pageSize == newPage.v().length);
                this.pages.add(newPage);
            }
            this.currentCapacity += additionalPagesNeeded * this.pageSize;
        }
    }

    private void nextPage() {
        ++this.pageIndex;
        this.currentPageOffset = 0;
        this.currentBytesRef = this.pages.get(this.pageIndex).v();
    }
}

