/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.stream.IntStream;
import org.HdrHistogram.DoubleHistogram;
import org.HdrHistogram.IntCountsHistogram;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.component.Lifecycle;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.telemetry.metric.DoubleWithAttributes;
import org.elasticsearch.telemetry.metric.LongWithAttributes;
import org.elasticsearch.telemetry.metric.MeterRegistry;

public class ShardWriteLoadDistributionMetrics {
    private static final Logger logger = LogManager.getLogger(ShardWriteLoadDistributionMetrics.class);
    private static final String WRITE_LOAD_DISTRIBUTION_METRIC_NAME = "es.allocator.shard_write_load.distribution.p%s.current";
    public static final String WRITE_LOAD_PRIORITISATION_THRESHOLD_METRIC_NAME = "es.allocator.shard_write_load.prioritisation_threshold.current";
    public static final String WRITE_LOAD_PRIORITISATION_THRESHOLD_PERCENTILE_RANK_METRIC_NAME = "es.allocator.shard_write_load.prioritisation_threshold.shard_count_exceeding.current";
    public static final String WRITE_LOAD_SUM_METRIC_NAME = "es.allocator.shard_write_load.sum.current";
    public static final Setting<Boolean> SHARD_WRITE_LOAD_METRICS_ENABLED_SETTING = Setting.boolSetting("cluster.routing.allocation.write_load_decider.shard_write_load_metrics.enabled", true, Setting.Property.Dynamic, Setting.Property.NodeScope);
    private final DoubleHistogram shardWeightHistogram;
    private final int[] trackedPercentiles;
    private final ClusterService clusterService;
    private final AtomicReferenceArray<List<DoubleWithAttributes>> lastWriteLoadDistributionMetrics;
    private final AtomicReference<List<DoubleWithAttributes>> lastWriteLoadPrioritisationThresholdMetrics = new AtomicReference(List.of());
    private final AtomicReference<List<LongWithAttributes>> lastShardCountExceedingPrioritisationThresholdMetrics = new AtomicReference(List.of());
    private final AtomicReference<List<DoubleWithAttributes>> lastWriteLoadSumMetrics = new AtomicReference(List.of());
    private volatile boolean metricsEnabled = false;
    private volatile boolean lastMetricsCollected = true;

    public ShardWriteLoadDistributionMetrics(MeterRegistry meterRegistry, ClusterService clusterService) {
        this(meterRegistry, clusterService, 2, 50, 90, 95, 99, 100);
    }

    public ShardWriteLoadDistributionMetrics(MeterRegistry meterRegistry, ClusterService clusterService, int numberOfSignificantDigits, int ... trackedPercentiles) {
        this.clusterService = clusterService;
        this.clusterService.getClusterSettings().initializeAndWatch(SHARD_WRITE_LOAD_METRICS_ENABLED_SETTING, value -> {
            this.metricsEnabled = value;
        });
        this.shardWeightHistogram = new DoubleHistogram(numberOfSignificantDigits, IntCountsHistogram.class);
        this.trackedPercentiles = trackedPercentiles;
        this.lastWriteLoadDistributionMetrics = new AtomicReferenceArray(trackedPercentiles.length);
        IntStream.range(0, trackedPercentiles.length).forEach(percentileIndex -> {
            this.lastWriteLoadDistributionMetrics.set(percentileIndex, List.of());
            meterRegistry.registerDoublesGauge(ShardWriteLoadDistributionMetrics.shardWriteLoadDistributionMetricName(trackedPercentiles[percentileIndex]), trackedPercentiles[percentileIndex] + "th percentile of shard write-load values, broken down by node", "write load", () -> this.getWriteLoadDistributionMetrics(percentileIndex));
        });
        meterRegistry.registerDoublesGauge(WRITE_LOAD_PRIORITISATION_THRESHOLD_METRIC_NAME, "The threshold over which shards will be prioritised for movement when hot-spotting, per node", "write load", this::getWriteLoadPrioritisationThresholdMetrics);
        meterRegistry.registerLongsGauge(WRITE_LOAD_PRIORITISATION_THRESHOLD_PERCENTILE_RANK_METRIC_NAME, "The number of shards whose write-load exceeds the prioritisation threshold, per node", "unit", this::getWriteLoadPrioritisationThresholdPercentileRankMetrics);
        meterRegistry.registerDoublesGauge(WRITE_LOAD_SUM_METRIC_NAME, "The sum of the shard write-loads for the shards allocated to each node", "write load", this::getWriteLoadSumMetrics);
    }

    public void onNewInfo(ClusterInfo clusterInfo) {
        if (!this.metricsEnabled || this.clusterService.lifecycleState() != Lifecycle.State.STARTED || clusterInfo.getShardWriteLoads().isEmpty() || !this.lastMetricsCollected) {
            return;
        }
        ClusterState clusterState = this.clusterService.state();
        Map<ShardId, Double> shardWriteLoads = clusterInfo.getShardWriteLoads();
        int ingestNodeCount = (int)clusterState.nodes().stream().filter(this::isIndexingNode).count();
        List<DoubleWithAttributes>[] writeLoadDistributionMetrics = this.createPercentileArrays(this.trackedPercentiles.length, ingestNodeCount);
        ArrayList<DoubleWithAttributes> writeLoadPrioritisationThresholdMetrics = new ArrayList<DoubleWithAttributes>(ingestNodeCount);
        ArrayList<LongWithAttributes> shardCountsExceedingPrioritisationThresholdMetrics = new ArrayList<LongWithAttributes>(ingestNodeCount);
        ArrayList<DoubleWithAttributes> shardWriteLoadSumMetrics = new ArrayList<DoubleWithAttributes>(ingestNodeCount);
        for (RoutingNode routingNode : clusterState.getRoutingNodes()) {
            DiscoveryNode node = routingNode.node();
            if (node == null || !this.isIndexingNode(node)) continue;
            double maxShardWriteLoad = Double.NEGATIVE_INFINITY;
            double totalShardWriteLoad = 0.0;
            this.shardWeightHistogram.reset();
            try {
                for (ShardRouting shardRouting : routingNode) {
                    double writeLoad = shardWriteLoads.getOrDefault(shardRouting.shardId(), 0.0);
                    this.shardWeightHistogram.recordValue(this.roundTinyValuesToZero(writeLoad));
                    maxShardWriteLoad = Math.max(maxShardWriteLoad, writeLoad);
                    totalShardWriteLoad += writeLoad;
                }
            }
            catch (ArrayIndexOutOfBoundsException e) {
                String message = "Failed to record shard write load distribution metrics for node " + node.getName() + " (" + e.getMessage() + ")";
                assert (false) : message;
                logger.error(message, (Throwable)e);
                continue;
            }
            Map<String, Object> nodeAttrs = this.getAttributesForNode(node);
            if (maxShardWriteLoad > Double.NEGATIVE_INFINITY) {
                for (int i = 0; i < this.trackedPercentiles.length; ++i) {
                    writeLoadDistributionMetrics[i].add(new DoubleWithAttributes(this.shardWeightHistogram.getValueAtPercentile((double)this.trackedPercentiles[i]), nodeAttrs));
                }
                double prioritisationThreshold = 0.5 * maxShardWriteLoad;
                writeLoadPrioritisationThresholdMetrics.add(new DoubleWithAttributes(prioritisationThreshold, nodeAttrs));
                long shardsExceedingThreshold = (long)this.shardWeightHistogram.getCountBetweenValues(prioritisationThreshold, Double.MAX_VALUE);
                shardCountsExceedingPrioritisationThresholdMetrics.add(new LongWithAttributes(shardsExceedingThreshold, nodeAttrs));
            }
            shardWriteLoadSumMetrics.add(new DoubleWithAttributes(totalShardWriteLoad, nodeAttrs));
        }
        this.lastMetricsCollected = false;
        for (int i = 0; i < this.trackedPercentiles.length; ++i) {
            this.lastWriteLoadDistributionMetrics.set(i, writeLoadDistributionMetrics[i]);
        }
        this.lastWriteLoadPrioritisationThresholdMetrics.set(writeLoadPrioritisationThresholdMetrics);
        this.lastShardCountExceedingPrioritisationThresholdMetrics.set(shardCountsExceedingPrioritisationThresholdMetrics);
        this.lastWriteLoadSumMetrics.set(shardWriteLoadSumMetrics);
    }

    private List<DoubleWithAttributes>[] createPercentileArrays(int percentileCount, int ingestNodeCount) {
        List[] lists = new List[percentileCount];
        for (int i = 0; i < percentileCount; ++i) {
            lists[i] = new ArrayList(ingestNodeCount);
        }
        return lists;
    }

    public static String shardWriteLoadDistributionMetricName(int percentile) {
        return Strings.format(WRITE_LOAD_DISTRIBUTION_METRIC_NAME, percentile);
    }

    private boolean isIndexingNode(DiscoveryNode discoveryNode) {
        return discoveryNode.getRoles().contains(DiscoveryNodeRole.INDEX_ROLE);
    }

    private double roundTinyValuesToZero(double value) {
        assert (value >= 0.0) : "We got a negative write load?! " + value;
        return value < 0.01 ? 0.0 : value;
    }

    private Map<String, Object> getAttributesForNode(DiscoveryNode node) {
        return Map.of("es_node_id", node.getId(), "es_node_name", node.getName());
    }

    final Collection<DoubleWithAttributes> getWriteLoadDistributionMetrics(int index) {
        List<DoubleWithAttributes> metrics = this.lastWriteLoadDistributionMetrics.getAndSet(index, List.of());
        this.lastMetricsCollected = true;
        return metrics;
    }

    final Collection<DoubleWithAttributes> getWriteLoadPrioritisationThresholdMetrics() {
        List<DoubleWithAttributes> metrics = this.lastWriteLoadPrioritisationThresholdMetrics.getAndSet(List.of());
        this.lastMetricsCollected = true;
        return metrics;
    }

    final Collection<LongWithAttributes> getWriteLoadPrioritisationThresholdPercentileRankMetrics() {
        List<LongWithAttributes> metrics = this.lastShardCountExceedingPrioritisationThresholdMetrics.getAndSet(List.of());
        this.lastMetricsCollected = true;
        return metrics;
    }

    final Collection<DoubleWithAttributes> getWriteLoadSumMetrics() {
        List<DoubleWithAttributes> metrics = this.lastWriteLoadSumMetrics.getAndSet(List.of());
        this.lastMetricsCollected = true;
        return metrics;
    }

    int[] getTrackedPercentiles() {
        return Arrays.copyOf(this.trackedPercentiles, this.trackedPercentiles.length);
    }
}

