/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing;

import java.io.IOException;
import java.util.Base64;
import java.util.List;
import java.util.OptionalInt;
import java.util.Set;
import java.util.function.IntConsumer;
import java.util.function.Predicate;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.StringHelper;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.RoutingMissingException;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.IndexReshardingMetadata;
import org.elasticsearch.cluster.metadata.IndexReshardingState;
import org.elasticsearch.cluster.metadata.MappingMetadata;
import org.elasticsearch.cluster.routing.Murmur3HashFunction;
import org.elasticsearch.cluster.routing.RoutingHashBuilder;
import org.elasticsearch.cluster.routing.TsidBuilder;
import org.elasticsearch.cluster.routing.XContentParserTsidFunnel;
import org.elasticsearch.common.ParsingException;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.regex.Regex;
import org.elasticsearch.common.util.ByteUtils;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.common.xcontent.XContentParserUtils;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.features.NodeFeature;
import org.elasticsearch.index.IndexMode;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.index.mapper.TimeSeriesRoutingHashFieldMapper;
import org.elasticsearch.index.mapper.TsidExtractingIdFieldMapper;
import org.elasticsearch.transport.Transports;
import org.elasticsearch.xcontent.XContentParser;
import org.elasticsearch.xcontent.XContentParserConfiguration;
import org.elasticsearch.xcontent.XContentType;

public abstract class IndexRouting {
    static final NodeFeature LOGSB_ROUTE_ON_SORT_FIELDS = new NodeFeature("routing.logsb_route_on_sort_fields");
    protected final String indexName;
    private final int routingNumShards;
    private final int routingFactor;
    @Nullable
    private final IndexReshardingMetadata indexReshardingMetadata;

    public static IndexRouting fromIndexMetadata(IndexMetadata metadata) {
        if (metadata.getIndexMode() == IndexMode.TIME_SERIES && !metadata.getTimeSeriesDimensions().isEmpty() && metadata.getCreationVersion().onOrAfter(IndexVersions.TSID_CREATED_DURING_ROUTING)) {
            return new ExtractFromSource.ForIndexDimensions(metadata);
        }
        if (!metadata.getRoutingPaths().isEmpty()) {
            return new ExtractFromSource.ForRoutingPath(metadata);
        }
        if (metadata.isRoutingPartitionedIndex()) {
            return new Partitioned(metadata);
        }
        return new Unpartitioned(metadata);
    }

    private IndexRouting(IndexMetadata metadata) {
        this.indexName = metadata.getIndex().getName();
        this.routingNumShards = metadata.getRoutingNumShards();
        this.routingFactor = metadata.getRoutingFactor();
        this.indexReshardingMetadata = metadata.getReshardingMetadata();
    }

    public void preProcess(IndexRequest indexRequest) {
    }

    public void postProcess(IndexRequest indexRequest) {
    }

    public abstract int indexShard(IndexRequest var1);

    public abstract int rerouteToTarget(IndexRequest var1);

    public abstract int updateShard(String var1, @Nullable String var2);

    public abstract int deleteShard(String var1, @Nullable String var2);

    public abstract int getShard(String var1, @Nullable String var2);

    public abstract void collectSearchShards(String var1, IntConsumer var2);

    protected final int hashToShardId(int hash) {
        return Math.floorMod(hash, this.routingNumShards) / this.routingFactor;
    }

    private static int effectiveRoutingToHash(String effectiveRouting) {
        return Murmur3HashFunction.hash(effectiveRouting);
    }

    public void checkIndexSplitAllowed() {
    }

    protected final int rerouteWritesIfResharding(int shardId) {
        return this.rerouteFromSplitTargetShard(shardId, IndexReshardingState.Split.TargetShardState.HANDOFF);
    }

    protected final int rerouteSearchIfResharding(int shardId) {
        return this.rerouteFromSplitTargetShard(shardId, IndexReshardingState.Split.TargetShardState.SPLIT);
    }

    private int rerouteFromSplitTargetShard(int shardId, IndexReshardingState.Split.TargetShardState minimumRequiredState) {
        assert (this.indexReshardingMetadata == null || this.indexReshardingMetadata.isSplit()) : "Index resharding state is not a split";
        if (this.indexReshardingMetadata != null && this.indexReshardingMetadata.getSplit().isTargetShard(shardId) && !this.indexReshardingMetadata.getSplit().targetStateAtLeast(shardId, minimumRequiredState)) {
            return this.indexReshardingMetadata.getSplit().sourceShard(shardId);
        }
        return shardId;
    }

    public static abstract class ExtractFromSource
    extends IndexRouting {
        protected final XContentParserConfiguration parserConfig;
        private final IndexMode indexMode;
        private final boolean trackTimeSeriesRoutingHash;
        private final boolean useTimeSeriesSyntheticId;
        private final boolean addIdWithRoutingHash;
        private int hash = Integer.MAX_VALUE;

        ExtractFromSource(IndexMetadata metadata, List<String> includePaths) {
            super(metadata);
            if (metadata.isRoutingPartitionedIndex()) {
                throw new IllegalArgumentException("routing_partition_size is incompatible with routing_path");
            }
            this.indexMode = metadata.getIndexMode();
            assert (this.indexMode != null) : "Index mode must be set for ExtractFromSource routing";
            this.trackTimeSeriesRoutingHash = this.indexMode == IndexMode.TIME_SERIES && metadata.getCreationVersion().onOrAfter(IndexVersions.TIME_SERIES_ROUTING_HASH_IN_ID);
            this.useTimeSeriesSyntheticId = metadata.useTimeSeriesSyntheticId();
            this.addIdWithRoutingHash = this.indexMode == IndexMode.LOGSDB;
            this.parserConfig = XContentParserConfiguration.EMPTY.withFiltering(null, Set.copyOf(includePaths), null, true);
        }

        @Override
        public void postProcess(IndexRequest indexRequest) {
            if (this.trackTimeSeriesRoutingHash) {
                indexRequest.routing(TimeSeriesRoutingHashFieldMapper.encode(this.hash));
            } else if (this.addIdWithRoutingHash) {
                assert (this.hash != Integer.MAX_VALUE);
                indexRequest.autoGenerateTimeBasedId(OptionalInt.of(this.hash));
            }
        }

        @Override
        public int indexShard(IndexRequest indexRequest) {
            assert (Transports.assertNotTransportThread("parsing the _source can get slow"));
            this.checkNoRouting(indexRequest.routing());
            this.hash = this.hashSource(indexRequest);
            int shardId = this.hashToShardId(this.hash);
            return this.rerouteWritesIfResharding(shardId);
        }

        @Override
        public int rerouteToTarget(IndexRequest indexRequest) {
            if (this.trackTimeSeriesRoutingHash) {
                String routing = indexRequest.routing();
                if (routing == null) {
                    throw new IllegalStateException("Routing should be set by the coordinator");
                }
                return this.hashToShardId(TimeSeriesRoutingHashFieldMapper.decode(indexRequest.routing()));
            }
            if (this.addIdWithRoutingHash) {
                return this.hashToShardId(this.idToHash(indexRequest.id()));
            }
            this.checkNoRouting(indexRequest.routing());
            return this.indexShard(indexRequest);
        }

        protected abstract int hashSource(IndexRequest var1);

        private static int defaultOnEmpty() {
            throw new IllegalArgumentException("Error extracting routing: source didn't contain any routing fields");
        }

        protected static int hash(BytesRef ref) {
            return StringHelper.murmurhash3_x86_32(ref, 0);
        }

        @Override
        public int updateShard(String id, @Nullable String routing) {
            throw new IllegalArgumentException(this.error("update"));
        }

        @Override
        public int deleteShard(String id, @Nullable String routing) {
            this.checkNoRouting(routing);
            int shardId = this.idToHash(id);
            return this.rerouteWritesIfResharding(shardId);
        }

        @Override
        public int getShard(String id, @Nullable String routing) {
            this.checkNoRouting(routing);
            int shardId = this.idToHash(id);
            return this.rerouteWritesIfResharding(shardId);
        }

        private void checkNoRouting(@Nullable String routing) {
            if (routing != null) {
                throw new IllegalArgumentException(this.error("specifying routing"));
            }
        }

        private int idToHash(String id) {
            byte[] idBytes;
            try {
                idBytes = Base64.getUrlDecoder().decode(id);
            }
            catch (IllegalArgumentException e) {
                throw new ResourceNotFoundException("invalid id [{}] for index [{}] in " + this.indexMode.getName() + " mode", id, this.indexName);
            }
            if (idBytes.length < 4) {
                throw new ResourceNotFoundException("invalid id [{}] for index [{}] in " + this.indexMode.getName() + " mode", id, this.indexName);
            }
            int hash = this.addIdWithRoutingHash ? ByteUtils.readIntLE(idBytes, idBytes.length - 9) : (this.useTimeSeriesSyntheticId ? TsidExtractingIdFieldMapper.extractRoutingHashFromSyntheticId(new BytesRef(idBytes)) : ByteUtils.readIntLE(idBytes, 0));
            return this.hashToShardId(hash);
        }

        @Override
        public void checkIndexSplitAllowed() {
            throw new IllegalArgumentException(this.error("index-split"));
        }

        @Override
        public void collectSearchShards(String routing, IntConsumer consumer) {
            throw new IllegalArgumentException(this.error("searching with a specified routing"));
        }

        private String error(String operation) {
            return operation + " is not supported because the destination index [" + this.indexName + "] is in " + this.indexMode.getName() + " mode";
        }

        public static class ForIndexDimensions
        extends ExtractFromSource {
            ForIndexDimensions(IndexMetadata metadata) {
                super(metadata, metadata.getTimeSeriesDimensions());
                assert (metadata.getIndexMode() == IndexMode.TIME_SERIES) : "Index mode must be time_series for ForIndexDimensions routing";
                assert (metadata.getCreationVersion().onOrAfter(IndexVersions.TSID_CREATED_DURING_ROUTING)) : "Index version must be at least " + String.valueOf(IndexVersions.TSID_CREATED_DURING_ROUTING) + " for ForIndexDimensions routing but was " + String.valueOf(metadata.getCreationVersion());
            }

            @Override
            protected int hashSource(IndexRequest indexRequest) {
                BytesRef tsid = indexRequest.tsid();
                if (tsid == null) {
                    tsid = this.buildTsid(indexRequest.getContentType(), indexRequest.indexSource().bytes());
                    indexRequest.tsid(tsid);
                }
                return ForIndexDimensions.hash(tsid);
            }

            public BytesRef buildTsid(XContentType sourceType, BytesReference source) {
                TsidBuilder b = new TsidBuilder();
                try (XContentParser parser = XContentHelper.createParserNotCompressed(this.parserConfig, source, sourceType);){
                    b.add(parser, XContentParserTsidFunnel.get());
                }
                catch (IOException | ParsingException e) {
                    throw new IllegalArgumentException("Error extracting tsid: " + e.getMessage(), e);
                }
                return b.buildTsid();
            }
        }

        public static class ForRoutingPath
        extends ExtractFromSource {
            private final Predicate<String> isRoutingPath;

            ForRoutingPath(IndexMetadata metadata) {
                super(metadata, metadata.getRoutingPaths());
                this.isRoutingPath = Regex.simpleMatcher((String[])metadata.getRoutingPaths().toArray(String[]::new));
            }

            @Override
            protected int hashSource(IndexRequest indexRequest) {
                return this.hashRoutingFields(indexRequest.getContentType(), indexRequest.source()).buildHash(ExtractFromSource::defaultOnEmpty);
            }

            public String createId(XContentType sourceType, BytesReference source, byte[] suffix) {
                return this.hashRoutingFields(sourceType, source).createId(suffix, ExtractFromSource::defaultOnEmpty);
            }

            public RoutingHashBuilder builder() {
                return new RoutingHashBuilder(this.isRoutingPath);
            }

            private RoutingHashBuilder hashRoutingFields(XContentType sourceType, BytesReference source) {
                RoutingHashBuilder b = this.builder();
                try (XContentParser parser = XContentHelper.createParserNotCompressed(this.parserConfig, source, sourceType);){
                    parser.nextToken();
                    if (parser.currentToken() == null) {
                        throw new IllegalArgumentException("Error extracting routing: source didn't contain any routing fields");
                    }
                    parser.nextToken();
                    b.extractObject(null, parser);
                    XContentParserUtils.ensureExpectedToken(null, parser.nextToken(), parser);
                }
                catch (IOException | ParsingException e) {
                    throw new IllegalArgumentException("Error extracting routing: " + e.getMessage(), e);
                }
                return b;
            }

            public boolean matchesField(String fieldName) {
                return this.isRoutingPath.test(fieldName);
            }
        }
    }

    private static class Partitioned
    extends IdAndRoutingOnly {
        private final int routingPartitionSize;

        Partitioned(IndexMetadata metadata) {
            super(metadata);
            this.routingPartitionSize = metadata.getRoutingPartitionSize();
        }

        @Override
        protected int shardId(String id, @Nullable String routing) {
            if (routing == null) {
                throw new IllegalArgumentException("A routing value is required for gets from a partitioned index");
            }
            int offset = Math.floorMod(IndexRouting.effectiveRoutingToHash(id), this.routingPartitionSize);
            return this.hashToShardId(IndexRouting.effectiveRoutingToHash(routing) + offset);
        }

        @Override
        public void collectSearchShards(String routing, IntConsumer consumer) {
            int hash = IndexRouting.effectiveRoutingToHash(routing);
            for (int i = 0; i < this.routingPartitionSize; ++i) {
                consumer.accept(this.rerouteSearchIfResharding(this.hashToShardId(hash + i)));
            }
        }
    }

    private static class Unpartitioned
    extends IdAndRoutingOnly {
        Unpartitioned(IndexMetadata metadata) {
            super(metadata);
        }

        @Override
        protected int shardId(String id, @Nullable String routing) {
            return this.hashToShardId(IndexRouting.effectiveRoutingToHash(routing == null ? id : routing));
        }

        @Override
        public void collectSearchShards(String routing, IntConsumer consumer) {
            consumer.accept(this.rerouteSearchIfResharding(this.hashToShardId(IndexRouting.effectiveRoutingToHash(routing))));
        }
    }

    private static abstract class IdAndRoutingOnly
    extends IndexRouting {
        private final boolean routingRequired;
        private final IndexVersion creationVersion;
        private final IndexMode indexMode;

        IdAndRoutingOnly(IndexMetadata metadata) {
            super(metadata);
            this.creationVersion = metadata.getCreationVersion();
            MappingMetadata mapping = metadata.mapping();
            this.routingRequired = mapping == null ? false : mapping.routingRequired();
            this.indexMode = metadata.getIndexMode();
        }

        protected abstract int shardId(String var1, @Nullable String var2);

        @Override
        public void preProcess(IndexRequest indexRequest) {
            String id = indexRequest.id();
            if (id == null) {
                if (IdAndRoutingOnly.shouldUseTimeBasedId(this.indexMode, this.creationVersion)) {
                    indexRequest.autoGenerateTimeBasedId();
                } else {
                    indexRequest.autoGenerateId();
                }
            } else if (id.isEmpty()) {
                throw new IllegalArgumentException("if _id is specified it must not be empty");
            }
        }

        private static boolean shouldUseTimeBasedId(IndexMode indexMode, IndexVersion creationVersion) {
            return indexMode == IndexMode.LOGSDB && IdAndRoutingOnly.isNewIndexVersion(creationVersion);
        }

        private static boolean isNewIndexVersion(IndexVersion creationVersion) {
            return creationVersion.between(IndexVersions.TIME_BASED_K_ORDERED_DOC_ID_BACKPORT, IndexVersions.UPGRADE_TO_LUCENE_10_0_0) || creationVersion.onOrAfter(IndexVersions.TIME_BASED_K_ORDERED_DOC_ID);
        }

        @Override
        public int indexShard(IndexRequest indexRequest) {
            String id = indexRequest.id();
            String routing = indexRequest.routing();
            if (id == null) {
                throw new IllegalStateException("id is required and should have been set by process");
            }
            this.checkRoutingRequired(id, routing);
            int shardId = this.shardId(id, routing);
            return this.rerouteWritesIfResharding(shardId);
        }

        @Override
        public int rerouteToTarget(IndexRequest indexRequest) {
            return this.indexShard(indexRequest);
        }

        @Override
        public int updateShard(String id, @Nullable String routing) {
            this.checkRoutingRequired(id, routing);
            int shardId = this.shardId(id, routing);
            return this.rerouteWritesIfResharding(shardId);
        }

        @Override
        public int deleteShard(String id, @Nullable String routing) {
            this.checkRoutingRequired(id, routing);
            int shardId = this.shardId(id, routing);
            return this.rerouteWritesIfResharding(shardId);
        }

        @Override
        public int getShard(String id, @Nullable String routing) {
            this.checkRoutingRequired(id, routing);
            return this.shardId(id, routing);
        }

        private void checkRoutingRequired(String id, @Nullable String routing) {
            if (this.routingRequired && routing == null) {
                throw new RoutingMissingException(this.indexName, id);
            }
        }
    }
}

