/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.action.admin.indices.rollover;

import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Pattern;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.admin.indices.create.CreateIndexClusterStateUpdateRequest;
import org.elasticsearch.action.admin.indices.create.CreateIndexRequest;
import org.elasticsearch.action.admin.indices.rollover.Condition;
import org.elasticsearch.action.admin.indices.rollover.RolloverInfo;
import org.elasticsearch.action.datastreams.autosharding.AutoShardingResult;
import org.elasticsearch.action.datastreams.autosharding.AutoShardingType;
import org.elasticsearch.action.support.ActiveShardCount;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ProjectState;
import org.elasticsearch.cluster.metadata.AliasAction;
import org.elasticsearch.cluster.metadata.AliasMetadata;
import org.elasticsearch.cluster.metadata.ComposableIndexTemplate;
import org.elasticsearch.cluster.metadata.DataStream;
import org.elasticsearch.cluster.metadata.DataStreamAutoShardingEvent;
import org.elasticsearch.cluster.metadata.IndexAbstraction;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.IndexMetadataStats;
import org.elasticsearch.cluster.metadata.IndexNameExpressionResolver;
import org.elasticsearch.cluster.metadata.IndexTemplateMetadata;
import org.elasticsearch.cluster.metadata.MetadataCreateDataStreamService;
import org.elasticsearch.cluster.metadata.MetadataCreateIndexService;
import org.elasticsearch.cluster.metadata.MetadataDataStreamsService;
import org.elasticsearch.cluster.metadata.MetadataIndexAliasesService;
import org.elasticsearch.cluster.metadata.MetadataIndexTemplateService;
import org.elasticsearch.cluster.metadata.ProjectId;
import org.elasticsearch.cluster.metadata.ProjectMetadata;
import org.elasticsearch.cluster.routing.allocation.WriteLoadForecaster;
import org.elasticsearch.cluster.routing.allocation.allocator.AllocationActionListener;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.IndexMode;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.indices.SystemDataStreamDescriptor;
import org.elasticsearch.indices.SystemIndices;
import org.elasticsearch.injection.guice.Inject;
import org.elasticsearch.snapshots.SnapshotInProgressException;
import org.elasticsearch.snapshots.SnapshotsServiceUtils;
import org.elasticsearch.telemetry.TelemetryProvider;
import org.elasticsearch.telemetry.metric.MeterRegistry;
import org.elasticsearch.threadpool.ThreadPool;

public class MetadataRolloverService {
    private static final Logger logger = LogManager.getLogger(MetadataRolloverService.class);
    private static final Pattern INDEX_NAME_PATTERN = Pattern.compile("^.*-\\d+$");
    private static final List<IndexAbstraction.Type> VALID_ROLLOVER_TARGETS = List.of(IndexAbstraction.Type.ALIAS, IndexAbstraction.Type.DATA_STREAM);
    public static final Settings HIDDEN_INDEX_SETTINGS = Settings.builder().put("index.hidden", true).build();
    public static final Map<AutoShardingType, String> AUTO_SHARDING_METRIC_NAMES = Map.of(AutoShardingType.INCREASE_SHARDS, "es.auto_sharding.increase_shards.total", AutoShardingType.DECREASE_SHARDS, "es.auto_sharding.decrease_shards.total", AutoShardingType.COOLDOWN_PREVENTED_INCREASE, "es.auto_sharding.cooldown_prevented_increase.total", AutoShardingType.COOLDOWN_PREVENTED_DECREASE, "es.auto_sharding.cooldown_prevented_decrease.total");
    private static final String NON_EXISTENT_SOURCE = "_none_";
    private final ThreadPool threadPool;
    private final MetadataCreateIndexService createIndexService;
    private final MetadataIndexAliasesService indexAliasesService;
    private final SystemIndices systemIndices;
    private final WriteLoadForecaster writeLoadForecaster;
    private final ClusterService clusterService;
    private final MeterRegistry meterRegistry;

    @Inject
    public MetadataRolloverService(ThreadPool threadPool, MetadataCreateIndexService createIndexService, MetadataIndexAliasesService indexAliasesService, SystemIndices systemIndices, WriteLoadForecaster writeLoadForecaster, ClusterService clusterService, TelemetryProvider telemetryProvider) {
        this.threadPool = threadPool;
        this.createIndexService = createIndexService;
        this.indexAliasesService = indexAliasesService;
        this.systemIndices = systemIndices;
        this.writeLoadForecaster = writeLoadForecaster;
        this.clusterService = clusterService;
        this.meterRegistry = telemetryProvider.getMeterRegistry();
        for (Map.Entry<AutoShardingType, String> entry : AUTO_SHARDING_METRIC_NAMES.entrySet()) {
            AutoShardingType type = entry.getKey();
            String metricName = entry.getValue();
            String description = String.format(Locale.ROOT, "auto-sharding %s counter", type.name().toLowerCase(Locale.ROOT));
            this.meterRegistry.registerLongCounter(metricName, description, "unit");
        }
    }

    public RolloverResult rolloverClusterState(ProjectState currentState, String rolloverTarget, String newIndexName, CreateIndexRequest createIndexRequest, List<Condition<?>> metConditions, Instant now, boolean silent, boolean onlyValidate, @Nullable IndexMetadataStats sourceIndexStats, @Nullable AutoShardingResult autoShardingResult, boolean isFailureStoreRollover) throws Exception {
        MetadataRolloverService.validate(currentState.metadata(), rolloverTarget, newIndexName, createIndexRequest);
        IndexAbstraction indexAbstraction = (IndexAbstraction)currentState.metadata().getIndicesLookup().get(rolloverTarget);
        return switch (indexAbstraction.getType()) {
            case IndexAbstraction.Type.ALIAS -> this.rolloverAlias(currentState, (IndexAbstraction.Alias)indexAbstraction, rolloverTarget, newIndexName, createIndexRequest, metConditions, silent, onlyValidate);
            case IndexAbstraction.Type.DATA_STREAM -> this.rolloverDataStream(currentState, (DataStream)indexAbstraction, rolloverTarget, createIndexRequest, metConditions, now, silent, onlyValidate, sourceIndexStats, autoShardingResult, isFailureStoreRollover);
            default -> throw new IllegalStateException("unable to roll over type [" + indexAbstraction.getType().getDisplayName() + "]");
        };
    }

    public static NameResolution resolveRolloverNames(ProjectMetadata project, String rolloverTarget, String newIndexName, CreateIndexRequest createIndexRequest, boolean isFailureStoreRollover) {
        MetadataRolloverService.validate(project, rolloverTarget, newIndexName, createIndexRequest);
        IndexAbstraction indexAbstraction = (IndexAbstraction)project.getIndicesLookup().get(rolloverTarget);
        return switch (indexAbstraction.getType()) {
            case IndexAbstraction.Type.ALIAS -> MetadataRolloverService.resolveAliasRolloverNames(project, indexAbstraction, newIndexName);
            case IndexAbstraction.Type.DATA_STREAM -> MetadataRolloverService.resolveDataStreamRolloverNames(project, (DataStream)indexAbstraction, isFailureStoreRollover);
            default -> throw new IllegalStateException("unable to roll over type [" + indexAbstraction.getType().getDisplayName() + "]");
        };
    }

    private static NameResolution resolveAliasRolloverNames(ProjectMetadata project, IndexAbstraction alias, String newIndexName) {
        IndexMetadata writeIndex = project.index(alias.getWriteIndex());
        String sourceProvidedName = writeIndex.getSettings().get("index.provided_name", writeIndex.getIndex().getName());
        String sourceIndexName = writeIndex.getIndex().getName();
        String unresolvedName = newIndexName != null ? newIndexName : MetadataRolloverService.generateRolloverIndexName(sourceProvidedName);
        String rolloverIndexName = IndexNameExpressionResolver.resolveDateMathExpression(unresolvedName);
        return new NameResolution(sourceIndexName, unresolvedName, rolloverIndexName);
    }

    private static NameResolution resolveDataStreamRolloverNames(ProjectMetadata project, DataStream dataStream, boolean isFailureStoreRollover) {
        DataStream.DataStreamIndices dataStreamIndices = dataStream.getDataStreamIndices(isFailureStoreRollover);
        assert (!dataStreamIndices.getIndices().isEmpty() || isFailureStoreRollover) : "Unable to roll over dataStreamIndices with no indices";
        String originalWriteIndex = dataStreamIndices.getIndices().isEmpty() && dataStreamIndices.isRolloverOnWrite() ? NON_EXISTENT_SOURCE : project.index(dataStreamIndices.getWriteIndex()).getIndex().getName();
        return new NameResolution(originalWriteIndex, null, dataStream.nextWriteIndexAndGeneration(project, dataStreamIndices).v1());
    }

    private RolloverResult rolloverAlias(ProjectState projectState, IndexAbstraction.Alias alias, String aliasName, String newIndexName, CreateIndexRequest createIndexRequest, List<Condition<?>> metConditions, boolean silent, boolean onlyValidate) throws Exception {
        ProjectMetadata projectMetadata = projectState.metadata();
        NameResolution names = MetadataRolloverService.resolveAliasRolloverNames(projectMetadata, alias, newIndexName);
        String sourceIndexName = names.sourceName;
        String rolloverIndexName = names.rolloverName;
        String unresolvedName = names.unresolvedName;
        IndexMetadata writeIndex = projectMetadata.index(alias.getWriteIndex());
        AliasMetadata aliasMetadata = writeIndex.getAliases().get(alias.getName());
        boolean explicitWriteIndex = Boolean.TRUE.equals(aliasMetadata.writeIndex());
        Boolean isHidden = IndexMetadata.INDEX_HIDDEN_SETTING.exists(createIndexRequest.settings()) ? IndexMetadata.INDEX_HIDDEN_SETTING.get(createIndexRequest.settings()) : null;
        MetadataCreateIndexService.validateIndexName(rolloverIndexName, projectMetadata, projectState.routingTable());
        MetadataRolloverService.checkNoDuplicatedAliasInIndexTemplate(projectMetadata, rolloverIndexName, aliasName, isHidden);
        if (onlyValidate) {
            return new RolloverResult(rolloverIndexName, sourceIndexName, projectState.cluster());
        }
        CreateIndexClusterStateUpdateRequest createIndexClusterStateRequest = MetadataRolloverService.prepareCreateIndexRequest(projectState.projectId(), unresolvedName, rolloverIndexName, createIndexRequest);
        assert (!createIndexClusterStateRequest.performReroute()) : "rerouteCompletionIsNotRequired() assumes reroute is not called by underlying service";
        ClusterState newState = this.createIndexService.applyCreateIndexRequest(projectState.cluster(), createIndexClusterStateRequest, silent, AllocationActionListener.rerouteCompletionIsNotRequired());
        newState = this.indexAliasesService.applyAliasActions(newState.projectState(projectState.projectId()), MetadataRolloverService.rolloverAliasToNewIndex(sourceIndexName, rolloverIndexName, explicitWriteIndex, aliasMetadata.isHidden(), aliasName));
        RolloverInfo rolloverInfo = new RolloverInfo(aliasName, metConditions, this.threadPool.absoluteTimeInMillis());
        ProjectMetadata newProject = newState.metadata().getProject(projectState.projectId());
        newState = ClusterState.builder(newState).putProjectMetadata(ProjectMetadata.builder(newProject).put(IndexMetadata.builder(newProject.index(sourceIndexName)).putRolloverInfo(rolloverInfo))).build();
        return new RolloverResult(rolloverIndexName, sourceIndexName, newState);
    }

    private RolloverResult rolloverDataStream(ProjectState projectState, DataStream dataStream, String dataStreamName, CreateIndexRequest createIndexRequest, List<Condition<?>> metConditions, Instant now, boolean silent, boolean onlyValidate, @Nullable IndexMetadataStats sourceIndexStats, @Nullable AutoShardingResult autoShardingResult, boolean isFailureStoreRollover) throws Exception {
        ClusterState newState;
        ComposableIndexTemplate templateV2;
        SystemDataStreamDescriptor systemDataStreamDescriptor;
        ProjectMetadata metadata = projectState.metadata();
        Set<String> snapshottingDataStreams = SnapshotsServiceUtils.snapshottingDataStreams(projectState, Collections.singleton(dataStream.getName()));
        if (!snapshottingDataStreams.isEmpty()) {
            throw new SnapshotInProgressException("Cannot roll over data stream that is being snapshotted: " + dataStream.getName() + ". Try again after snapshot finishes or cancel the currently running snapshot.");
        }
        if (!dataStream.isSystem()) {
            systemDataStreamDescriptor = null;
            templateV2 = dataStream.getEffectiveIndexTemplate(projectState.metadata());
        } else {
            systemDataStreamDescriptor = this.systemIndices.findMatchingDataStreamDescriptor(dataStreamName);
            if (systemDataStreamDescriptor == null) {
                throw new IllegalArgumentException("no system data stream descriptor found for data stream [" + dataStreamName + "]");
            }
            templateV2 = systemDataStreamDescriptor.getComposableIndexTemplate();
        }
        DataStream.DataStreamIndices dataStreamIndices = dataStream.getDataStreamIndices(isFailureStoreRollover);
        boolean isLazyCreation = dataStreamIndices.getIndices().isEmpty() && dataStreamIndices.isRolloverOnWrite();
        Index originalWriteIndex = isLazyCreation ? null : dataStreamIndices.getWriteIndex();
        Tuple<String, Long> nextIndexAndGeneration = dataStream.nextWriteIndexAndGeneration(metadata, dataStreamIndices);
        String newWriteIndexName = nextIndexAndGeneration.v1();
        long newGeneration = nextIndexAndGeneration.v2();
        MetadataCreateIndexService.validateIndexName(newWriteIndexName, metadata, projectState.routingTable());
        if (onlyValidate) {
            return new RolloverResult(newWriteIndexName, isLazyCreation ? NON_EXISTENT_SOURCE : originalWriteIndex.getName(), projectState.cluster());
        }
        if (isFailureStoreRollover) {
            newState = MetadataCreateDataStreamService.createFailureStoreIndex(this.createIndexService, "rollover_failure_store", projectState.projectId(), this.clusterService.getSettings(), projectState.cluster(), now.toEpochMilli(), dataStreamName, templateV2, systemDataStreamDescriptor, newWriteIndexName, (builder, indexMetadata) -> builder.put(dataStream.rolloverFailureStore(indexMetadata.getIndex(), newGeneration)));
        } else {
            DataStreamAutoShardingEvent dataStreamAutoShardingEvent;
            String metricName;
            if (autoShardingResult != null && (metricName = AUTO_SHARDING_METRIC_NAMES.get((Object)autoShardingResult.type())) != null) {
                this.meterRegistry.getLongCounter(metricName).increment();
            }
            if (autoShardingResult == null) {
                v0 = dataStream.getAutoShardingEvent();
            } else {
                switch (autoShardingResult.type()) {
                    default: {
                        throw new MatchException(null, null);
                    }
                    case NO_CHANGE_REQUIRED: 
                    case COOLDOWN_PREVENTED_INCREASE: 
                    case COOLDOWN_PREVENTED_DECREASE: {
                        if (dataStream.getAutoShardingEvent() != null) {
                            logger.info("Rolling over data stream [{}] using existing auto-sharding recommendation [{}]", (Object)dataStreamName, (Object)dataStream.getAutoShardingEvent());
                        }
                        v0 = dataStream.getAutoShardingEvent();
                        break;
                    }
                    case INCREASE_SHARDS: 
                    case DECREASE_SHARDS: {
                        logger.info("Auto sharding data stream [{}] to [{}]", (Object)dataStreamName, (Object)autoShardingResult);
                        v0 = new DataStreamAutoShardingEvent(dataStream.getWriteIndex().getName(), autoShardingResult.targetNumberOfShards(), now.toEpochMilli());
                        break;
                    }
                    case NOT_APPLICABLE: {
                        logger.debug("auto sharding is not applicable for data stream [{}]", (Object)dataStreamName);
                        v0 = dataStreamAutoShardingEvent = null;
                    }
                }
            }
            if (dataStreamAutoShardingEvent != null) {
                Settings settingsWithAutoSharding = Settings.builder().put(createIndexRequest.settings()).put(IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.getKey(), dataStreamAutoShardingEvent.targetNumberOfShards()).build();
                createIndexRequest.settings(settingsWithAutoSharding);
            }
            CreateIndexClusterStateUpdateRequest createIndexClusterStateRequest = MetadataRolloverService.prepareDataStreamCreateIndexRequest(projectState.projectId(), dataStreamName, newWriteIndexName, createIndexRequest, systemDataStreamDescriptor, now);
            createIndexClusterStateRequest.setMatchingTemplate(templateV2);
            assert (!createIndexClusterStateRequest.performReroute()) : "rerouteCompletionIsNotRequired() assumes reroute is not called by underlying service";
            newState = this.createIndexService.applyCreateIndexRequest(projectState.cluster(), createIndexClusterStateRequest, silent, (builder, indexMetadata) -> {
                MetadataRolloverService.downgradeBrokenTsdbBackingIndices(dataStream, builder);
                builder.put(dataStream.rollover(indexMetadata.getIndex(), newGeneration, metadata.retrieveIndexModeFromTemplate(templateV2), dataStreamAutoShardingEvent));
            }, AllocationActionListener.rerouteCompletionIsNotRequired());
        }
        RolloverInfo rolloverInfo = new RolloverInfo(dataStreamName, metConditions, this.threadPool.absoluteTimeInMillis());
        ProjectMetadata newProject = newState.metadata().getProject(projectState.projectId());
        ProjectMetadata.Builder metadataBuilder = ProjectMetadata.builder(newProject);
        if (!isLazyCreation) {
            metadataBuilder.put(IndexMetadata.builder(newProject.index(originalWriteIndex)).stats(sourceIndexStats).putRolloverInfo(rolloverInfo));
        }
        this.writeLoadForecaster.refreshLicense();
        metadataBuilder = this.writeLoadForecaster.withWriteLoadForecastForWriteIndex(dataStreamName, metadataBuilder);
        metadataBuilder = this.withShardSizeForecastForWriteIndex(dataStreamName, metadataBuilder);
        newState = ClusterState.builder(newState).putProjectMetadata(metadataBuilder).build();
        newState = MetadataDataStreamsService.setRolloverOnWrite(newState.projectState(projectState.projectId()), dataStreamName, false, isFailureStoreRollover);
        return new RolloverResult(newWriteIndexName, isLazyCreation ? NON_EXISTENT_SOURCE : originalWriteIndex.getName(), newState);
    }

    private static void downgradeBrokenTsdbBackingIndices(DataStream dataStream, ProjectMetadata.Builder projectBuilder) {
        for (Index indexName : dataStream.getIndices()) {
            IndexMetadata index = projectBuilder.getSafe(indexName);
            Settings originalSettings = index.getSettings();
            if (!index.getCreationVersion().before(IndexVersions.FIRST_DETACHED_INDEX_VERSION) || index.getIndexMode() != IndexMode.TIME_SERIES || originalSettings.keySet().contains(IndexSettings.TIME_SERIES_START_TIME.getKey()) || originalSettings.keySet().contains(IndexSettings.TIME_SERIES_END_TIME.getKey())) continue;
            Settings.Builder settingsBuilder = Settings.builder().put(originalSettings);
            settingsBuilder.remove(IndexSettings.MODE.getKey());
            settingsBuilder.remove(IndexMetadata.INDEX_ROUTING_PATH.getKey());
            long newVersion = index.getSettingsVersion() + 1L;
            projectBuilder.put(IndexMetadata.builder(index).settings(settingsBuilder.build()).settingsVersion(newVersion));
        }
    }

    public ProjectMetadata.Builder withShardSizeForecastForWriteIndex(String dataStreamName, ProjectMetadata.Builder metadata) {
        DataStream dataStream = metadata.dataStream(dataStreamName);
        if (dataStream == null) {
            return metadata;
        }
        List<IndexMetadataStats> indicesStats = dataStream.getIndices().stream().map(metadata::getSafe).map(IndexMetadata::getStats).filter(Objects::nonNull).toList();
        long totalSizeInBytes = 0L;
        int shardCount = 0;
        for (IndexMetadataStats stats : indicesStats) {
            IndexMetadataStats.AverageShardSize averageShardSize = stats.averageShardSize();
            totalSizeInBytes += averageShardSize.totalSizeInBytes();
            shardCount += averageShardSize.numberOfShards();
        }
        if (shardCount == 0) {
            return metadata;
        }
        long shardSizeInBytesForecast = totalSizeInBytes / (long)shardCount;
        IndexMetadata writeIndex = metadata.getSafe(dataStream.getWriteIndex());
        metadata.put(IndexMetadata.builder(writeIndex).shardSizeInBytesForecast(shardSizeInBytesForecast).build(), false);
        return metadata;
    }

    static String generateRolloverIndexName(String sourceIndexName) {
        boolean isDateMath;
        String resolvedName = IndexNameExpressionResolver.resolveDateMathExpression(sourceIndexName);
        boolean bl = isDateMath = !sourceIndexName.equals(resolvedName);
        if (INDEX_NAME_PATTERN.matcher(resolvedName).matches()) {
            int numberIndex = sourceIndexName.lastIndexOf(45);
            assert (numberIndex != -1) : "no separator '-' found";
            int counter = Integer.parseInt(sourceIndexName.substring(numberIndex + 1, isDateMath ? sourceIndexName.length() - 1 : sourceIndexName.length()));
            String newName = sourceIndexName.substring(0, numberIndex) + "-" + String.format(Locale.ROOT, "%06d", ++counter) + (isDateMath ? ">" : "");
            return newName;
        }
        throw new IllegalArgumentException("index name [" + sourceIndexName + "] does not match pattern '^.*-\\d+$'");
    }

    static CreateIndexClusterStateUpdateRequest prepareDataStreamCreateIndexRequest(ProjectId projectId, String dataStreamName, String targetIndexName, CreateIndexRequest createIndexRequest, SystemDataStreamDescriptor descriptor, Instant now) {
        Settings settings = descriptor != null ? Settings.EMPTY : HIDDEN_INDEX_SETTINGS;
        return MetadataRolloverService.prepareCreateIndexRequest(projectId, targetIndexName, targetIndexName, "rollover_data_stream", createIndexRequest, settings).dataStreamName(dataStreamName).nameResolvedInstant(now.toEpochMilli()).systemDataStreamDescriptor(descriptor);
    }

    static CreateIndexClusterStateUpdateRequest prepareCreateIndexRequest(ProjectId projectId, String providedIndexName, String targetIndexName, CreateIndexRequest createIndexRequest) {
        return MetadataRolloverService.prepareCreateIndexRequest(projectId, providedIndexName, targetIndexName, "rollover_index", createIndexRequest, null);
    }

    static CreateIndexClusterStateUpdateRequest prepareCreateIndexRequest(ProjectId projectId, String providedIndexName, String targetIndexName, String cause, CreateIndexRequest createIndexRequest, Settings settings) {
        Settings.Builder b = Settings.builder().put(createIndexRequest.settings());
        if (settings != null) {
            b.put(settings);
        }
        return new CreateIndexClusterStateUpdateRequest(cause, projectId, targetIndexName, providedIndexName).settings(b.build()).aliases(createIndexRequest.aliases()).waitForActiveShards(ActiveShardCount.NONE).mappings(createIndexRequest.mappings()).performReroute(false);
    }

    static List<AliasAction> rolloverAliasToNewIndex(String oldIndex, String newIndex, boolean explicitWriteIndex, @Nullable Boolean isHidden, String alias) {
        if (explicitWriteIndex) {
            return List.of(new AliasAction.Add(newIndex, alias, null, null, null, true, isHidden), new AliasAction.Add(oldIndex, alias, null, null, null, false, isHidden));
        }
        return List.of(new AliasAction.Add(newIndex, alias, null, null, null, null, isHidden), new AliasAction.Remove(oldIndex, alias, null));
    }

    static void checkNoDuplicatedAliasInIndexTemplate(ProjectMetadata projectMetadata, String rolloverIndexName, String rolloverRequestAlias, @Nullable Boolean isHidden) {
        String matchedV2Template = MetadataIndexTemplateService.findV2Template(projectMetadata, rolloverIndexName, isHidden == null ? false : isHidden);
        if (matchedV2Template != null) {
            List<Map<String, AliasMetadata>> aliases = MetadataIndexTemplateService.resolveAliases(projectMetadata, matchedV2Template);
            for (Map<String, AliasMetadata> aliasConfig : aliases) {
                if (!aliasConfig.containsKey(rolloverRequestAlias)) continue;
                throw new IllegalArgumentException(String.format(Locale.ROOT, "Rollover alias [%s] can point to multiple indices, found duplicated alias [%s] in index template [%s]", rolloverRequestAlias, aliasConfig.keySet(), matchedV2Template));
            }
            return;
        }
        List<IndexTemplateMetadata> matchedTemplates = MetadataIndexTemplateService.findV1Templates(projectMetadata, rolloverIndexName, isHidden);
        for (IndexTemplateMetadata template : matchedTemplates) {
            if (!template.aliases().containsKey(rolloverRequestAlias)) continue;
            throw new IllegalArgumentException(String.format(Locale.ROOT, "Rollover alias [%s] can point to multiple indices, found duplicated alias [%s] in index template [%s]", rolloverRequestAlias, template.aliases().keySet(), template.name()));
        }
    }

    static void validate(ProjectMetadata project, String rolloverTarget, String newIndexName, CreateIndexRequest request) {
        IndexAbstraction indexAbstraction = (IndexAbstraction)project.getIndicesLookup().get(rolloverTarget);
        if (indexAbstraction == null) {
            throw new ResourceNotFoundException("rollover target [" + rolloverTarget + "] does not exist", new Object[0]);
        }
        if (!VALID_ROLLOVER_TARGETS.contains((Object)indexAbstraction.getType())) {
            throw new IllegalArgumentException("rollover target is a [" + indexAbstraction.getType().getDisplayName() + "] but one of [" + Strings.collectionToCommaDelimitedString(VALID_ROLLOVER_TARGETS.stream().map(IndexAbstraction.Type::getDisplayName).toList()) + "] was expected");
        }
        if (indexAbstraction.getWriteIndex() == null) {
            throw new IllegalArgumentException("rollover target [" + indexAbstraction.getName() + "] does not point to a write index");
        }
        if (indexAbstraction.getType() == IndexAbstraction.Type.DATA_STREAM) {
            if (!Strings.isNullOrEmpty(newIndexName)) {
                throw new IllegalArgumentException("new index name may not be specified when rolling over a data stream");
            }
            if (!request.settings().equals(Settings.EMPTY) || request.aliases().size() > 0 || !request.mappings().equals("{}")) {
                throw new IllegalArgumentException("aliases, mappings, and index settings may not be specified when rolling over a data stream");
            }
        }
    }

    public record RolloverResult(String rolloverIndexName, String sourceIndexName, ClusterState clusterState) {
        @Override
        public String toString() {
            return String.format(Locale.ROOT, "cluster state version [%d], rollover index name [%s], source index name [%s]", this.clusterState.version(), this.rolloverIndexName, this.sourceIndexName);
        }
    }

    public record NameResolution(String sourceName, @Nullable String unresolvedName, String rolloverName) {
    }
}

