/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.unit;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Locale;
import java.util.Objects;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.TransportVersion;
import org.elasticsearch.TransportVersions;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.ToXContentFragment;
import org.elasticsearch.xcontent.XContentBuilder;

public class ByteSizeValue
implements Writeable,
Comparable<ByteSizeValue>,
ToXContentFragment {
    public static final ByteSizeValue ZERO = new ByteSizeValue(0L, ByteSizeUnit.BYTES);
    public static final ByteSizeValue ONE = new ByteSizeValue(1L, ByteSizeUnit.BYTES);
    public static final ByteSizeValue MINUS_ONE = new ByteSizeValue(-1L, ByteSizeUnit.BYTES);
    private final long sizeInBytes;
    private final ByteSizeUnit desiredUnit;

    public static ByteSizeValue of(long size, ByteSizeUnit unit) {
        if (size < -1L || size == -1L && unit != ByteSizeUnit.BYTES) {
            throw new IllegalArgumentException("Values less than -1 bytes are not supported: " + size + unit.getSuffix());
        }
        if (size > Long.MAX_VALUE / unit.toBytes(1L)) {
            throw new IllegalArgumentException("Values greater than 9223372036854775807 bytes are not supported: " + size + unit.getSuffix());
        }
        return ByteSizeValue.newByteSizeValue(size * unit.toBytes(1L), unit);
    }

    public static ByteSizeValue ofBytes(long size) {
        return ByteSizeValue.of(size, ByteSizeUnit.BYTES);
    }

    public static ByteSizeValue ofKb(long size) {
        return ByteSizeValue.of(size, ByteSizeUnit.KB);
    }

    public static ByteSizeValue ofMb(long size) {
        return ByteSizeValue.of(size, ByteSizeUnit.MB);
    }

    public static ByteSizeValue ofGb(long size) {
        return ByteSizeValue.of(size, ByteSizeUnit.GB);
    }

    public static ByteSizeValue ofTb(long size) {
        return ByteSizeValue.of(size, ByteSizeUnit.TB);
    }

    public static ByteSizeValue ofPb(long size) {
        return ByteSizeValue.of(size, ByteSizeUnit.PB);
    }

    static ByteSizeValue newByteSizeValue(long sizeInBytes, ByteSizeUnit desiredUnit) {
        if (desiredUnit == ByteSizeUnit.BYTES) {
            if (sizeInBytes == 0L) {
                return ZERO;
            }
            if (sizeInBytes == 1L) {
                return ONE;
            }
            if (sizeInBytes == -1L) {
                return MINUS_ONE;
            }
        }
        if (sizeInBytes < 0L) {
            throw new IllegalArgumentException("Values less than -1 bytes are not supported: " + sizeInBytes);
        }
        return new ByteSizeValue(sizeInBytes, desiredUnit);
    }

    public static ByteSizeValue readFrom(StreamInput in) throws IOException {
        long size = in.readZLong();
        ByteSizeUnit unit = ByteSizeUnit.readFrom(in);
        if (ByteSizeValue.alwaysUseBytes(in.getTransportVersion())) {
            return ByteSizeValue.newByteSizeValue(size, unit);
        }
        return ByteSizeValue.of(size, unit);
    }

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        if (ByteSizeValue.alwaysUseBytes(out.getTransportVersion())) {
            out.writeZLong(this.sizeInBytes);
        } else {
            out.writeZLong(Math.divideExact(this.sizeInBytes, this.desiredUnit.toBytes(1L)));
        }
        this.desiredUnit.writeTo(out);
    }

    private static boolean alwaysUseBytes(TransportVersion tv) {
        return tv.onOrAfter(TransportVersions.BYTE_SIZE_VALUE_ALWAYS_USES_BYTES) || tv.isPatchFrom(TransportVersions.V_9_0_0) || tv.between(TransportVersions.BYTE_SIZE_VALUE_ALWAYS_USES_BYTES_1, TransportVersions.REVERT_BYTE_SIZE_VALUE_ALWAYS_USES_BYTES_1);
    }

    ByteSizeValue(long sizeInBytes, ByteSizeUnit desiredUnit) {
        this.sizeInBytes = sizeInBytes;
        this.desiredUnit = desiredUnit;
    }

    long getSizeInBytes() {
        return this.sizeInBytes;
    }

    ByteSizeUnit getDesiredUnit() {
        return this.desiredUnit;
    }

    @Deprecated
    public int bytesAsInt() {
        long bytes = this.getBytes();
        if (bytes > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("size [" + this.toString() + "] is bigger than max int");
        }
        return (int)bytes;
    }

    public long getBytes() {
        return this.sizeInBytes;
    }

    public long getKb() {
        return this.getBytes() / ByteSizeUnit.KB.toBytes(1L);
    }

    public long getMb() {
        return this.getBytes() / ByteSizeUnit.MB.toBytes(1L);
    }

    public long getGb() {
        return this.getBytes() / ByteSizeUnit.GB.toBytes(1L);
    }

    public long getTb() {
        return this.getBytes() / ByteSizeUnit.TB.toBytes(1L);
    }

    public long getPb() {
        return this.getBytes() / ByteSizeUnit.PB.toBytes(1L);
    }

    public double getKbFrac() {
        return (double)this.getBytes() / 1024.0;
    }

    public double getMbFrac() {
        return (double)this.getBytes() / 1048576.0;
    }

    public double getGbFrac() {
        return (double)this.getBytes() / 1.073741824E9;
    }

    public double getTbFrac() {
        return (double)this.getBytes() / 1.099511627776E12;
    }

    public double getPbFrac() {
        return (double)this.getBytes() / 1.125899906842624E15;
    }

    public String getStringRep() {
        if (this.sizeInBytes <= 0L) {
            return String.valueOf(this.sizeInBytes);
        }
        long numUnits = this.sizeInBytes / this.desiredUnit.toBytes(1L);
        long residue = this.sizeInBytes % this.desiredUnit.toBytes(1L);
        if (residue == 0L) {
            return numUnits + this.desiredUnit.getSuffix();
        }
        return this.sizeInBytes + ByteSizeUnit.BYTES.getSuffix();
    }

    public String toString() {
        long bytes = this.getBytes();
        double value = bytes;
        String suffix = ByteSizeUnit.BYTES.getSuffix();
        if (bytes >= 0x4000000000000L) {
            value = this.getPbFrac();
            suffix = ByteSizeUnit.PB.getSuffix();
        } else if (bytes >= 0x10000000000L) {
            value = this.getTbFrac();
            suffix = ByteSizeUnit.TB.getSuffix();
        } else if (bytes >= 0x40000000L) {
            value = this.getGbFrac();
            suffix = ByteSizeUnit.GB.getSuffix();
        } else if (bytes >= 0x100000L) {
            value = this.getMbFrac();
            suffix = ByteSizeUnit.MB.getSuffix();
        } else if (bytes >= 1024L) {
            value = this.getKbFrac();
            suffix = ByteSizeUnit.KB.getSuffix();
        }
        return Strings.format1Decimals(value, suffix);
    }

    public static ByteSizeValue parseBytesSizeValue(String sValue, String settingName) throws ElasticsearchParseException {
        return ByteSizeValue.parseBytesSizeValue(sValue, null, settingName);
    }

    public static ByteSizeValue parseBytesSizeValue(String sValue, ByteSizeValue defaultValue, String settingName) throws ElasticsearchParseException {
        settingName = Objects.requireNonNull(settingName);
        if (sValue == null) {
            return defaultValue;
        }
        switch (sValue) {
            case "0": 
            case "0b": 
            case "0B": {
                return ZERO;
            }
            case "1b": 
            case "1B": {
                return ONE;
            }
            case "-1": 
            case "-1b": 
            case "-1B": {
                return MINUS_ONE;
            }
        }
        String lowerSValue = sValue.toLowerCase(Locale.ROOT).trim();
        if (lowerSValue.endsWith("k")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "k", ByteSizeUnit.KB, settingName);
        }
        if (lowerSValue.endsWith("kb")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "kb", ByteSizeUnit.KB, settingName);
        }
        if (lowerSValue.endsWith("m")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "m", ByteSizeUnit.MB, settingName);
        }
        if (lowerSValue.endsWith("mb")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "mb", ByteSizeUnit.MB, settingName);
        }
        if (lowerSValue.endsWith("g")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "g", ByteSizeUnit.GB, settingName);
        }
        if (lowerSValue.endsWith("gb")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "gb", ByteSizeUnit.GB, settingName);
        }
        if (lowerSValue.endsWith("t")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "t", ByteSizeUnit.TB, settingName);
        }
        if (lowerSValue.endsWith("tb")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "tb", ByteSizeUnit.TB, settingName);
        }
        if (lowerSValue.endsWith("p")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "p", ByteSizeUnit.PB, settingName);
        }
        if (lowerSValue.endsWith("pb")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "pb", ByteSizeUnit.PB, settingName);
        }
        if (lowerSValue.endsWith("b")) {
            return ByteSizeValue.parseBytes(lowerSValue, settingName, sValue);
        }
        throw new ElasticsearchParseException("failed to parse setting [{}] with value [{}] as a size in bytes: unit is missing or unrecognized", settingName, sValue);
    }

    private static ByteSizeValue parseBytes(String lowerSValue, String settingName, String initialInput) {
        String s = lowerSValue.substring(0, lowerSValue.length() - 1).trim();
        try {
            return ByteSizeValue.ofBytes(Long.parseLong(s));
        }
        catch (NumberFormatException e) {
            throw new ElasticsearchParseException("failed to parse setting [{}] with value [{}]", (Throwable)e, settingName, initialInput);
        }
        catch (IllegalArgumentException e) {
            throw new ElasticsearchParseException("failed to parse setting [{}] with value [{}] as a size in bytes", (Throwable)e, settingName, initialInput);
        }
    }

    private static ByteSizeValue parse(String initialInput, String normalized, String suffix, ByteSizeUnit unit, String settingName) {
        assert (unit != ByteSizeUnit.BYTES) : "Use parseBytes";
        String s = normalized.substring(0, normalized.length() - suffix.length()).trim();
        try {
            try {
                return ByteSizeValue.of(Long.parseLong(s), unit);
            }
            catch (NumberFormatException e) {
                BigDecimal decimalValue = ByteSizeValue.parseDecimal(s, settingName, initialInput, e);
                long sizeInBytes = ByteSizeValue.convertToBytes(decimalValue, unit, settingName, initialInput, e);
                return new ByteSizeValue(sizeInBytes, unit);
            }
        }
        catch (IllegalArgumentException e) {
            throw new ElasticsearchParseException("failed to parse setting [{}] with value [{}] as a size in bytes", (Throwable)e, settingName, initialInput);
        }
    }

    private static BigDecimal parseDecimal(String numericPortion, String settingName, String settingValue, NumberFormatException originalException) {
        BigDecimal decimalValue;
        try {
            decimalValue = new BigDecimal(numericPortion);
        }
        catch (NumberFormatException e) {
            ElasticsearchParseException toThrow = new ElasticsearchParseException("failed to parse setting [{}] with value [{}]", (Throwable)originalException, settingName, settingValue);
            toThrow.addSuppressed(e);
            throw toThrow;
        }
        if (decimalValue.signum() < 0) {
            throw new ElasticsearchParseException("failed to parse setting [{}] with value [{}]", settingName, settingValue);
        }
        if (decimalValue.scale() > 2) {
            throw new ElasticsearchParseException("failed to parse setting [{}] with more than two decimals in value [{}]", settingName, settingValue);
        }
        return decimalValue;
    }

    private static long convertToBytes(BigDecimal decimalValue, ByteSizeUnit unit, String settingName, String settingValue, NumberFormatException originalException) {
        BigDecimal sizeInBytes = decimalValue.multiply(new BigDecimal(unit.toBytes(1L)));
        try {
            return sizeInBytes.setScale(0, RoundingMode.UP).longValueExact();
        }
        catch (ArithmeticException e) {
            ElasticsearchParseException toThrow = new ElasticsearchParseException("failed to parse setting [{}] with value beyond {}: [{}]", (Throwable)e, settingName, Long.MAX_VALUE, settingValue);
            toThrow.addSuppressed(originalException);
            throw toThrow;
        }
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        return this.compareTo((ByteSizeValue)o) == 0;
    }

    public int hashCode() {
        return Long.hashCode(this.getBytes());
    }

    @Override
    public int compareTo(ByteSizeValue other) {
        return Long.compare(this.getBytes(), other.getBytes());
    }

    @Override
    public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
        return builder.value(this.toString());
    }

    public static ByteSizeValue add(ByteSizeValue x, ByteSizeValue y) {
        if (x.equals(MINUS_ONE) || y.equals(MINUS_ONE)) {
            throw new IllegalArgumentException("one of the arguments has -1 bytes");
        }
        return ByteSizeValue.ofBytes(Math.addExact(x.getBytes(), y.getBytes()));
    }

    public static ByteSizeValue subtract(ByteSizeValue x, ByteSizeValue y) {
        if (x.equals(MINUS_ONE) || y.equals(MINUS_ONE)) {
            throw new IllegalArgumentException("one of the arguments has -1 bytes");
        }
        ByteSizeValue res = ByteSizeValue.ofBytes(x.getBytes() - y.getBytes());
        if (res.equals(MINUS_ONE)) {
            throw new IllegalArgumentException("subtraction result has -1 bytes");
        }
        return res;
    }

    public static ByteSizeValue min(ByteSizeValue x, ByteSizeValue y) {
        if (x.equals(MINUS_ONE) || y.equals(MINUS_ONE)) {
            throw new IllegalArgumentException("one of the arguments has -1 bytes");
        }
        return x.compareTo(y) <= 0 ? x : y;
    }

    static class DeprecationLoggerHolder {
        static DeprecationLogger deprecationLogger = DeprecationLogger.getLogger(ByteSizeValue.class);

        DeprecationLoggerHolder() {
        }
    }
}

