/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.shard;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.SegmentInfos;
import org.apache.lucene.misc.store.HardlinkCopyDirectoryWrapper;
import org.apache.lucene.search.Sort;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FilterDirectory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.SubscribableListener;
import org.elasticsearch.action.support.ThreadedActionListener;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.IndexMetadataVerifier;
import org.elasticsearch.cluster.metadata.MappingMetadata;
import org.elasticsearch.cluster.routing.RecoverySource;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.lucene.Lucene;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.Maps;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.core.Releasables;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.index.mapper.MapperService;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardClosedException;
import org.elasticsearch.index.shard.IndexShardNotStartedException;
import org.elasticsearch.index.shard.IndexShardRecoveryException;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.shard.LocalShardSnapshot;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.shard.ShardSplittingQuery;
import org.elasticsearch.index.snapshots.IndexShardRestoreFailedException;
import org.elasticsearch.index.store.Store;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.indices.recovery.RecoveryState;
import org.elasticsearch.repositories.IndexId;
import org.elasticsearch.repositories.Repository;
import org.elasticsearch.repositories.RepositoryData;
import org.elasticsearch.threadpool.ThreadPool;

public final class StoreRecovery {
    private final Logger logger;
    private final ShardId shardId;

    StoreRecovery(ShardId shardId, Logger logger) {
        this.logger = logger;
        this.shardId = shardId;
    }

    void recoverFromStore(IndexShard indexShard, ActionListener<Boolean> listener) {
        if (this.canRecover(indexShard)) {
            RecoverySource.Type recoveryType = indexShard.recoveryState().getRecoverySource().getType();
            assert (recoveryType == RecoverySource.Type.EMPTY_STORE || recoveryType == RecoverySource.Type.EXISTING_STORE) : "expected store recovery type but was: " + String.valueOf((Object)recoveryType);
            this.logger.debug("starting recovery from store ...");
            ActionListener<Boolean> recoveryListener = this.recoveryListener(indexShard, listener);
            try {
                this.internalRecoverFromStore(indexShard, recoveryListener.map(ignored -> true));
            }
            catch (Exception e) {
                recoveryListener.onFailure(e);
            }
        } else {
            listener.onResponse(false);
        }
    }

    void recoverFromLocalShards(BiConsumer<MappingMetadata, ActionListener<Void>> mappingUpdateConsumer, IndexShard indexShard, List<LocalShardSnapshot> shards, ActionListener<Boolean> outerListener) {
        if (this.canRecover(indexShard)) {
            RecoverySource.Type recoveryType = indexShard.recoveryState().getRecoverySource().getType();
            assert (recoveryType == RecoverySource.Type.LOCAL_SHARDS) : "expected local shards recovery type: " + String.valueOf((Object)recoveryType);
            if (shards.isEmpty()) {
                throw new IllegalArgumentException("shards must not be empty");
            }
            Set indices = shards.stream().map(s -> s.getIndex()).collect(Collectors.toSet());
            if (indices.size() > 1) {
                throw new IllegalArgumentException("can't add shards from more than one index");
            }
            IndexMetadata sourceMetadata = shards.get(0).getIndexMetadata();
            SubscribableListener<Void> mappingStep = new SubscribableListener<Void>();
            if (sourceMetadata.mapping() == null) {
                mappingStep.onResponse(null);
            } else {
                mappingUpdateConsumer.accept(sourceMetadata.mapping(), mappingStep);
            }
            mappingStep.addListener(outerListener.delegateFailure((listener, ignored) -> {
                indexShard.mapperService().merge(sourceMetadata, MapperService.MergeReason.MAPPING_RECOVERY);
                Sort indexSort = indexShard.getIndexSort();
                boolean hasNested = indexShard.mapperService().hasNested();
                boolean isSplit = sourceMetadata.getNumberOfShards() < indexShard.indexSettings().getNumberOfShards();
                ActionListener<Boolean> recoveryListener = this.recoveryListener(indexShard, (ActionListener<Boolean>)listener);
                this.logger.debug("starting recovery from local shards {}", (Object)shards);
                try {
                    Directory directory = indexShard.store().directory();
                    Directory[] sources = (Directory[])shards.stream().map(LocalShardSnapshot::getSnapshotDirectory).toArray(Directory[]::new);
                    long maxSeqNo = shards.stream().mapToLong(LocalShardSnapshot::maxSeqNo).max().getAsLong();
                    long maxUnsafeAutoIdTimestamp = shards.stream().mapToLong(LocalShardSnapshot::maxUnsafeAutoIdTimestamp).max().getAsLong();
                    StoreRecovery.addIndices(indexShard.recoveryState().getIndex(), directory, indexSort, sources, maxSeqNo, maxUnsafeAutoIdTimestamp, indexShard.indexSettings().getIndexMetadata(), indexShard.shardId().id(), isSplit, hasNested);
                    this.internalRecoverFromStore(indexShard, recoveryListener.delegateFailure((delegate, v) -> ActionListener.completeWith(delegate, () -> {
                        indexShard.getEngine().forceMerge(false, -1, false, UUIDs.randomBase64UUID());
                        return true;
                    })));
                }
                catch (IOException e) {
                    recoveryListener.onFailure(new IndexShardRecoveryException(indexShard.shardId(), "failed to recover from local shards", e));
                }
                catch (Exception e) {
                    recoveryListener.onFailure(e);
                }
            }));
        } else {
            outerListener.onResponse(false);
        }
    }

    static void addIndices(RecoveryState.Index indexRecoveryStats, Directory target, Sort indexSort, Directory[] sources, long maxSeqNo, long maxUnsafeAutoIdTimestamp, IndexMetadata indexMetadata, int shardId, boolean split, boolean hasNested) throws IOException {
        assert (sources.length > 0);
        int luceneIndexCreatedVersionMajor = Lucene.readSegmentInfos(sources[0]).getIndexCreatedVersionMajor();
        HardlinkCopyDirectoryWrapper hardLinkOrCopyTarget = new HardlinkCopyDirectoryWrapper(target);
        IndexWriterConfig iwc = Lucene.indexWriterConfigWithNoMerging(null).setSoftDeletesField("__soft_deletes").setCommitOnClose(false).setOpenMode(IndexWriterConfig.OpenMode.CREATE).setIndexCreatedVersionMajor(luceneIndexCreatedVersionMajor);
        if (indexSort != null) {
            iwc.setIndexSort(indexSort);
            if (indexMetadata != null && indexMetadata.getCreationVersion().onOrAfter(IndexVersions.INDEX_SORTING_ON_NESTED)) {
                iwc.setParentField("__root_doc_for_nested");
            }
        }
        try (IndexWriter writer = new IndexWriter(new StatsDirectoryWrapper((Directory)hardLinkOrCopyTarget, indexRecoveryStats), iwc);){
            writer.addIndexes(sources);
            indexRecoveryStats.setFileDetailsComplete();
            if (split) {
                writer.deleteDocuments(new ShardSplittingQuery(indexMetadata, shardId, hasNested));
            }
            writer.setLiveCommitData(() -> {
                Map<String, String> liveCommitData = Maps.newMapWithExpectedSize(4);
                liveCommitData.put("max_seq_no", Long.toString(maxSeqNo));
                liveCommitData.put("local_checkpoint", Long.toString(maxSeqNo));
                liveCommitData.put("max_unsafe_auto_id_timestamp", Long.toString(maxUnsafeAutoIdTimestamp));
                liveCommitData.put("es_version", IndexVersion.current().toString());
                return liveCommitData.entrySet().iterator();
            });
            writer.commit();
        }
    }

    void recoverFromRepository(IndexShard indexShard, Repository repository, ActionListener<Boolean> listener) {
        try {
            if (this.canRecover(indexShard)) {
                RecoverySource.Type recoveryType = indexShard.recoveryState().getRecoverySource().getType();
                assert (recoveryType == RecoverySource.Type.SNAPSHOT) : "expected snapshot recovery type: " + String.valueOf((Object)recoveryType);
                RecoverySource.SnapshotRecoverySource recoverySource = (RecoverySource.SnapshotRecoverySource)indexShard.recoveryState().getRecoverySource();
                this.recoverFromRepository(indexShard, repository, recoverySource, this.recoveryListener(indexShard, listener).map(ignored -> true));
            } else {
                listener.onResponse(false);
            }
        }
        catch (Exception e) {
            listener.onFailure(e);
        }
    }

    private boolean canRecover(IndexShard indexShard) {
        if (indexShard.state() == IndexShardState.CLOSED) {
            return false;
        }
        if (!indexShard.routingEntry().primary()) {
            throw new IndexShardRecoveryException(this.shardId, "Trying to recover when the shard is in backup state", null);
        }
        return true;
    }

    private ActionListener<Boolean> recoveryListener(IndexShard indexShard, ActionListener<Boolean> listener) {
        return ActionListener.wrap(res -> {
            if (res.booleanValue()) {
                IndexShardState shardState = indexShard.state();
                RecoveryState recoveryState = indexShard.recoveryState();
                assert (shardState != IndexShardState.CREATED && shardState != IndexShardState.RECOVERING) : "recovery process of " + String.valueOf(this.shardId) + " didn't get to post_recovery. shardState [" + String.valueOf((Object)shardState) + "]";
                if (this.logger.isTraceEnabled()) {
                    RecoveryState.Index index = recoveryState.getIndex();
                    StringBuilder sb = new StringBuilder();
                    sb.append("    index    : files           [").append(index.totalFileCount()).append("] with total_size [").append(ByteSizeValue.ofBytes(index.totalBytes())).append("], took[").append(TimeValue.timeValueMillis(index.time())).append("]\n");
                    sb.append("             : recovered_files [").append(index.recoveredFileCount()).append("] with total_size [").append(ByteSizeValue.ofBytes(index.recoveredBytes())).append("]\n");
                    sb.append("             : reusing_files   [").append(index.reusedFileCount()).append("] with total_size [").append(ByteSizeValue.ofBytes(index.reusedBytes())).append("]\n");
                    sb.append("    verify_index    : took [").append(TimeValue.timeValueMillis(recoveryState.getVerifyIndex().time())).append("], check_index [").append(TimeValue.timeValueMillis(recoveryState.getVerifyIndex().checkIndexTime())).append("]\n");
                    sb.append("    translog : number_of_operations [").append(recoveryState.getTranslog().recoveredOperations()).append("], took [").append(TimeValue.timeValueMillis(recoveryState.getTranslog().time())).append("]");
                    this.logger.trace("recovery completed from [shard_store], took [{}]\n{}", (Object)TimeValue.timeValueMillis(recoveryState.getTimer().time()), (Object)sb);
                } else if (this.logger.isDebugEnabled()) {
                    this.logger.debug("recovery completed from [shard_store], took [{}]", (Object)TimeValue.timeValueMillis(recoveryState.getTimer().time()));
                }
            }
            listener.onResponse((Boolean)res);
        }, ex -> {
            if (ex instanceof IndexShardRecoveryException) {
                if (indexShard.state() == IndexShardState.CLOSED) {
                    listener.onResponse(false);
                    return;
                }
                if (ex.getCause() instanceof IndexShardClosedException || ex.getCause() instanceof IndexShardNotStartedException) {
                    listener.onResponse(false);
                    return;
                }
                listener.onFailure((Exception)ex);
            } else if (ex instanceof IndexShardClosedException || ex instanceof IndexShardNotStartedException) {
                listener.onResponse(false);
            } else if (indexShard.state() == IndexShardState.CLOSED) {
                listener.onResponse(false);
            } else {
                listener.onFailure(new IndexShardRecoveryException(this.shardId, "failed recovery", (Throwable)ex));
            }
        });
    }

    private void internalRecoverFromStore(IndexShard indexShard, ActionListener<Void> outerListener) {
        ArrayList releasables = new ArrayList(1);
        SubscribableListener.newForked(indexShard::preRecovery).andThen(l -> {
            RecoveryState recoveryState = indexShard.recoveryState();
            boolean indexShouldExists = recoveryState.getRecoverySource().getType() != RecoverySource.Type.EMPTY_STORE;
            indexShard.prepareForIndexRecovery();
            SegmentInfos si = null;
            Store store = indexShard.store();
            store.incRef();
            releasables.add(store::decRef);
            try {
                block16: {
                    store.failIfCorrupted();
                    try {
                        si = store.readLastCommittedSegmentsInfo();
                    }
                    catch (Exception e) {
                        Object files = "_unknown_";
                        try {
                            files = Arrays.toString(store.directory().listAll());
                        }
                        catch (Exception inner) {
                            files = (String)files + " (failure=" + ExceptionsHelper.stackTrace(inner) + ")";
                        }
                        if (!indexShouldExists) break block16;
                        throw new IndexShardRecoveryException(this.shardId, "shard allocated for local recovery (post api), should exist, but doesn't, current files: " + (String)files, e);
                    }
                }
                if (si != null && !indexShouldExists) {
                    this.logger.trace("cleaning existing shard, shouldn't exists");
                    Lucene.cleanLuceneIndex(store.directory());
                    si = null;
                }
            }
            catch (Exception e) {
                throw new IndexShardRecoveryException(this.shardId, "failed to fetch index version after copying it over", e);
            }
            if (recoveryState.getRecoverySource().getType() == RecoverySource.Type.LOCAL_SHARDS) {
                assert (indexShouldExists);
                StoreRecovery.bootstrap(indexShard);
                StoreRecovery.writeEmptyRetentionLeasesFile(indexShard);
            } else if (indexShouldExists) {
                if (recoveryState.getRecoverySource().shouldBootstrapNewHistoryUUID()) {
                    store.bootstrapNewHistory();
                    StoreRecovery.writeEmptyRetentionLeasesFile(indexShard);
                }
                RecoveryState.Index index = recoveryState.getIndex();
                try {
                    if (si != null) {
                        StoreRecovery.addRecoveredFileDetails(si, store, index);
                    }
                }
                catch (IOException e) {
                    this.logger.debug("failed to list file details", (Throwable)e);
                }
                index.setFileDetailsComplete();
            } else {
                store.createEmpty();
                String translogUUID = Translog.createEmptyTranslog(indexShard.shardPath().resolveTranslog(), -1L, this.shardId, indexShard.getPendingPrimaryTerm());
                store.associateIndexWithNewTranslog(translogUUID);
                StoreRecovery.writeEmptyRetentionLeasesFile(indexShard);
                indexShard.recoveryState().getIndex().setFileDetailsComplete();
            }
            indexShard.openEngineAndRecoverFromTranslog((ActionListener<Void>)l);
        }).andThen(l -> {
            indexShard.getEngine().fillSeqNoGaps(indexShard.getPendingPrimaryTerm());
            indexShard.finalizeRecovery();
            indexShard.postRecovery("post recovery from shard_store", (ActionListener<Void>)l);
        }).addListener(ActionListener.runBefore(outerListener.delegateResponse((l, e) -> {
            if (e instanceof IndexShardRecoveryException) {
                l.onFailure((Exception)e);
            } else {
                l.onFailure(new IndexShardRecoveryException(this.shardId, "failed to recover from gateway", (Throwable)e));
            }
        }), () -> Releasables.close(releasables)));
    }

    private static void writeEmptyRetentionLeasesFile(IndexShard indexShard) throws IOException {
        assert (indexShard.getRetentionLeases().leases().isEmpty()) : indexShard.getRetentionLeases();
        indexShard.persistRetentionLeases();
        assert (indexShard.loadRetentionLeases().leases().isEmpty());
    }

    private static void addRecoveredFileDetails(SegmentInfos si, Store store, RecoveryState.Index index) throws IOException {
        Directory directory = store.directory();
        for (String name : Lucene.files(si)) {
            long length = directory.fileLength(name);
            index.addFileDetail(name, length, true);
        }
    }

    private void recoverFromRepository(IndexShard indexShard, Repository repository, RecoverySource.SnapshotRecoverySource restoreSource, ActionListener<Void> outerListener) {
        assert (indexShard.shardRouting.primary()) : "only primary shards can recover from snapshot";
        this.logger.debug("restoring from {} ...", (Object)indexShard.recoveryState().getRecoverySource());
        SubscribableListener.newForked(indexShard::preRecovery).andThen(shardAndIndexIdsListener -> {
            record ShardAndIndexIds(IndexId indexId, ShardId shardId) {
            }
            if (restoreSource == null) {
                throw new IndexShardRestoreFailedException(this.shardId, "empty restore source");
            }
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("[{}] restoring shard [{}]", (Object)restoreSource.snapshot(), (Object)this.shardId);
            }
            RecoveryState.Translog translogState = indexShard.recoveryState().getTranslog();
            translogState.totalOperations(0);
            translogState.totalOperationsOnStart(0);
            indexShard.prepareForIndexRecovery();
            IndexId indexId = restoreSource.index();
            ShardId snapshotShardId = this.shardId.getIndexName().equals(indexId.getName()) ? this.shardId : new ShardId(indexId.getName(), "_na_", this.shardId.id());
            if (indexId.getId().equals("_na_")) {
                repository.getRepositoryData(EsExecutors.DIRECT_EXECUTOR_SERVICE, new ThreadedActionListener<RepositoryData>(indexShard.getThreadPool().generic(), shardAndIndexIdsListener.map(repositoryData -> new ShardAndIndexIds(repositoryData.resolveIndexId(indexId.getName()), snapshotShardId))));
            } else {
                shardAndIndexIdsListener.onResponse(new ShardAndIndexIds(indexId, snapshotShardId));
            }
        }).andThen((restoreListener, shardAndIndexId) -> {
            assert (indexShard.getEngineOrNull() == null);
            assert (ThreadPool.assertCurrentThreadPool("generic", "snapshot"));
            repository.restoreShard(indexShard.store(), restoreSource.snapshot().getSnapshotId(), shardAndIndexId.indexId(), shardAndIndexId.shardId(), indexShard.recoveryState(), (ActionListener<Void>)restoreListener);
        }).andThen(l -> {
            indexShard.getIndexEventListener().afterFilesRestoredFromRepository(indexShard);
            StoreRecovery.bootstrap(indexShard);
            StoreRecovery.writeEmptyRetentionLeasesFile(indexShard);
            indexShard.openEngineAndRecoverFromTranslog((ActionListener<Void>)l);
        }).andThen(l -> {
            indexShard.getEngine().fillSeqNoGaps(indexShard.getPendingPrimaryTerm());
            indexShard.finalizeRecovery();
            indexShard.postRecovery("restore done", (ActionListener<Void>)l);
        }).addListener(outerListener.delegateResponse((l, e) -> {
            if (e instanceof IndexShardRestoreFailedException) {
                l.onFailure((Exception)e);
            } else {
                l.onFailure(new IndexShardRestoreFailedException(this.shardId, "restore failed", (Throwable)e));
            }
        }));
    }

    @Deprecated(forRemoval=true)
    public static void bootstrap(IndexShard indexShard, Store store) throws IOException {
        assert (indexShard.store() == store);
        StoreRecovery.bootstrap(indexShard);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void bootstrap(IndexShard indexShard) throws IOException {
        assert (indexShard.routingEntry().primary());
        Store store = indexShard.store();
        store.incRef();
        try {
            Path translogLocation = indexShard.shardPath().resolveTranslog();
            if (!indexShard.hasTranslog()) {
                if (IndexMetadataVerifier.isReadOnlyVerified(indexShard.indexSettings().getIndexMetadata())) {
                    Translog.deleteAll(translogLocation);
                }
                return;
            }
            store.bootstrapNewHistory();
            SegmentInfos segmentInfos = store.readLastCommittedSegmentsInfo();
            long localCheckpoint = Long.parseLong(segmentInfos.userData.get("local_checkpoint"));
            String translogUUID = Translog.createEmptyTranslog(translogLocation, localCheckpoint, indexShard.shardId(), indexShard.getPendingPrimaryTerm());
            store.associateIndexWithNewTranslog(translogUUID);
        }
        finally {
            store.decRef();
        }
    }

    static final class StatsDirectoryWrapper
    extends FilterDirectory {
        private final RecoveryState.Index index;

        StatsDirectoryWrapper(Directory in, RecoveryState.Index indexRecoveryStats) {
            super(in);
            this.index = indexRecoveryStats;
        }

        @Override
        public void copyFrom(Directory from, String src, final String dest, IOContext context) throws IOException {
            final long l = from.fileLength(src);
            final AtomicBoolean copies = new AtomicBoolean(false);
            this.in.copyFrom(new FilterDirectory(from){

                @Override
                public IndexInput openInput(String name, IOContext context) throws IOException {
                    index.addFileDetail(dest, l, false);
                    copies.set(true);
                    final IndexInput input = this.in.openInput(name, context);
                    return new IndexInput("StatsDirectoryWrapper(" + input.toString() + ")"){

                        @Override
                        public void close() throws IOException {
                            input.close();
                        }

                        @Override
                        public long getFilePointer() {
                            throw new UnsupportedOperationException("only straight copies are supported");
                        }

                        @Override
                        public void seek(long pos) throws IOException {
                            throw new UnsupportedOperationException("seeks are not supported");
                        }

                        @Override
                        public long length() {
                            return input.length();
                        }

                        @Override
                        public IndexInput slice(String sliceDescription, long offset, long length) throws IOException {
                            throw new UnsupportedOperationException("slices are not supported");
                        }

                        @Override
                        public byte readByte() throws IOException {
                            throw new UnsupportedOperationException("use a buffer if you wanna perform well");
                        }

                        @Override
                        public void readBytes(byte[] b, int offset, int len) throws IOException {
                            input.readBytes(b, offset, len);
                            index.addRecoveredBytesToFile(dest, len);
                        }
                    };
                }
            }, src, dest, context);
            if (!copies.get()) {
                this.index.addFileDetail(dest, l, true);
            } else {
                assert (this.index.getFileDetails(dest) != null) : "File [" + dest + "] has no file details";
                assert (this.index.getFileDetails(dest).recovered() == l) : this.index.getFileDetails(dest).toString();
            }
        }
    }
}

