/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation.allocator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.IntroSorter;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.metadata.SingleNodeShutdownMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.ExpectedShardSizeEstimator;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.RoutingNodes;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.cluster.routing.UnassignedInfo;
import org.elasticsearch.cluster.routing.allocation.AllocateUnassignedDecision;
import org.elasticsearch.cluster.routing.allocation.AllocationDecision;
import org.elasticsearch.cluster.routing.allocation.MoveDecision;
import org.elasticsearch.cluster.routing.allocation.NodeAllocationResult;
import org.elasticsearch.cluster.routing.allocation.RoutingAllocation;
import org.elasticsearch.cluster.routing.allocation.ShardAllocationDecision;
import org.elasticsearch.cluster.routing.allocation.WriteLoadForecaster;
import org.elasticsearch.cluster.routing.allocation.allocator.DesiredBalanceMetrics;
import org.elasticsearch.cluster.routing.allocation.allocator.ShardsAllocator;
import org.elasticsearch.cluster.routing.allocation.allocator.WeightFunction;
import org.elasticsearch.cluster.routing.allocation.decider.AllocationDeciders;
import org.elasticsearch.cluster.routing.allocation.decider.Decision;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.Maps;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.gateway.PriorityComparator;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.injection.guice.Inject;

public class BalancedShardsAllocator
implements ShardsAllocator {
    private static final Logger logger = LogManager.getLogger(BalancedShardsAllocator.class);
    public static final Setting<Float> SHARD_BALANCE_FACTOR_SETTING = Setting.floatSetting("cluster.routing.allocation.balance.shard", 0.45f, 0.0f, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Float> INDEX_BALANCE_FACTOR_SETTING = Setting.floatSetting("cluster.routing.allocation.balance.index", 0.55f, 0.0f, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Float> WRITE_LOAD_BALANCE_FACTOR_SETTING = Setting.floatSetting("cluster.routing.allocation.balance.write_load", 10.0f, 0.0f, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Float> DISK_USAGE_BALANCE_FACTOR_SETTING = Setting.floatSetting("cluster.routing.allocation.balance.disk_usage", 2.0E-11f, 0.0f, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Float> THRESHOLD_SETTING = Setting.floatSetting("cluster.routing.allocation.balance.threshold", 1.0f, 1.0f, Setting.Property.Dynamic, Setting.Property.NodeScope);
    private volatile float indexBalanceFactor;
    private volatile float shardBalanceFactor;
    private volatile float writeLoadBalanceFactor;
    private volatile float diskUsageBalanceFactor;
    private volatile float threshold;
    private final WriteLoadForecaster writeLoadForecaster;

    public BalancedShardsAllocator() {
        this(Settings.EMPTY);
    }

    public BalancedShardsAllocator(Settings settings) {
        this(ClusterSettings.createBuiltInClusterSettings(settings), WriteLoadForecaster.DEFAULT);
    }

    public BalancedShardsAllocator(ClusterSettings clusterSettings) {
        this(clusterSettings, WriteLoadForecaster.DEFAULT);
    }

    @Inject
    public BalancedShardsAllocator(ClusterSettings clusterSettings, WriteLoadForecaster writeLoadForecaster) {
        clusterSettings.initializeAndWatch(SHARD_BALANCE_FACTOR_SETTING, value -> {
            this.shardBalanceFactor = value.floatValue();
        });
        clusterSettings.initializeAndWatch(INDEX_BALANCE_FACTOR_SETTING, value -> {
            this.indexBalanceFactor = value.floatValue();
        });
        clusterSettings.initializeAndWatch(WRITE_LOAD_BALANCE_FACTOR_SETTING, value -> {
            this.writeLoadBalanceFactor = value.floatValue();
        });
        clusterSettings.initializeAndWatch(DISK_USAGE_BALANCE_FACTOR_SETTING, value -> {
            this.diskUsageBalanceFactor = value.floatValue();
        });
        clusterSettings.initializeAndWatch(THRESHOLD_SETTING, value -> {
            this.threshold = value.floatValue();
        });
        this.writeLoadForecaster = writeLoadForecaster;
    }

    @Override
    public void allocate(RoutingAllocation allocation) {
        if (!allocation.metadata().indices().isEmpty()) {
            this.writeLoadForecaster.refreshLicense();
        }
        assert (!allocation.ignoreDisable());
        if (allocation.routingNodes().size() == 0) {
            this.failAllocationOfNewPrimaries(allocation);
            return;
        }
        WeightFunction weightFunction = new WeightFunction(this.shardBalanceFactor, this.indexBalanceFactor, this.writeLoadBalanceFactor, this.diskUsageBalanceFactor);
        Balancer balancer = new Balancer(this.writeLoadForecaster, allocation, weightFunction, this.threshold);
        balancer.allocateUnassigned();
        balancer.moveShards();
        balancer.balance();
        this.collectAndRecordNodeWeightStats(balancer, weightFunction, allocation);
    }

    private void collectAndRecordNodeWeightStats(Balancer balancer, WeightFunction weightFunction, RoutingAllocation allocation) {
        HashMap<DiscoveryNode, DesiredBalanceMetrics.NodeWeightStats> nodeLevelWeights = new HashMap<DiscoveryNode, DesiredBalanceMetrics.NodeWeightStats>();
        for (Map.Entry<String, ModelNode> entry : balancer.nodes.entrySet()) {
            ModelNode node = entry.getValue();
            float nodeWeight = weightFunction.calculateNodeWeight(node.numShards(), balancer.avgShardsPerNode(), node.writeLoad(), balancer.avgWriteLoadPerNode(), node.diskUsageInBytes(), balancer.avgDiskUsageInBytesPerNode());
            nodeLevelWeights.put(node.routingNode.node(), new DesiredBalanceMetrics.NodeWeightStats(node.numShards(), node.diskUsageInBytes(), node.writeLoad(), nodeWeight));
        }
        allocation.routingNodes().setBalanceWeightStatsPerNode(nodeLevelWeights);
    }

    @Override
    public ShardAllocationDecision decideShardAllocation(ShardRouting shard, RoutingAllocation allocation) {
        WeightFunction weightFunction = new WeightFunction(this.shardBalanceFactor, this.indexBalanceFactor, this.writeLoadBalanceFactor, this.diskUsageBalanceFactor);
        Balancer balancer = new Balancer(this.writeLoadForecaster, allocation, weightFunction, this.threshold);
        AllocateUnassignedDecision allocateUnassignedDecision = AllocateUnassignedDecision.NOT_TAKEN;
        MoveDecision moveDecision = MoveDecision.NOT_TAKEN;
        if (shard.unassigned()) {
            allocateUnassignedDecision = balancer.decideAllocateUnassigned(shard);
        } else {
            moveDecision = balancer.decideMove(shard);
            if (moveDecision.isDecisionTaken() && moveDecision.canRemain()) {
                moveDecision = balancer.decideRebalance(shard, moveDecision.getCanRemainDecision());
            }
        }
        return new ShardAllocationDecision(allocateUnassignedDecision, moveDecision);
    }

    private void failAllocationOfNewPrimaries(RoutingAllocation allocation) {
        RoutingNodes routingNodes = allocation.routingNodes();
        assert (routingNodes.size() == 0) : routingNodes;
        RoutingNodes.UnassignedShards.UnassignedIterator unassignedIterator = routingNodes.unassigned().iterator();
        while (unassignedIterator.hasNext()) {
            ShardRouting shardRouting = unassignedIterator.next();
            UnassignedInfo unassignedInfo = shardRouting.unassignedInfo();
            if (!shardRouting.primary() || unassignedInfo.lastAllocationStatus() != UnassignedInfo.AllocationStatus.NO_ATTEMPT) continue;
            unassignedIterator.updateUnassigned(new UnassignedInfo(unassignedInfo.reason(), unassignedInfo.message(), unassignedInfo.failure(), unassignedInfo.failedAllocations(), unassignedInfo.unassignedTimeNanos(), unassignedInfo.unassignedTimeMillis(), unassignedInfo.delayed(), UnassignedInfo.AllocationStatus.DECIDERS_NO, unassignedInfo.failedNodeIds(), unassignedInfo.lastAllocatedNodeId()), shardRouting.recoverySource(), allocation.changes());
        }
    }

    public float getThreshold() {
        return this.threshold;
    }

    public float getIndexBalance() {
        return this.indexBalanceFactor;
    }

    public float getShardBalance() {
        return this.shardBalanceFactor;
    }

    public static class Balancer {
        private final WriteLoadForecaster writeLoadForecaster;
        private final RoutingAllocation allocation;
        private final RoutingNodes routingNodes;
        private final Metadata metadata;
        private final WeightFunction weightFunction;
        private final float threshold;
        private final float avgShardsPerNode;
        private final double avgWriteLoadPerNode;
        private final double avgDiskUsageInBytesPerNode;
        private final Map<String, ModelNode> nodes;
        private final NodeSorter sorter;
        private static final Comparator<ShardRouting> BY_DESCENDING_SHARD_ID = Comparator.comparing(ShardRouting::shardId).reversed();

        private Balancer(WriteLoadForecaster writeLoadForecaster, RoutingAllocation allocation, WeightFunction weightFunction, float threshold) {
            this.writeLoadForecaster = writeLoadForecaster;
            this.allocation = allocation;
            this.routingNodes = allocation.routingNodes();
            this.metadata = allocation.metadata();
            this.weightFunction = weightFunction;
            this.threshold = threshold;
            this.avgShardsPerNode = WeightFunction.avgShardPerNode(this.metadata, this.routingNodes);
            this.avgWriteLoadPerNode = WeightFunction.avgWriteLoadPerNode(writeLoadForecaster, this.metadata, this.routingNodes);
            this.avgDiskUsageInBytesPerNode = WeightFunction.avgDiskUsageInBytesPerNode(allocation.clusterInfo(), this.metadata, this.routingNodes);
            this.nodes = Collections.unmodifiableMap(this.buildModelFromAssigned());
            this.sorter = this.newNodeSorter();
        }

        private static long getShardDiskUsageInBytes(ShardRouting shardRouting, IndexMetadata indexMetadata, ClusterInfo clusterInfo) {
            if (indexMetadata.ignoreDiskWatermarks()) {
                return 0L;
            }
            return Math.max(indexMetadata.getForecastedShardSizeInBytes().orElse(0L), clusterInfo.getShardSize(shardRouting, 0L));
        }

        private float getShardWriteLoad(String index) {
            return (float)this.writeLoadForecaster.getForecastedWriteLoad(this.metadata.index(index)).orElse(0.0);
        }

        private float maxShardSizeBytes(String index) {
            IndexMetadata indexMetadata = this.metadata.index(index);
            if (indexMetadata.ignoreDiskWatermarks()) {
                return 0.0f;
            }
            long maxShardSizeBytes = indexMetadata.getForecastedShardSizeInBytes().orElse(0L);
            for (int shard = 0; shard < indexMetadata.getNumberOfShards(); ++shard) {
                ShardId shardId = new ShardId(indexMetadata.getIndex(), shard);
                maxShardSizeBytes = Balancer.maxWithNullable(maxShardSizeBytes, this.allocation.clusterInfo().getShardSize(shardId, true));
                maxShardSizeBytes = Balancer.maxWithNullable(maxShardSizeBytes, this.allocation.clusterInfo().getShardSize(shardId, false));
            }
            return maxShardSizeBytes;
        }

        private static long maxWithNullable(long accumulator, Long newValue) {
            return newValue == null ? accumulator : Math.max(accumulator, newValue);
        }

        private ModelNode[] nodesArray() {
            return (ModelNode[])this.nodes.values().toArray(ModelNode[]::new);
        }

        public float avgShardsPerNode(String index) {
            return (float)this.metadata.index(index).getTotalNumberOfShards() / (float)this.nodes.size();
        }

        public float avgShardsPerNode() {
            return this.avgShardsPerNode;
        }

        public double avgWriteLoadPerNode() {
            return this.avgWriteLoadPerNode;
        }

        public double avgDiskUsageInBytesPerNode() {
            return this.avgDiskUsageInBytesPerNode;
        }

        private NodeSorter newNodeSorter() {
            return new NodeSorter(this.nodesArray(), this.weightFunction, this);
        }

        private static float absDelta(float lower, float higher) {
            assert (higher >= lower) : higher + " lt " + lower + " but was expected to be gte";
            return Math.abs(higher - lower);
        }

        private static boolean lessThan(float delta, float threshold) {
            return delta <= threshold + 0.001f;
        }

        private void balance() {
            if (logger.isTraceEnabled()) {
                logger.trace("Start balancing cluster");
            }
            if (this.allocation.hasPendingAsyncFetch()) {
                logger.debug("skipping rebalance due to in-flight shard/store fetches");
                return;
            }
            if (this.allocation.deciders().canRebalance(this.allocation).type() != Decision.Type.YES) {
                logger.trace("skipping rebalance as it is disabled");
                return;
            }
            if (this.nodes.size() < 2) {
                logger.trace("skipping rebalance as single node only");
                return;
            }
            this.balanceByWeights();
        }

        private MoveDecision decideRebalance(ShardRouting shard, Decision canRemain) {
            if (!shard.started()) {
                return MoveDecision.NOT_TAKEN;
            }
            Decision canRebalance = this.allocation.deciders().canRebalance(shard, this.allocation);
            this.sorter.reset(shard.getIndexName());
            ModelNode[] modelNodes = this.sorter.modelNodes;
            String currentNodeId = shard.currentNodeId();
            ModelNode currentNode = null;
            for (ModelNode node : modelNodes) {
                if (!node.getNodeId().equals(currentNodeId)) continue;
                currentNode = node;
                break;
            }
            assert (currentNode != null) : "currently assigned node could not be found";
            String idxName = shard.getIndexName();
            float currentWeight = this.weightFunction.calculateNodeWeightWithIndex(this, currentNode, idxName);
            AllocationDeciders deciders = this.allocation.deciders();
            Decision.Type rebalanceDecisionType = Decision.Type.NO;
            ModelNode targetNode = null;
            ArrayList<Tuple<ModelNode, Decision>> betterBalanceNodes = new ArrayList<Tuple<ModelNode, Decision>>();
            ArrayList<Tuple<ModelNode, Decision>> sameBalanceNodes = new ArrayList<Tuple<ModelNode, Decision>>();
            ArrayList<Tuple<ModelNode, Decision>> worseBalanceNodes = new ArrayList<Tuple<ModelNode, Decision>>();
            for (ModelNode modelNode : modelNodes) {
                if (modelNode == currentNode) continue;
                Decision decision = deciders.canAllocate(shard, modelNode.getRoutingNode(), this.allocation);
                float nodeWeight = this.weightFunction.calculateNodeWeightWithIndex(this, modelNode, idxName);
                boolean betterWeightThanCurrent = nodeWeight <= currentWeight;
                boolean rebalanceConditionsMet = false;
                if (betterWeightThanCurrent) {
                    float localThreshold;
                    float currentDelta = Balancer.absDelta(nodeWeight, currentWeight);
                    boolean deltaAboveThreshold = !Balancer.lessThan(currentDelta, localThreshold = this.sorter.minWeightDelta() * this.threshold);
                    boolean betterWeightWithShardAdded = nodeWeight + localThreshold < currentWeight;
                    boolean bl = rebalanceConditionsMet = deltaAboveThreshold && betterWeightWithShardAdded;
                    if (rebalanceConditionsMet && decision.type().higherThan(rebalanceDecisionType)) {
                        rebalanceDecisionType = decision.type();
                        targetNode = modelNode;
                    }
                }
                Tuple<ModelNode, Decision> nodeResult = Tuple.tuple(modelNode, decision);
                if (rebalanceConditionsMet) {
                    betterBalanceNodes.add(nodeResult);
                    continue;
                }
                if (betterWeightThanCurrent) {
                    sameBalanceNodes.add(nodeResult);
                    continue;
                }
                worseBalanceNodes.add(nodeResult);
            }
            int weightRanking = 0;
            ArrayList<NodeAllocationResult> nodeDecisions = new ArrayList<NodeAllocationResult>(modelNodes.length - 1);
            for (Tuple tuple : betterBalanceNodes) {
                nodeDecisions.add(new NodeAllocationResult(((ModelNode)tuple.v1()).routingNode.node(), AllocationDecision.fromDecisionType(((Decision)tuple.v2()).type()), (Decision)tuple.v2(), ++weightRanking));
            }
            int currentNodeWeightRanking = ++weightRanking;
            for (Tuple tuple : sameBalanceNodes) {
                AllocationDecision nodeDecision = ((Decision)tuple.v2()).type() == Decision.Type.NO ? AllocationDecision.NO : AllocationDecision.WORSE_BALANCE;
                nodeDecisions.add(new NodeAllocationResult(((ModelNode)tuple.v1()).routingNode.node(), nodeDecision, (Decision)tuple.v2(), currentNodeWeightRanking));
            }
            for (Tuple tuple : worseBalanceNodes) {
                AllocationDecision nodeDecision = ((Decision)tuple.v2()).type() == Decision.Type.NO ? AllocationDecision.NO : AllocationDecision.WORSE_BALANCE;
                nodeDecisions.add(new NodeAllocationResult(((ModelNode)tuple.v1()).routingNode.node(), nodeDecision, (Decision)tuple.v2(), ++weightRanking));
            }
            if (canRebalance.type() != Decision.Type.YES || this.allocation.hasPendingAsyncFetch()) {
                return MoveDecision.rebalance(canRemain, canRebalance, this.allocation.hasPendingAsyncFetch() ? AllocationDecision.AWAITING_INFO : AllocationDecision.fromDecisionType(canRebalance.type()), null, currentNodeWeightRanking, nodeDecisions);
            }
            return MoveDecision.rebalance(canRemain, canRebalance, AllocationDecision.fromDecisionType(rebalanceDecisionType), targetNode != null ? targetNode.routingNode.node() : null, currentNodeWeightRanking, nodeDecisions);
        }

        private void balanceByWeights() {
            AllocationDeciders deciders = this.allocation.deciders();
            ModelNode[] modelNodes = this.sorter.modelNodes;
            float[] weights = this.sorter.weights;
            block0: for (String index : this.buildWeightOrderedIndices()) {
                IndexMetadata indexMetadata = this.metadata.index(index);
                int relevantNodes = 0;
                for (int i = 0; i < modelNodes.length; ++i) {
                    ModelNode modelNode = modelNodes[i];
                    if (modelNode.getIndex(index) == null && deciders.canAllocate(indexMetadata, modelNode.getRoutingNode(), this.allocation).type() == Decision.Type.NO) continue;
                    modelNodes[i] = modelNodes[relevantNodes];
                    modelNodes[relevantNodes] = modelNode;
                    ++relevantNodes;
                }
                if (relevantNodes < 2) continue;
                this.sorter.reset(index, 0, relevantNodes);
                int lowIdx = 0;
                int highIdx = relevantNodes - 1;
                float localThreshold = this.sorter.minWeightDelta() * this.threshold;
                while (true) {
                    ModelNode minNode = modelNodes[lowIdx];
                    ModelNode maxNode = modelNodes[highIdx];
                    if (maxNode.numShards(index) > 0) {
                        float delta = Balancer.absDelta(weights[lowIdx], weights[highIdx]);
                        if (Balancer.lessThan(delta, localThreshold)) {
                            if (lowIdx <= 0 || highIdx - 1 <= 0 || !(Balancer.absDelta(weights[0], weights[highIdx - 1]) > localThreshold)) {
                                if (!logger.isTraceEnabled()) continue block0;
                                logger.trace("Stop balancing index [{}]  min_node [{}] weight: [{}] max_node [{}] weight: [{}] delta: [{}]", (Object)index, (Object)maxNode.getNodeId(), (Object)Float.valueOf(weights[highIdx]), (Object)minNode.getNodeId(), (Object)Float.valueOf(weights[lowIdx]), (Object)Float.valueOf(delta));
                                continue block0;
                            }
                        } else {
                            if (logger.isTraceEnabled()) {
                                logger.trace("Balancing from node [{}] weight: [{}] to node [{}] weight: [{}] delta: [{}]", (Object)maxNode.getNodeId(), (Object)Float.valueOf(weights[highIdx]), (Object)minNode.getNodeId(), (Object)Float.valueOf(weights[lowIdx]), (Object)Float.valueOf(delta));
                            }
                            if (delta <= localThreshold) {
                                logger.trace("Couldn't find shard to relocate from node [{}] to node [{}]", (Object)maxNode.getNodeId(), (Object)minNode.getNodeId());
                            } else if (this.tryRelocateShard(minNode, maxNode, index)) {
                                weights[lowIdx] = this.sorter.weight(modelNodes[lowIdx]);
                                weights[highIdx] = this.sorter.weight(modelNodes[highIdx]);
                                this.sorter.sort(0, relevantNodes);
                                lowIdx = 0;
                                highIdx = relevantNodes - 1;
                                continue;
                            }
                        }
                    }
                    if (lowIdx < highIdx - 1) {
                        ++lowIdx;
                        continue;
                    }
                    if (lowIdx <= 0) continue block0;
                    lowIdx = 0;
                    --highIdx;
                }
            }
        }

        private String[] buildWeightOrderedIndices() {
            final String[] indices = (String[])this.allocation.routingTable().indicesRouting().keySet().toArray(String[]::new);
            final float[] deltas = new float[indices.length];
            for (int i = 0; i < deltas.length; ++i) {
                this.sorter.reset(indices[i]);
                deltas[i] = this.sorter.delta();
            }
            new IntroSorter(this){
                float pivotWeight;

                @Override
                protected void swap(int i, int j) {
                    String tmpIdx = indices[i];
                    indices[i] = indices[j];
                    indices[j] = tmpIdx;
                    float tmpDelta = deltas[i];
                    deltas[i] = deltas[j];
                    deltas[j] = tmpDelta;
                }

                @Override
                protected int compare(int i, int j) {
                    return Float.compare(deltas[j], deltas[i]);
                }

                @Override
                protected void setPivot(int i) {
                    this.pivotWeight = deltas[i];
                }

                @Override
                protected int comparePivot(int j) {
                    return Float.compare(deltas[j], this.pivotWeight);
                }
            }.sort(0, deltas.length);
            return indices;
        }

        public void moveShards() {
            Iterator<ShardRouting> it = this.allocation.routingNodes().nodeInterleavedShardIterator();
            while (it.hasNext()) {
                ShardRouting shardRouting = it.next();
                MoveDecision moveDecision = this.decideMove(shardRouting);
                if (moveDecision.isDecisionTaken() && moveDecision.forceMove()) {
                    ModelNode sourceNode = this.nodes.get(shardRouting.currentNodeId());
                    ModelNode targetNode = this.nodes.get(moveDecision.getTargetNode().getId());
                    sourceNode.removeShard(shardRouting);
                    Tuple<ShardRouting, ShardRouting> relocatingShards = this.routingNodes.relocateShard(shardRouting, targetNode.getNodeId(), this.allocation.clusterInfo().getShardSize(shardRouting, -1L), "move", this.allocation.changes());
                    targetNode.addShard(relocatingShards.v2());
                    if (!logger.isTraceEnabled()) continue;
                    logger.trace("Moved shard [{}] to node [{}]", (Object)shardRouting, (Object)targetNode.getRoutingNode());
                    continue;
                }
                if (!moveDecision.isDecisionTaken() || moveDecision.canRemain()) continue;
                logger.trace("[{}][{}] can't move", (Object)shardRouting.index(), (Object)shardRouting.id());
            }
        }

        public MoveDecision decideMove(ShardRouting shardRouting) {
            boolean shardsOnReplacedNode;
            if (!shardRouting.started()) {
                return MoveDecision.NOT_TAKEN;
            }
            ModelNode sourceNode = this.nodes.get(shardRouting.currentNodeId());
            assert (sourceNode != null && sourceNode.containsShard(shardRouting));
            RoutingNode routingNode = sourceNode.getRoutingNode();
            Decision canRemain = this.allocation.deciders().canRemain(shardRouting, routingNode, this.allocation);
            if (canRemain.type() != Decision.Type.NO) {
                return MoveDecision.remain(canRemain);
            }
            this.sorter.reset(shardRouting.getIndexName());
            MoveDecision moveDecision = this.decideMove(shardRouting, sourceNode, canRemain, this::decideCanAllocate);
            if (!moveDecision.canRemain() && !moveDecision.forceMove() && (shardsOnReplacedNode = this.allocation.metadata().nodeShutdowns().contains(shardRouting.currentNodeId(), SingleNodeShutdownMetadata.Type.REPLACE))) {
                return this.decideMove(shardRouting, sourceNode, canRemain, this::decideCanForceAllocateForVacate);
            }
            return moveDecision;
        }

        private MoveDecision decideMove(ShardRouting shardRouting, ModelNode sourceNode, Decision remainDecision, BiFunction<ShardRouting, RoutingNode, Decision> decider) {
            boolean explain = this.allocation.debugDecision();
            Decision.Type bestDecision = Decision.Type.NO;
            RoutingNode targetNode = null;
            ArrayList<NodeAllocationResult> nodeResults = explain ? new ArrayList<NodeAllocationResult>() : null;
            int weightRanking = 0;
            for (ModelNode currentNode : this.sorter.modelNodes) {
                if (currentNode == sourceNode) continue;
                RoutingNode target = currentNode.getRoutingNode();
                Decision allocationDecision = decider.apply(shardRouting, target);
                if (explain) {
                    nodeResults.add(new NodeAllocationResult(currentNode.getRoutingNode().node(), allocationDecision, ++weightRanking));
                }
                if (!allocationDecision.type().higherThan(bestDecision) || (bestDecision = allocationDecision.type()) != Decision.Type.YES) continue;
                targetNode = target;
                if (!explain) break;
            }
            return MoveDecision.move(remainDecision, AllocationDecision.fromDecisionType(bestDecision), targetNode != null ? targetNode.node() : null, nodeResults);
        }

        private Decision decideCanAllocate(ShardRouting shardRouting, RoutingNode target) {
            return this.allocation.deciders().canAllocate(shardRouting, target, this.allocation);
        }

        private Decision decideCanForceAllocateForVacate(ShardRouting shardRouting, RoutingNode target) {
            return this.allocation.deciders().canForceAllocateDuringReplace(shardRouting, target, this.allocation);
        }

        private Map<String, ModelNode> buildModelFromAssigned() {
            Map<String, ModelNode> nodes = Maps.newMapWithExpectedSize(this.routingNodes.size());
            for (RoutingNode rn : this.routingNodes) {
                ModelNode node = new ModelNode(this.writeLoadForecaster, this.metadata, this.allocation.clusterInfo(), rn);
                nodes.put(rn.nodeId(), node);
                for (ShardRouting shard : rn) {
                    assert (rn.nodeId().equals(shard.currentNodeId()));
                    if (shard.state() == ShardRoutingState.RELOCATING) continue;
                    node.addShard(shard);
                    if (!logger.isTraceEnabled()) continue;
                    logger.trace("Assigned shard [{}] to node [{}]", (Object)shard, (Object)node.getNodeId());
                }
            }
            return nodes;
        }

        private void allocateUnassigned() {
            RoutingNodes.UnassignedShards unassigned = this.routingNodes.unassigned();
            assert (!this.nodes.isEmpty());
            if (logger.isTraceEnabled()) {
                logger.trace("Start allocating unassigned shards");
            }
            if (unassigned.isEmpty()) {
                return;
            }
            PriorityComparator secondaryComparator = PriorityComparator.getAllocationComparator(this.allocation);
            Comparator comparator = (o1, o2) -> {
                if (o1.primary() ^ o2.primary()) {
                    return o1.primary() ? -1 : 1;
                }
                if (o1.getIndexName().compareTo(o2.getIndexName()) == 0) {
                    return o1.getId() - o2.getId();
                }
                int secondary = secondaryComparator.compare((ShardRouting)o1, (ShardRouting)o2);
                assert (secondary != 0) : "Index names are equal, should be returned early.";
                return secondary;
            };
            ShardRouting[] primary = unassigned.drain();
            ShardRouting[] secondary = new ShardRouting[primary.length];
            int secondaryLength = 0;
            int primaryLength = primary.length;
            ArrayUtil.timSort(primary, comparator);
            do {
                for (int i = 0; i < primaryLength; ++i) {
                    long shardSize;
                    ModelNode minNode;
                    ShardRouting shard = primary[i];
                    AllocateUnassignedDecision allocationDecision = this.decideAllocateUnassigned(shard);
                    String assignedNodeId = allocationDecision.getTargetNode() != null ? allocationDecision.getTargetNode().getId() : null;
                    ModelNode modelNode = minNode = assignedNodeId != null ? this.nodes.get(assignedNodeId) : null;
                    if (allocationDecision.getAllocationDecision() == AllocationDecision.YES) {
                        if (logger.isTraceEnabled()) {
                            logger.trace("Assigned shard [{}] to [{}]", (Object)shard, (Object)minNode.getNodeId());
                        }
                        shardSize = ExpectedShardSizeEstimator.getExpectedShardSize(shard, -1L, this.allocation);
                        shard = this.routingNodes.initializeShard(shard, minNode.getNodeId(), null, shardSize, this.allocation.changes());
                        minNode.addShard(shard);
                        if (shard.primary()) continue;
                        while (i < primaryLength - 1 && comparator.compare(primary[i], primary[i + 1]) == 0) {
                            secondary[secondaryLength++] = primary[++i];
                        }
                        continue;
                    }
                    if (logger.isTraceEnabled()) {
                        logger.trace("No eligible node found to assign shard [{}] allocation_status [{}]", (Object)shard, (Object)allocationDecision.getAllocationStatus());
                    }
                    if (minNode != null) {
                        assert (allocationDecision.getAllocationStatus() == UnassignedInfo.AllocationStatus.DECIDERS_THROTTLED);
                        shardSize = ExpectedShardSizeEstimator.getExpectedShardSize(shard, -1L, this.allocation);
                        minNode.addShard(shard.initialize(minNode.getNodeId(), null, shardSize));
                    } else if (logger.isTraceEnabled()) {
                        logger.trace("No Node found to assign shard [{}]", (Object)shard);
                    }
                    unassigned.ignoreShard(shard, allocationDecision.getAllocationStatus(), this.allocation.changes());
                    if (shard.primary()) continue;
                    while (i < primaryLength - 1 && comparator.compare(primary[i], primary[i + 1]) == 0) {
                        unassigned.ignoreShard(primary[++i], allocationDecision.getAllocationStatus(), this.allocation.changes());
                    }
                }
                primaryLength = secondaryLength;
                ShardRouting[] tmp = primary;
                primary = secondary;
                secondary = tmp;
                secondaryLength = 0;
            } while (primaryLength > 0);
        }

        private AllocateUnassignedDecision decideAllocateUnassigned(ShardRouting shard) {
            if (shard.assignedToNode()) {
                return AllocateUnassignedDecision.NOT_TAKEN;
            }
            boolean explain = this.allocation.debugDecision();
            Decision shardLevelDecision = this.allocation.deciders().canAllocate(shard, this.allocation);
            if (shardLevelDecision.type() == Decision.Type.NO && !explain) {
                return AllocateUnassignedDecision.no(UnassignedInfo.AllocationStatus.DECIDERS_NO, null);
            }
            float minWeight = Float.POSITIVE_INFINITY;
            ModelNode minNode = null;
            Decision decision = null;
            HashMap<String, NodeAllocationResult> nodeExplanationMap = explain ? new HashMap<String, NodeAllocationResult>() : null;
            ArrayList<Tuple<String, Float>> nodeWeights = explain ? new ArrayList<Tuple<String, Float>>() : null;
            for (ModelNode node : this.nodes.values()) {
                boolean updateMinNode;
                float currentWeight;
                if (node.containsShard(shard) && !explain || (currentWeight = this.weightFunction.calculateNodeWeightWithIndex(this, node, shard.getIndexName())) > minWeight && !explain) continue;
                Decision decision2 = this.allocation.deciders().canAllocate(shard, node.getRoutingNode(), this.allocation);
                if (explain) {
                    nodeExplanationMap.put(node.getNodeId(), new NodeAllocationResult(node.getRoutingNode().node(), decision2, 0));
                    nodeWeights.add(Tuple.tuple(node.getNodeId(), Float.valueOf(currentWeight)));
                }
                if (decision2.type() != Decision.Type.YES && decision2.type() != Decision.Type.THROTTLE) continue;
                if (currentWeight == minWeight) {
                    if (decision2.type() == decision.type()) {
                        int repId = shard.id();
                        int nodeHigh = node.highestPrimary(shard.index().getName());
                        int minNodeHigh = minNode.highestPrimary(shard.getIndexName());
                        updateMinNode = (nodeHigh > repId && minNodeHigh > repId || nodeHigh < repId && minNodeHigh < repId) && nodeHigh < minNodeHigh || nodeHigh > repId && minNodeHigh < repId;
                    } else {
                        updateMinNode = decision2.type() == Decision.Type.YES;
                    }
                } else {
                    boolean bl = updateMinNode = currentWeight < minWeight;
                }
                if (!updateMinNode) continue;
                minNode = node;
                minWeight = currentWeight;
                decision = decision2;
            }
            if (decision == null) {
                decision = Decision.NO;
            }
            ArrayList<NodeAllocationResult> nodeDecisions = null;
            if (explain) {
                nodeDecisions = new ArrayList<NodeAllocationResult>();
                nodeWeights.sort((nodeWeight1, nodeWeight2) -> Float.compare(((Float)nodeWeight1.v2()).floatValue(), ((Float)nodeWeight2.v2()).floatValue()));
                int weightRanking = 0;
                for (Tuple tuple : nodeWeights) {
                    NodeAllocationResult current = (NodeAllocationResult)nodeExplanationMap.get(tuple.v1());
                    nodeDecisions.add(new NodeAllocationResult(current.getNode(), current.getCanAllocateDecision(), ++weightRanking));
                }
            }
            return AllocateUnassignedDecision.fromDecision(decision, minNode != null ? minNode.routingNode.node() : null, nodeDecisions);
        }

        private boolean tryRelocateShard(ModelNode minNode, ModelNode maxNode, String idx) {
            ModelIndex index = maxNode.getIndex(idx);
            if (index != null) {
                logger.trace("Try relocating shard of [{}] from [{}] to [{}]", (Object)idx, (Object)maxNode.getNodeId(), (Object)minNode.getNodeId());
                Iterable shardRoutings = StreamSupport.stream(index.spliterator(), false).filter(ShardRouting::started).sorted(BY_DESCENDING_SHARD_ID)::iterator;
                AllocationDeciders deciders = this.allocation.deciders();
                for (ShardRouting shard : shardRoutings) {
                    Decision allocationDecision;
                    Decision rebalanceDecision = deciders.canRebalance(shard, this.allocation);
                    if (rebalanceDecision.type() == Decision.Type.NO || (allocationDecision = deciders.canAllocate(shard, minNode.getRoutingNode(), this.allocation)).type() == Decision.Type.NO) continue;
                    Decision.Multi decision = new Decision.Multi().add(allocationDecision).add(rebalanceDecision);
                    maxNode.removeShard(shard);
                    long shardSize = this.allocation.clusterInfo().getShardSize(shard, -1L);
                    assert (((Decision)decision).type() == Decision.Type.YES || ((Decision)decision).type() == Decision.Type.THROTTLE) : ((Decision)decision).type();
                    logger.debug("decision [{}]: relocate [{}] from [{}] to [{}]", (Object)((Decision)decision).type(), (Object)shard, (Object)maxNode.getNodeId(), (Object)minNode.getNodeId());
                    minNode.addShard(((Decision)decision).type() == Decision.Type.YES ? this.routingNodes.relocateShard(shard, minNode.getNodeId(), shardSize, "rebalance", this.allocation.changes()).v1() : shard.relocate(minNode.getNodeId(), shardSize));
                    return true;
                }
            }
            logger.trace("No shards of [{}] can relocate from [{}] to [{}]", (Object)idx, (Object)maxNode.getNodeId(), (Object)minNode.getNodeId());
            return false;
        }
    }

    static class ModelNode
    implements Iterable<ModelIndex> {
        private int numShards = 0;
        private double writeLoad = 0.0;
        private double diskUsageInBytes = 0.0;
        private final WriteLoadForecaster writeLoadForecaster;
        private final Metadata metadata;
        private final ClusterInfo clusterInfo;
        private final RoutingNode routingNode;
        private final Map<String, ModelIndex> indices;

        ModelNode(WriteLoadForecaster writeLoadForecaster, Metadata metadata, ClusterInfo clusterInfo, RoutingNode routingNode) {
            this.writeLoadForecaster = writeLoadForecaster;
            this.metadata = metadata;
            this.clusterInfo = clusterInfo;
            this.routingNode = routingNode;
            this.indices = Maps.newMapWithExpectedSize(routingNode.size() + 10);
        }

        public ModelIndex getIndex(String indexName) {
            return this.indices.get(indexName);
        }

        public String getNodeId() {
            return this.routingNode.nodeId();
        }

        public RoutingNode getRoutingNode() {
            return this.routingNode;
        }

        public int numShards() {
            return this.numShards;
        }

        public int numShards(String idx) {
            ModelIndex index = this.indices.get(idx);
            return index == null ? 0 : index.numShards();
        }

        public double writeLoad() {
            return this.writeLoad;
        }

        public double diskUsageInBytes() {
            return this.diskUsageInBytes;
        }

        public int highestPrimary(String index) {
            ModelIndex idx = this.indices.get(index);
            if (idx != null) {
                return idx.highestPrimary();
            }
            return -1;
        }

        public void addShard(ShardRouting shard) {
            this.indices.computeIfAbsent(shard.getIndexName(), t -> new ModelIndex()).addShard(shard);
            IndexMetadata indexMetadata = this.metadata.index(shard.index());
            this.writeLoad += this.writeLoadForecaster.getForecastedWriteLoad(indexMetadata).orElse(0.0);
            this.diskUsageInBytes += (double)Balancer.getShardDiskUsageInBytes(shard, indexMetadata, this.clusterInfo);
            ++this.numShards;
        }

        public void removeShard(ShardRouting shard) {
            ModelIndex index = this.indices.get(shard.getIndexName());
            if (index != null) {
                index.removeShard(shard);
                if (index.numShards() == 0) {
                    this.indices.remove(shard.getIndexName());
                }
            }
            IndexMetadata indexMetadata = this.metadata.index(shard.index());
            this.writeLoad -= this.writeLoadForecaster.getForecastedWriteLoad(indexMetadata).orElse(0.0);
            this.diskUsageInBytes -= (double)Balancer.getShardDiskUsageInBytes(shard, indexMetadata, this.clusterInfo);
            --this.numShards;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("Node(").append(this.routingNode.nodeId()).append(")");
            return sb.toString();
        }

        @Override
        public Iterator<ModelIndex> iterator() {
            return this.indices.values().iterator();
        }

        public boolean containsShard(ShardRouting shard) {
            ModelIndex index = this.getIndex(shard.getIndexName());
            return index != null && index.containsShard(shard);
        }
    }

    static final class NodeSorter
    extends IntroSorter {
        final ModelNode[] modelNodes;
        final float[] weights;
        private final WeightFunction function;
        private String index;
        private final Balancer balancer;
        private float pivotWeight;

        NodeSorter(ModelNode[] modelNodes, WeightFunction function, Balancer balancer) {
            this.function = function;
            this.balancer = balancer;
            this.modelNodes = modelNodes;
            this.weights = new float[modelNodes.length];
        }

        public void reset(String index, int from, int to) {
            this.index = index;
            for (int i = from; i < to; ++i) {
                this.weights[i] = this.weight(this.modelNodes[i]);
            }
            this.sort(from, to);
        }

        public void reset(String index) {
            this.reset(index, 0, this.modelNodes.length);
        }

        public float weight(ModelNode node) {
            return this.function.calculateNodeWeightWithIndex(this.balancer, node, this.index);
        }

        public float minWeightDelta() {
            return this.function.minWeightDelta(this.balancer.getShardWriteLoad(this.index), this.balancer.maxShardSizeBytes(this.index));
        }

        @Override
        protected void swap(int i, int j) {
            ModelNode tmpNode = this.modelNodes[i];
            this.modelNodes[i] = this.modelNodes[j];
            this.modelNodes[j] = tmpNode;
            float tmpWeight = this.weights[i];
            this.weights[i] = this.weights[j];
            this.weights[j] = tmpWeight;
        }

        @Override
        protected int compare(int i, int j) {
            return Float.compare(this.weights[i], this.weights[j]);
        }

        @Override
        protected void setPivot(int i) {
            this.pivotWeight = this.weights[i];
        }

        @Override
        protected int comparePivot(int j) {
            return Float.compare(this.pivotWeight, this.weights[j]);
        }

        public float delta() {
            return this.weights[this.weights.length - 1] - this.weights[0];
        }
    }

    static final class ModelIndex
    implements Iterable<ShardRouting> {
        private final Set<ShardRouting> shards = Sets.newHashSetWithExpectedSize(4);
        private int highestPrimary = -1;

        ModelIndex() {
        }

        public int highestPrimary() {
            if (this.highestPrimary == -1) {
                int maxId = -1;
                for (ShardRouting shard : this.shards) {
                    if (!shard.primary()) continue;
                    maxId = Math.max(maxId, shard.id());
                }
                this.highestPrimary = maxId;
                return this.highestPrimary;
            }
            return this.highestPrimary;
        }

        public int numShards() {
            return this.shards.size();
        }

        @Override
        public Iterator<ShardRouting> iterator() {
            return this.shards.iterator();
        }

        public void removeShard(ShardRouting shard) {
            this.highestPrimary = -1;
            assert (this.shards.contains(shard)) : "Shard not allocated on current node: " + String.valueOf(shard);
            this.shards.remove(shard);
        }

        public void addShard(ShardRouting shard) {
            this.highestPrimary = -1;
            assert (!this.shards.contains(shard)) : "Shard already allocated on current node: " + String.valueOf(shard);
            this.shards.add(shard);
        }

        public boolean containsShard(ShardRouting shard) {
            return this.shards.contains(shard);
        }
    }
}

