/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.metadata;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateTaskExecutor;
import org.elasticsearch.cluster.ClusterStateTaskListener;
import org.elasticsearch.cluster.metadata.AliasMetadata;
import org.elasticsearch.cluster.metadata.DataStream;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.cluster.service.MasterServiceTaskQueue;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.index.Index;
import org.elasticsearch.indices.SystemIndices;

public class SystemIndexMetadataUpgradeService
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(SystemIndexMetadataUpgradeService.class);
    private final SystemIndices systemIndices;
    private final ClusterService clusterService;
    private final MasterServiceTaskQueue<SystemIndexMetadataUpgradeTask> taskQueue;

    public SystemIndexMetadataUpgradeService(SystemIndices systemIndices, ClusterService clusterService) {
        this.systemIndices = systemIndices;
        this.clusterService = clusterService;
        this.taskQueue = clusterService.createTaskQueue("system-indices-metadata-upgrade", Priority.NORMAL, new SystemIndexMetadataUpgradeExecutor());
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        Metadata currentMetadata = event.state().metadata();
        Metadata previousMetadata = event.previousState().metadata();
        if (event.localNodeMaster() && (!event.previousState().nodes().isLocalNodeElectedMaster() || currentMetadata.indices() != previousMetadata.indices() || currentMetadata.dataStreams() != previousMetadata.dataStreams())) {
            final Map<String, IndexMetadata> indexMetadataMap = currentMetadata.indices();
            final Map<String, IndexMetadata> previousIndices = previousMetadata.indices();
            final Map<String, DataStream> dataStreams = currentMetadata.dataStreams();
            final Map<String, DataStream> previousDataStreams = previousMetadata.dataStreams();
            this.clusterService.threadPool().executor("management").execute(new AbstractRunnable(){

                @Override
                protected void doRun() {
                    ArrayList<DataStream> changedDataStreams = new ArrayList<DataStream>();
                    HashSet dataStreamIndices = new HashSet();
                    for (Map.Entry cursor : dataStreams.entrySet()) {
                        DataStream dataStream = (DataStream)cursor.getValue();
                        if (dataStream != previousDataStreams.get(cursor.getKey()) && SystemIndexMetadataUpgradeService.this.requiresUpdate(dataStream)) {
                            changedDataStreams.add(dataStream);
                        }
                        SystemIndexMetadataUpgradeService.getIndicesBackingDataStream(dataStream).forEach(dataStreamIndices::add);
                    }
                    ArrayList<Index> changedIndices = new ArrayList<Index>();
                    for (Map.Entry cursor : indexMetadataMap.entrySet()) {
                        IndexMetadata indexMetadata = (IndexMetadata)cursor.getValue();
                        Index index = indexMetadata.getIndex();
                        if (cursor.getValue() == previousIndices.get(cursor.getKey()) || dataStreamIndices.contains(index) || !SystemIndexMetadataUpgradeService.this.requiresUpdate(indexMetadata)) continue;
                        changedIndices.add(index);
                    }
                    if (!changedIndices.isEmpty() || !changedDataStreams.isEmpty()) {
                        SystemIndexMetadataUpgradeService.this.submitUpdateTask(changedIndices, changedDataStreams);
                    }
                }

                @Override
                public void onFailure(Exception e) {
                    logger.error("unexpected exception on checking for metadata upgrades", (Throwable)e);
                    assert (false) : e;
                }
            });
        }
    }

    void submitUpdateTask(Collection<Index> changedIndices, Collection<DataStream> changedDataStreams) {
        SystemIndexMetadataUpgradeTask task = new SystemIndexMetadataUpgradeTask(changedIndices, changedDataStreams);
        this.taskQueue.submitTask("system-index-metadata-upgrade-service", task, null);
    }

    boolean requiresUpdate(IndexMetadata indexMetadata) {
        boolean shouldBeSystem = this.shouldBeSystem(indexMetadata);
        if (shouldBeSystem != indexMetadata.isSystem()) {
            return true;
        }
        if (shouldBeSystem) {
            return SystemIndexMetadataUpgradeService.isVisible(indexMetadata) || SystemIndexMetadataUpgradeService.hasVisibleAlias(indexMetadata);
        }
        return false;
    }

    boolean requiresUpdate(DataStream dataStream) {
        boolean shouldBeSystem = this.shouldBeSystem(dataStream);
        if (shouldBeSystem != dataStream.isSystem()) {
            return true;
        }
        if (shouldBeSystem) {
            return !dataStream.isHidden();
        }
        return false;
    }

    private boolean shouldBeSystem(DataStream dataStream) {
        return this.systemIndices.isSystemDataStream(dataStream.getName());
    }

    private static Stream<Index> getIndicesBackingDataStream(DataStream dataStream) {
        return Stream.concat(dataStream.getIndices().stream(), dataStream.getFailureIndices().stream());
    }

    static boolean isVisible(IndexMetadata indexMetadata) {
        return indexMetadata.getSettings().getAsBoolean("index.hidden", false) == false;
    }

    boolean shouldBeSystem(IndexMetadata indexMetadata) {
        return this.systemIndices.isSystemIndex(indexMetadata.getIndex());
    }

    static boolean hasVisibleAlias(IndexMetadata indexMetadata) {
        return indexMetadata.getAliases().values().stream().anyMatch(a -> Boolean.FALSE.equals(a.isHidden()));
    }

    private class SystemIndexMetadataUpgradeExecutor
    implements ClusterStateTaskExecutor<SystemIndexMetadataUpgradeTask> {
        private SystemIndexMetadataUpgradeExecutor() {
        }

        @Override
        public ClusterState execute(ClusterStateTaskExecutor.BatchExecutionContext<SystemIndexMetadataUpgradeTask> batchExecutionContext) {
            ClusterState initialState = batchExecutionContext.initialState();
            List<ClusterStateTaskExecutor.TaskContext<SystemIndexMetadataUpgradeTask>> taskContexts = batchExecutionContext.taskContexts();
            List<Index> indices = taskContexts.stream().map(ClusterStateTaskExecutor.TaskContext::getTask).map(SystemIndexMetadataUpgradeTask::changedIndices).flatMap(Collection::stream).toList();
            List<IndexMetadata> updatedMetadata = this.updateIndices(initialState, indices);
            List<DataStream> dataStreams = taskContexts.stream().map(ClusterStateTaskExecutor.TaskContext::getTask).map(SystemIndexMetadataUpgradeTask::changedDataStreams).flatMap(Collection::stream).toList();
            List<DataStream> updatedDataStreams = this.updateDataStreams(dataStreams);
            List<IndexMetadata> updatedBackingIndices = this.updateIndicesBackingDataStreams(initialState, updatedDataStreams);
            for (ClusterStateTaskExecutor.TaskContext<SystemIndexMetadataUpgradeTask> taskContext : taskContexts) {
                taskContext.success(() -> {});
            }
            if (!updatedMetadata.isEmpty() || !updatedDataStreams.isEmpty()) {
                Metadata.Builder builder = Metadata.builder(initialState.metadata());
                updatedMetadata.forEach(idxMeta -> builder.put((IndexMetadata)idxMeta, true));
                updatedDataStreams.forEach(builder::put);
                updatedBackingIndices.forEach(idxMeta -> builder.put((IndexMetadata)idxMeta, true));
                return ClusterState.builder(initialState).metadata(builder).build();
            }
            return initialState;
        }

        private List<IndexMetadata> updateIndices(ClusterState currentState, List<Index> indices) {
            if (indices.isEmpty()) {
                return Collections.emptyList();
            }
            Metadata metadata = currentState.metadata();
            ArrayList<IndexMetadata> updatedMetadata = new ArrayList<IndexMetadata>();
            for (Index index : indices) {
                boolean shouldBeSystem;
                IndexMetadata updatedIndexMetadata;
                IndexMetadata indexMetadata = metadata.index(index);
                if (indexMetadata == null || (updatedIndexMetadata = this.updateIndexIfNecessary(indexMetadata, shouldBeSystem = SystemIndexMetadataUpgradeService.this.shouldBeSystem(indexMetadata))) == null) continue;
                updatedMetadata.add(updatedIndexMetadata);
            }
            return updatedMetadata;
        }

        private IndexMetadata updateIndexIfNecessary(IndexMetadata indexMetadata, boolean shouldBeSystem) {
            IndexMetadata.Builder builder = IndexMetadata.builder(indexMetadata);
            boolean updated = false;
            if (shouldBeSystem != indexMetadata.isSystem()) {
                builder.system(!indexMetadata.isSystem());
                updated = true;
            }
            if (shouldBeSystem && SystemIndexMetadataUpgradeService.isVisible(indexMetadata)) {
                builder.settings(Settings.builder().put(indexMetadata.getSettings()).put("index.hidden", true));
                builder.settingsVersion(builder.settingsVersion() + 1L);
                updated = true;
            }
            if (shouldBeSystem && SystemIndexMetadataUpgradeService.hasVisibleAlias(indexMetadata)) {
                for (AliasMetadata aliasMetadata : indexMetadata.getAliases().values()) {
                    if (!Boolean.FALSE.equals(aliasMetadata.isHidden())) continue;
                    builder.removeAlias(aliasMetadata.alias());
                    builder.putAlias(AliasMetadata.builder(aliasMetadata.alias()).filter(aliasMetadata.filter()).indexRouting(aliasMetadata.indexRouting()).isHidden(true).searchRouting(aliasMetadata.searchRouting()).writeIndex(aliasMetadata.writeIndex()));
                    updated = true;
                }
            }
            return updated ? builder.build() : null;
        }

        private List<DataStream> updateDataStreams(List<DataStream> dataStreams) {
            if (dataStreams.isEmpty()) {
                return Collections.emptyList();
            }
            ArrayList<DataStream> updatedDataStreams = new ArrayList<DataStream>();
            for (DataStream dataStream : dataStreams) {
                boolean shouldBeSystem = SystemIndexMetadataUpgradeService.this.shouldBeSystem(dataStream);
                if (dataStream.isSystem() == shouldBeSystem) continue;
                DataStream.Builder dataStreamBuilder = dataStream.copy().setSystem(shouldBeSystem);
                if (shouldBeSystem) {
                    dataStreamBuilder.setHidden(true);
                }
                updatedDataStreams.add(dataStreamBuilder.build());
            }
            return updatedDataStreams;
        }

        private List<IndexMetadata> updateIndicesBackingDataStreams(ClusterState currentState, List<DataStream> updatedDataStreams) {
            if (updatedDataStreams.isEmpty()) {
                return Collections.emptyList();
            }
            Metadata metadata = currentState.metadata();
            ArrayList<IndexMetadata> updatedMetadata = new ArrayList<IndexMetadata>();
            for (DataStream updatedDataStream : updatedDataStreams) {
                boolean shouldBeSystem = updatedDataStream.isSystem();
                List<IndexMetadata> updatedIndicesMetadata = this.getIndicesBackingDataStreamMetadata(metadata, updatedDataStream).map(idx -> this.updateIndexIfNecessary((IndexMetadata)idx, shouldBeSystem)).filter(Objects::nonNull).toList();
                updatedMetadata.addAll(updatedIndicesMetadata);
            }
            return updatedMetadata;
        }

        private Stream<IndexMetadata> getIndicesBackingDataStreamMetadata(Metadata metadata, DataStream dataStream) {
            return SystemIndexMetadataUpgradeService.getIndicesBackingDataStream(dataStream).map(metadata::index);
        }
    }

    private record SystemIndexMetadataUpgradeTask(Collection<Index> changedIndices, Collection<DataStream> changedDataStreams) implements ClusterStateTaskListener
    {
        @Override
        public void onFailure(Exception e) {
            logger.error("System index metadata upgrade failed", (Throwable)e);
        }

        @Override
        public String toString() {
            return "SystemIndexMetadataUpgradeTask[changedIndices=" + this.changedIndices.stream().map(Index::getName).collect(Collectors.joining(",")) + ";changedDataStreams=" + this.changedDataStreams.stream().map(DataStream::getName).collect(Collectors.joining(",")) + "]";
        }
    }
}

