/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.http;

import java.net.InetSocketAddress;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Semaphore;
import java.util.function.LongPredicate;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.common.network.NetworkAddress;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.http.HttpChannel;
import org.elasticsearch.http.HttpRequest;
import org.elasticsearch.http.HttpStats;
import org.elasticsearch.http.HttpTransportSettings;
import org.elasticsearch.threadpool.ThreadPool;

public class HttpClientStatsTracker {
    private static final Logger logger = LogManager.getLogger(HttpClientStatsTracker.class);
    private final ThreadPool threadPool;
    private final Map<HttpChannel, ClientStatsBuilder> httpChannelStats = new ConcurrentHashMap<HttpChannel, ClientStatsBuilder>();
    private final Semaphore closedChannelPermits;
    private final ConcurrentLinkedQueue<HttpStats.ClientStats> closedChannelStats = new ConcurrentLinkedQueue();
    private final long maxClosedChannelAgeMillis;
    private volatile boolean clientStatsEnabled;

    HttpClientStatsTracker(Settings settings, ClusterSettings clusterSettings, ThreadPool threadPool) {
        this.threadPool = threadPool;
        this.closedChannelPermits = new Semaphore(HttpTransportSettings.SETTING_HTTP_CLIENT_STATS_MAX_CLOSED_CHANNEL_COUNT.get(settings));
        this.maxClosedChannelAgeMillis = HttpTransportSettings.SETTING_HTTP_CLIENT_STATS_MAX_CLOSED_CHANNEL_AGE.get(settings).millis();
        this.clientStatsEnabled = HttpTransportSettings.SETTING_HTTP_CLIENT_STATS_ENABLED.get(settings);
        clusterSettings.addSettingsUpdateConsumer(HttpTransportSettings.SETTING_HTTP_CLIENT_STATS_ENABLED, this::enableClientStats);
    }

    private void enableClientStats(boolean enabled) {
        this.clientStatsEnabled = enabled;
        if (!enabled) {
            this.httpChannelStats.clear();
            this.pruneStaleClosedChannelStats(l -> false);
        }
    }

    void addClientStats(HttpChannel httpChannel) {
        if (!this.clientStatsEnabled) {
            return;
        }
        if (httpChannel == null) {
            return;
        }
        this.httpChannelStats.putIfAbsent(httpChannel, new ClientStatsBuilder(System.identityHashCode(httpChannel), HttpClientStatsTracker.formatAddress(httpChannel.getRemoteAddress()), this.threadPool.absoluteTimeInMillis()));
        httpChannel.addCloseListener(ActionListener.running(() -> {
            try {
                ClientStatsBuilder disconnectedClientStats = this.httpChannelStats.remove(httpChannel);
                if (disconnectedClientStats != null) {
                    this.addClosedChannelStats(disconnectedClientStats.build(this.threadPool.absoluteTimeInMillis()));
                }
            }
            catch (Exception e) {
                assert (false) : e;
                logger.warn("error removing HTTP channel listener", (Throwable)e);
            }
        }));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addClosedChannelStats(HttpStats.ClientStats clientStats) {
        if (!this.clientStatsEnabled) {
            return;
        }
        if (!this.closedChannelPermits.tryAcquire()) {
            ConcurrentLinkedQueue<HttpStats.ClientStats> concurrentLinkedQueue = this.closedChannelStats;
            synchronized (concurrentLinkedQueue) {
                HttpStats.ClientStats oldest = this.closedChannelStats.poll();
                if (oldest == null && !this.closedChannelPermits.tryAcquire()) {
                    return;
                }
            }
        }
        this.closedChannelStats.add(clientStats);
    }

    void updateClientStats(HttpRequest httpRequest, HttpChannel httpChannel) {
        ClientStatsBuilder clientStats;
        if (this.clientStatsEnabled && httpChannel != null && (clientStats = this.httpChannelStats.get(httpChannel)) != null) {
            clientStats.update(httpRequest, httpChannel, this.threadPool.absoluteTimeInMillis());
        }
    }

    List<HttpStats.ClientStats> getClientStats() {
        if (this.clientStatsEnabled) {
            long currentTimeMillis = this.threadPool.absoluteTimeInMillis();
            LongPredicate keepTimePredicate = closeTimeMillis -> currentTimeMillis - closeTimeMillis <= this.maxClosedChannelAgeMillis;
            this.pruneStaleClosedChannelStats(keepTimePredicate);
            return Stream.concat(this.closedChannelStats.stream().filter(c -> keepTimePredicate.test(c.closedTimeMillis())), this.httpChannelStats.values().stream().map(c -> c.build(-1L))).toList();
        }
        this.httpChannelStats.clear();
        this.pruneStaleClosedChannelStats(l -> false);
        return Collections.emptyList();
    }

    private void pruneStaleClosedChannelStats(LongPredicate keepTimePredicate) {
        ConcurrentLinkedQueue<HttpStats.ClientStats> concurrentLinkedQueue = this.closedChannelStats;
        synchronized (concurrentLinkedQueue) {
            while (true) {
                HttpStats.ClientStats nextStats;
                if ((nextStats = this.closedChannelStats.peek()) == null) {
                    return;
                }
                if (keepTimePredicate.test(nextStats.closedTimeMillis())) {
                    return;
                }
                HttpStats.ClientStats removed = this.closedChannelStats.poll();
                assert (removed == nextStats);
                this.closedChannelPermits.release();
            }
        }
    }

    @Nullable
    private static String formatAddress(@Nullable InetSocketAddress localAddress) {
        return localAddress == null ? null : NetworkAddress.format(localAddress);
    }

    private static class ClientStatsBuilder {
        final int id;
        final long openedTimeMillis;
        String agent;
        String localAddress;
        String remoteAddress;
        String lastUri;
        String forwardedFor;
        String opaqueId;
        long lastRequestTimeMillis = -1L;
        long requestCount;
        long requestSizeBytes;

        ClientStatsBuilder(int id, @Nullable String remoteAddress, long openedTimeMillis) {
            this.id = id;
            this.remoteAddress = remoteAddress;
            this.openedTimeMillis = openedTimeMillis;
        }

        synchronized void update(HttpRequest httpRequest, HttpChannel httpChannel, long currentTimeMillis) {
            if (this.agent == null) {
                String elasticProductOrigin = ClientStatsBuilder.getFirstValueForHeader(httpRequest, "x-elastic-product-origin");
                this.agent = elasticProductOrigin != null ? elasticProductOrigin : ClientStatsBuilder.getFirstValueForHeader(httpRequest, "User-Agent");
            }
            if (this.localAddress == null) {
                this.localAddress = HttpClientStatsTracker.formatAddress(httpChannel.getLocalAddress());
            }
            if (this.remoteAddress == null) {
                this.remoteAddress = HttpClientStatsTracker.formatAddress(httpChannel.getRemoteAddress());
            }
            if (this.forwardedFor == null) {
                this.forwardedFor = ClientStatsBuilder.getFirstValueForHeader(httpRequest, "x-forwarded-for");
            }
            if (this.opaqueId == null) {
                this.opaqueId = ClientStatsBuilder.getFirstValueForHeader(httpRequest, "x-opaque-id");
            }
            this.lastRequestTimeMillis = currentTimeMillis;
            this.lastUri = httpRequest.uri();
            ++this.requestCount;
            this.requestSizeBytes += (long)httpRequest.content().length();
        }

        private static String getFirstValueForHeader(HttpRequest request, String header) {
            for (Map.Entry<String, List<String>> entry : request.getHeaders().entrySet()) {
                if (!entry.getKey().equalsIgnoreCase(header) || entry.getValue().size() <= 0) continue;
                return entry.getValue().get(0);
            }
            return null;
        }

        synchronized HttpStats.ClientStats build(long closedTimeMillis) {
            return new HttpStats.ClientStats(this.id, this.agent, this.localAddress, this.remoteAddress, this.lastUri, this.forwardedFor, this.opaqueId, this.openedTimeMillis, closedTimeMillis, this.lastRequestTimeMillis, this.requestCount, this.requestSizeBytes);
        }
    }
}

