/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.bootstrap;

import java.io.BufferedReader;
import java.io.IOException;
import java.lang.invoke.CallSite;
import java.nio.ByteOrder;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.AllPermission;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.util.Constants;
import org.elasticsearch.bootstrap.BootstrapCheck;
import org.elasticsearch.bootstrap.BootstrapContext;
import org.elasticsearch.bootstrap.BootstrapInfo;
import org.elasticsearch.bootstrap.BootstrapSettings;
import org.elasticsearch.bootstrap.Natives;
import org.elasticsearch.cluster.coordination.ClusterBootstrapService;
import org.elasticsearch.common.ReferenceDocs;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.transport.BoundTransportAddress;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.core.PathUtils;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.discovery.DiscoveryModule;
import org.elasticsearch.discovery.SettingsBasedSeedHostsProvider;
import org.elasticsearch.index.IndexModule;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.monitor.process.ProcessProbe;
import org.elasticsearch.nativeaccess.NativeAccess;
import org.elasticsearch.node.NodeValidationException;

final class BootstrapChecks {
    static final String ES_ENFORCE_BOOTSTRAP_CHECKS = "es.enforce.bootstrap.checks";

    private BootstrapChecks() {
    }

    static void check(BootstrapContext context, BoundTransportAddress boundTransportAddress, List<BootstrapCheck> additionalChecks) throws NodeValidationException {
        List<BootstrapCheck> builtInChecks = BootstrapChecks.checks();
        ArrayList<BootstrapCheck> combinedChecks = new ArrayList<BootstrapCheck>(builtInChecks);
        combinedChecks.addAll(additionalChecks);
        BootstrapChecks.check(context, BootstrapChecks.enforceLimits(boundTransportAddress, DiscoveryModule.DISCOVERY_TYPE_SETTING.get(context.settings())), Collections.unmodifiableList(combinedChecks));
    }

    static void check(BootstrapContext context, boolean enforceLimits, List<BootstrapCheck> checks) throws NodeValidationException {
        BootstrapChecks.check(context, enforceLimits, checks, LogManager.getLogger(BootstrapChecks.class));
    }

    static void check(BootstrapContext context, boolean enforceLimits, List<BootstrapCheck> checks, Logger logger) throws NodeValidationException {
        boolean enforceBootstrapChecks;
        ArrayList<CallSite> errors = new ArrayList<CallSite>();
        ArrayList<CallSite> ignoredErrors = new ArrayList<CallSite>();
        String esEnforceBootstrapChecks = System.getProperty(ES_ENFORCE_BOOTSTRAP_CHECKS);
        if (esEnforceBootstrapChecks == null) {
            enforceBootstrapChecks = false;
        } else if (Boolean.TRUE.toString().equals(esEnforceBootstrapChecks)) {
            enforceBootstrapChecks = true;
        } else {
            String message = String.format(Locale.ROOT, "[%s] must be [true] but was [%s]", ES_ENFORCE_BOOTSTRAP_CHECKS, esEnforceBootstrapChecks);
            throw new IllegalArgumentException(message);
        }
        if (enforceLimits) {
            logger.info("bound or publishing to a non-loopback address, enforcing bootstrap checks");
        } else if (enforceBootstrapChecks) {
            logger.info("explicitly enforcing bootstrap checks");
        }
        for (BootstrapCheck check : checks) {
            BootstrapCheck.BootstrapCheckResult result = check.check(context);
            if (!result.isFailure()) continue;
            String message = result.getMessage() + "; for more information see [" + check.referenceDocs() + "]";
            if (!(enforceLimits || enforceBootstrapChecks || check.alwaysEnforce())) {
                ignoredErrors.add((CallSite)((Object)message));
                continue;
            }
            errors.add((CallSite)((Object)message));
        }
        if (!ignoredErrors.isEmpty()) {
            ignoredErrors.forEach(error -> BootstrapChecks.log(logger, error));
        }
        if (!errors.isEmpty()) {
            ArrayList<CallSite> messages = new ArrayList<CallSite>(1 + errors.size());
            messages.add((CallSite)((Object)("[" + errors.size() + "] bootstrap checks failed. You must address the points described in the following [" + errors.size() + "] lines before starting Elasticsearch. For more information see [" + ReferenceDocs.BOOTSTRAP_CHECKS + "]")));
            for (int i = 0; i < errors.size(); ++i) {
                messages.add((CallSite)((Object)("bootstrap check failure [" + (i + 1) + "] of [" + errors.size() + "]: " + (String)errors.get(i))));
            }
            NodeValidationException ne = new NodeValidationException(String.join((CharSequence)"\n", messages));
            errors.stream().map(IllegalStateException::new).forEach(ne::addSuppressed);
            throw ne;
        }
    }

    static void log(Logger logger, String error) {
        logger.warn(error);
    }

    static boolean enforceLimits(BoundTransportAddress boundTransportAddress, String discoveryType) {
        Predicate<TransportAddress> isLoopbackAddress = t -> t.address().getAddress().isLoopbackAddress();
        boolean bound = !(Arrays.stream(boundTransportAddress.boundAddresses()).allMatch(isLoopbackAddress) && isLoopbackAddress.test(boundTransportAddress.publishAddress()));
        return bound && !"single-node".equals(discoveryType);
    }

    static List<BootstrapCheck> checks() {
        ArrayList<BootstrapCheck> checks = new ArrayList<BootstrapCheck>();
        checks.add(new HeapSizeCheck());
        FileDescriptorCheck fileDescriptorCheck = Constants.MAC_OS_X ? new OsXFileDescriptorCheck() : new FileDescriptorCheck();
        checks.add(fileDescriptorCheck);
        checks.add(new MlockallCheck());
        if (Constants.LINUX) {
            checks.add(new MaxNumberOfThreadsCheck());
        }
        if (Constants.LINUX || Constants.MAC_OS_X) {
            checks.add(new MaxSizeVirtualMemoryCheck());
        }
        if (Constants.LINUX || Constants.MAC_OS_X) {
            checks.add(new MaxFileSizeCheck());
        }
        if (Constants.LINUX) {
            checks.add(new MaxMapCountCheck());
        }
        checks.add(new ClientJvmCheck());
        checks.add(new UseSerialGCCheck());
        checks.add(new SystemCallFilterCheck());
        checks.add(new OnErrorCheck());
        checks.add(new OnOutOfMemoryErrorCheck());
        checks.add(new EarlyAccessCheck());
        checks.add(new AllPermissionCheck());
        checks.add(new DiscoveryConfiguredCheck());
        checks.add(new ByteOrderCheck());
        return Collections.unmodifiableList(checks);
    }

    static class HeapSizeCheck
    implements BootstrapCheck {
        HeapSizeCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            long initialHeapSize = this.getInitialHeapSize();
            long maxHeapSize = this.getMaxHeapSize();
            if (initialHeapSize != 0L && maxHeapSize != 0L && initialHeapSize != maxHeapSize) {
                String message = this.isMemoryLocked() ? String.format(Locale.ROOT, "initial heap size [%d] not equal to maximum heap size [%d]; this can cause resize pauses and prevents memory locking from locking the entire heap", this.getInitialHeapSize(), this.getMaxHeapSize()) : String.format(Locale.ROOT, "initial heap size [%d] not equal to maximum heap size [%d]; this can cause resize pauses", this.getInitialHeapSize(), this.getMaxHeapSize());
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_HEAP_SIZE;
        }

        long getInitialHeapSize() {
            return JvmInfo.jvmInfo().getConfiguredInitialHeapSize();
        }

        long getMaxHeapSize() {
            return JvmInfo.jvmInfo().getConfiguredMaxHeapSize();
        }

        boolean isMemoryLocked() {
            return NativeAccess.instance().isMemoryLocked();
        }
    }

    static class OsXFileDescriptorCheck
    extends FileDescriptorCheck {
        OsXFileDescriptorCheck() {
            super(10240);
        }
    }

    static class FileDescriptorCheck
    implements BootstrapCheck {
        private final int limit;

        FileDescriptorCheck() {
            this(65535);
        }

        protected FileDescriptorCheck(int limit) {
            if (limit <= 0) {
                throw new IllegalArgumentException("limit must be positive but was [" + limit + "]");
            }
            this.limit = limit;
        }

        @Override
        public final BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            long maxFileDescriptorCount = this.getMaxFileDescriptorCount();
            if (maxFileDescriptorCount != -1L && maxFileDescriptorCount < (long)this.limit) {
                String message = String.format(Locale.ROOT, "max file descriptors [%d] for elasticsearch process is too low, increase to at least [%d]", this.getMaxFileDescriptorCount(), this.limit);
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_FILE_DESCRIPTOR;
        }

        long getMaxFileDescriptorCount() {
            return ProcessProbe.getMaxFileDescriptorCount();
        }
    }

    static class MlockallCheck
    implements BootstrapCheck {
        MlockallCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (BootstrapSettings.MEMORY_LOCK_SETTING.get(context.settings()).booleanValue() && !this.isMemoryLocked()) {
                return BootstrapCheck.BootstrapCheckResult.failure("memory locking requested for elasticsearch process but memory is not locked");
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        boolean isMemoryLocked() {
            return NativeAccess.instance().isMemoryLocked();
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_MEMORY_LOCK;
        }
    }

    static class MaxNumberOfThreadsCheck
    implements BootstrapCheck {
        private static final long MAX_NUMBER_OF_THREADS_THRESHOLD = 4096L;

        MaxNumberOfThreadsCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.getMaxNumberOfThreads() != -1L && this.getMaxNumberOfThreads() < 4096L) {
                String message = String.format(Locale.ROOT, "max number of threads [%d] for user [%s] is too low, increase to at least [%d]", this.getMaxNumberOfThreads(), BootstrapInfo.getSystemProperties().get("user.name"), 4096L);
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        long getMaxNumberOfThreads() {
            return NativeAccess.instance().getProcessLimits().maxThreads();
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_MAX_NUMBER_THREADS;
        }
    }

    static class MaxSizeVirtualMemoryCheck
    implements BootstrapCheck {
        MaxSizeVirtualMemoryCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.getMaxSizeVirtualMemory() != Long.MIN_VALUE && this.getMaxSizeVirtualMemory() != Long.MAX_VALUE) {
                String message = String.format(Locale.ROOT, "max size virtual memory [%d] for user [%s] is too low, increase to [unlimited]", this.getMaxSizeVirtualMemory(), BootstrapInfo.getSystemProperties().get("user.name"));
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        long getMaxSizeVirtualMemory() {
            return NativeAccess.instance().getProcessLimits().maxVirtualMemorySize();
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_MAX_SIZE_VIRTUAL_MEMORY;
        }
    }

    static class MaxFileSizeCheck
    implements BootstrapCheck {
        MaxFileSizeCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            long maxFileSize = this.getMaxFileSize();
            if (maxFileSize != Long.MIN_VALUE && maxFileSize != Long.MAX_VALUE) {
                String message = String.format(Locale.ROOT, "max file size [%d] for user [%s] is too low, increase to [unlimited]", this.getMaxFileSize(), BootstrapInfo.getSystemProperties().get("user.name"));
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        long getMaxFileSize() {
            return NativeAccess.instance().getProcessLimits().maxVirtualMemorySize();
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_MAX_FILE_SIZE;
        }
    }

    static class MaxMapCountCheck
    implements BootstrapCheck {
        static final long LIMIT = 262144L;

        MaxMapCountCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (IndexModule.NODE_STORE_ALLOW_MMAP.get(context.settings()).booleanValue()) {
                if (this.getMaxMapCount() != -1L && this.getMaxMapCount() < 262144L) {
                    String message = String.format(Locale.ROOT, "max virtual memory areas vm.max_map_count [%d] is too low, increase to at least [%d]", this.getMaxMapCount(), 262144L);
                    return BootstrapCheck.BootstrapCheckResult.failure(message);
                }
                return BootstrapCheck.BootstrapCheckResult.success();
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        long getMaxMapCount() {
            return this.getMaxMapCount(LogManager.getLogger(BootstrapChecks.class));
        }

        /*
         * Enabled aggressive exception aggregation
         */
        long getMaxMapCount(Logger logger) {
            block11: {
                Path path = MaxMapCountCheck.getProcSysVmMaxMapCountPath();
                try (BufferedReader bufferedReader = this.getBufferedReader(path);){
                    String rawProcSysVmMaxMapCount = MaxMapCountCheck.readProcSysVmMaxMapCount(bufferedReader);
                    if (rawProcSysVmMaxMapCount == null) break block11;
                    try {
                        long l = MaxMapCountCheck.parseProcSysVmMaxMapCount(rawProcSysVmMaxMapCount);
                        return l;
                    }
                    catch (NumberFormatException e) {
                        logger.warn(() -> "unable to parse vm.max_map_count [" + rawProcSysVmMaxMapCount + "]", (Throwable)e);
                    }
                }
                catch (IOException e) {
                    logger.warn(() -> "I/O exception while trying to read [" + path + "]", (Throwable)e);
                }
            }
            return -1L;
        }

        @SuppressForbidden(reason="access /proc/sys/vm/max_map_count")
        private static Path getProcSysVmMaxMapCountPath() {
            return PathUtils.get("/proc/sys/vm/max_map_count", new String[0]);
        }

        BufferedReader getBufferedReader(Path path) throws IOException {
            return Files.newBufferedReader(path);
        }

        static String readProcSysVmMaxMapCount(BufferedReader bufferedReader) throws IOException {
            return bufferedReader.readLine();
        }

        static long parseProcSysVmMaxMapCount(String procSysVmMaxMapCount) throws NumberFormatException {
            return Long.parseLong(procSysVmMaxMapCount);
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_MAXIMUM_MAP_COUNT;
        }
    }

    static class ClientJvmCheck
    implements BootstrapCheck {
        ClientJvmCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.getVmName().toLowerCase(Locale.ROOT).contains("client")) {
                String message = String.format(Locale.ROOT, "JVM is using the client VM [%s] but should be using a server VM for the best performance", this.getVmName());
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        String getVmName() {
            return JvmInfo.jvmInfo().getVmName();
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_CLIENT_JVM;
        }
    }

    static class UseSerialGCCheck
    implements BootstrapCheck {
        UseSerialGCCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.getUseSerialGC().equals("true")) {
                String message = String.format(Locale.ROOT, "JVM is using the serial collector but should not be for the best performance; either it's the default for the VM [%s] or -XX:+UseSerialGC was explicitly specified", JvmInfo.jvmInfo().getVmName());
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        String getUseSerialGC() {
            return JvmInfo.jvmInfo().useSerialGC();
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_USE_SERIAL_COLLECTOR;
        }
    }

    static class SystemCallFilterCheck
    implements BootstrapCheck {
        SystemCallFilterCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (!this.isSystemCallFilterInstalled()) {
                String message = "system call filters failed to install; check the logs and fix your configuration";
                return BootstrapCheck.BootstrapCheckResult.failure("system call filters failed to install; check the logs and fix your configuration");
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        boolean isSystemCallFilterInstalled() {
            return Natives.isSystemCallFilterInstalled();
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_SYSTEM_CALL_FILTER;
        }
    }

    static class OnErrorCheck
    extends MightForkCheck {
        OnErrorCheck() {
        }

        @Override
        boolean mightFork() {
            String onError = this.onError();
            return onError != null && !onError.isEmpty();
        }

        String onError() {
            return JvmInfo.jvmInfo().onError();
        }

        @Override
        String message(BootstrapContext context) {
            return String.format(Locale.ROOT, "OnError [%s] requires forking but is prevented by system call filters; upgrade to at least Java 8u92 and use ExitOnOutOfMemoryError", this.onError());
        }
    }

    static class OnOutOfMemoryErrorCheck
    extends MightForkCheck {
        OnOutOfMemoryErrorCheck() {
        }

        @Override
        boolean mightFork() {
            String onOutOfMemoryError = this.onOutOfMemoryError();
            return onOutOfMemoryError != null && !onOutOfMemoryError.isEmpty();
        }

        String onOutOfMemoryError() {
            return JvmInfo.jvmInfo().onOutOfMemoryError();
        }

        @Override
        String message(BootstrapContext context) {
            return String.format(Locale.ROOT, "OnOutOfMemoryError [%s] requires forking but is prevented by system call filters; upgrade to at least Java 8u92 and use ExitOnOutOfMemoryError", this.onOutOfMemoryError());
        }
    }

    static class EarlyAccessCheck
    implements BootstrapCheck {
        EarlyAccessCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            String javaVersion = this.javaVersion();
            if ("Oracle Corporation".equals(this.jvmVendor()) && javaVersion.endsWith("-ea")) {
                String message = String.format(Locale.ROOT, "Java version [%s] is an early-access build, only use release builds", javaVersion);
                return BootstrapCheck.BootstrapCheckResult.failure(message);
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        String jvmVendor() {
            return Constants.JVM_VENDOR;
        }

        String javaVersion() {
            return Constants.JAVA_VERSION;
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_EARLY_ACCESS;
        }
    }

    static class AllPermissionCheck
    implements BootstrapCheck {
        AllPermissionCheck() {
        }

        @Override
        public final BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.isAllPermissionGranted()) {
                return BootstrapCheck.BootstrapCheckResult.failure("granting the all permission effectively disables security");
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        boolean isAllPermissionGranted() {
            SecurityManager sm = System.getSecurityManager();
            assert (sm != null);
            try {
                sm.checkPermission(new AllPermission());
            }
            catch (SecurityException e) {
                return false;
            }
            return true;
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_ALL_PERMISSION;
        }
    }

    static class DiscoveryConfiguredCheck
    implements BootstrapCheck {
        DiscoveryConfiguredCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (!"multi-node".equals(DiscoveryModule.DISCOVERY_TYPE_SETTING.get(context.settings()))) {
                return BootstrapCheck.BootstrapCheckResult.success();
            }
            if (ClusterBootstrapService.discoveryIsConfigured(context.settings())) {
                return BootstrapCheck.BootstrapCheckResult.success();
            }
            return BootstrapCheck.BootstrapCheckResult.failure(String.format(Locale.ROOT, "the default discovery settings are unsuitable for production use; at least one of [%s] must be configured", Stream.of(SettingsBasedSeedHostsProvider.DISCOVERY_SEED_HOSTS_SETTING, DiscoveryModule.DISCOVERY_SEED_PROVIDERS_SETTING, ClusterBootstrapService.INITIAL_MASTER_NODES_SETTING).map(Setting::getKey).collect(Collectors.joining(", "))));
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_DISCOVERY_CONFIGURATION;
        }
    }

    static class ByteOrderCheck
    implements BootstrapCheck {
        ByteOrderCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.nativeByteOrder() != ByteOrder.LITTLE_ENDIAN) {
                return BootstrapCheck.BootstrapCheckResult.failure("Little-endian native byte order is required to run Elasticsearch");
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        ByteOrder nativeByteOrder() {
            return ByteOrder.nativeOrder();
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECKS;
        }
    }

    static abstract class MightForkCheck
    implements BootstrapCheck {
        MightForkCheck() {
        }

        @Override
        public BootstrapCheck.BootstrapCheckResult check(BootstrapContext context) {
            if (this.isSystemCallFilterInstalled() && this.mightFork()) {
                return BootstrapCheck.BootstrapCheckResult.failure(this.message(context));
            }
            return BootstrapCheck.BootstrapCheckResult.success();
        }

        abstract String message(BootstrapContext var1);

        boolean isSystemCallFilterInstalled() {
            return Natives.isSystemCallFilterInstalled();
        }

        abstract boolean mightFork();

        @Override
        public final boolean alwaysEnforce() {
            return true;
        }

        @Override
        public ReferenceDocs referenceDocs() {
            return ReferenceDocs.BOOTSTRAP_CHECK_ONERROR_AND_ONOUTOFMEMORYERROR;
        }
    }
}

