/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.env;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import org.apache.lucene.util.Constants;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.PathUtils;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.env.ESFileStore;

@SuppressForbidden(reason="configures paths for the system")
public class Environment {
    private static final Path[] EMPTY_PATH_ARRAY = new Path[0];
    public static final Setting<String> PATH_HOME_SETTING = Setting.simpleString("path.home", Setting.Property.NodeScope);
    public static final Setting<List<String>> PATH_DATA_SETTING = Setting.stringListSetting("path.data", Setting.Property.NodeScope);
    public static final Setting<String> PATH_LOGS_SETTING = new Setting("path.logs", "", Function.identity(), Setting.Property.NodeScope);
    public static final Setting<List<String>> PATH_REPO_SETTING = Setting.stringListSetting("path.repo", Setting.Property.NodeScope);
    public static final Setting<String> PATH_SHARED_DATA_SETTING = Setting.simpleString("path.shared_data", Setting.Property.NodeScope);
    private final Settings settings;
    private final Path[] dataFiles;
    private final Path[] repoFiles;
    private final Path configFile;
    private final Path pluginsFile;
    private final Path modulesFile;
    private final Path sharedDataFile;
    private final Path binFile;
    private final Path libFile;
    private final Path logsFile;
    private final Path tmpFile;
    private static final String LIBFFI_TMPDIR_ENVIRONMENT_VARIABLE = "LIBFFI_TMPDIR";

    public Environment(Settings settings, Path configPath) {
        this(settings, configPath, PathUtils.get(System.getProperty("java.io.tmpdir"), new String[0]));
    }

    Environment(Settings settings, Path configPath, Path tmpPath) {
        if (!PATH_HOME_SETTING.exists(settings)) {
            throw new IllegalStateException(PATH_HOME_SETTING.getKey() + " is not configured");
        }
        Path homeFile = PathUtils.get(PATH_HOME_SETTING.get(settings), new String[0]).toAbsolutePath().normalize();
        this.configFile = configPath != null ? configPath.toAbsolutePath().normalize() : homeFile.resolve("config");
        this.tmpFile = Objects.requireNonNull(tmpPath);
        this.pluginsFile = homeFile.resolve("plugins");
        List<String> dataPaths = PATH_DATA_SETTING.get(settings);
        if (!dataPaths.isEmpty()) {
            this.dataFiles = new Path[dataPaths.size()];
            for (int i = 0; i < dataPaths.size(); ++i) {
                this.dataFiles[i] = PathUtils.get(dataPaths.get(i), new String[0]).toAbsolutePath().normalize();
            }
        } else {
            this.dataFiles = new Path[]{homeFile.resolve("data")};
        }
        this.sharedDataFile = PATH_SHARED_DATA_SETTING.exists(settings) ? PathUtils.get(PATH_SHARED_DATA_SETTING.get(settings), new String[0]).toAbsolutePath().normalize() : null;
        List<String> repoPaths = PATH_REPO_SETTING.get(settings);
        if (repoPaths.isEmpty()) {
            this.repoFiles = EMPTY_PATH_ARRAY;
        } else {
            this.repoFiles = new Path[repoPaths.size()];
            for (int i = 0; i < repoPaths.size(); ++i) {
                this.repoFiles[i] = PathUtils.get(repoPaths.get(i), new String[0]).toAbsolutePath().normalize();
            }
        }
        this.logsFile = PATH_LOGS_SETTING.exists(settings) ? PathUtils.get(PATH_LOGS_SETTING.get(settings), new String[0]).toAbsolutePath().normalize() : homeFile.resolve("logs");
        this.binFile = homeFile.resolve("bin");
        this.libFile = homeFile.resolve("lib");
        this.modulesFile = homeFile.resolve("modules");
        Settings.Builder finalSettings = Settings.builder().put(settings);
        if (PATH_DATA_SETTING.exists(settings)) {
            if (Environment.dataPathUsesList(settings)) {
                finalSettings.putList(PATH_DATA_SETTING.getKey(), Arrays.stream(this.dataFiles).map(Path::toString).toList());
            } else {
                assert (this.dataFiles.length == 1);
                finalSettings.put(PATH_DATA_SETTING.getKey(), this.dataFiles[0]);
            }
        }
        finalSettings.put(PATH_HOME_SETTING.getKey(), homeFile);
        finalSettings.put(PATH_LOGS_SETTING.getKey(), this.logsFile.toString());
        if (PATH_REPO_SETTING.exists(settings)) {
            finalSettings.putList(PATH_REPO_SETTING.getKey(), Arrays.stream(this.repoFiles).map(Path::toString).toList());
        }
        if (PATH_SHARED_DATA_SETTING.exists(settings)) {
            assert (this.sharedDataFile != null);
            finalSettings.put(PATH_SHARED_DATA_SETTING.getKey(), this.sharedDataFile.toString());
        }
        this.settings = finalSettings.build();
    }

    public Settings settings() {
        return this.settings;
    }

    public Path[] dataFiles() {
        return this.dataFiles;
    }

    public Path sharedDataFile() {
        return this.sharedDataFile;
    }

    public Path[] repoFiles() {
        return this.repoFiles;
    }

    public Path resolveRepoFile(String location) {
        return PathUtils.get(this.repoFiles, location);
    }

    public URL resolveRepoURL(URL url) {
        try {
            if ("file".equalsIgnoreCase(url.getProtocol())) {
                if (url.getHost() == null || "".equals(url.getHost())) {
                    Path path = PathUtils.get(this.repoFiles, url.toURI());
                    if (path == null) {
                        return null;
                    }
                    return path.toUri().toURL();
                }
                return null;
            }
            if ("jar".equals(url.getProtocol())) {
                String file = url.getFile();
                int pos = file.indexOf("!/");
                if (pos < 0) {
                    return null;
                }
                String jarTail = file.substring(pos);
                String filePath = file.substring(0, pos);
                URL internalUrl = new URL(filePath);
                URL normalizedUrl = this.resolveRepoURL(internalUrl);
                if (normalizedUrl == null) {
                    return null;
                }
                return new URL("jar", "", normalizedUrl.toExternalForm() + jarTail);
            }
            return null;
        }
        catch (MalformedURLException ex) {
            return null;
        }
        catch (URISyntaxException ex) {
            return null;
        }
    }

    public Path configFile() {
        return this.configFile;
    }

    public Path pluginsFile() {
        return this.pluginsFile;
    }

    public Path binFile() {
        return this.binFile;
    }

    public Path libFile() {
        return this.libFile;
    }

    public Path modulesFile() {
        return this.modulesFile;
    }

    public Path logsFile() {
        return this.logsFile;
    }

    public Path tmpFile() {
        return this.tmpFile;
    }

    public void validateTmpFile() throws IOException {
        Environment.validateTemporaryDirectory("Temporary directory", this.tmpFile);
    }

    public void validateNativesConfig() throws IOException {
        this.validateTmpFile();
        if (Constants.LINUX) {
            Environment.validateTemporaryDirectory("LIBFFI_TMPDIR environment variable", Environment.getLibffiTemporaryDirectory());
        }
    }

    private static void validateTemporaryDirectory(String description, Path path) throws IOException {
        if (path == null) {
            throw new NullPointerException(description + " was not specified");
        }
        if (!Files.exists(path, new LinkOption[0])) {
            throw new FileNotFoundException(description + " [" + path + "] does not exist or is not accessible");
        }
        if (!Files.isDirectory(path, new LinkOption[0])) {
            throw new IOException(description + " [" + path + "] is not a directory");
        }
    }

    @SuppressForbidden(reason="using PathUtils#get since libffi resolves paths without interference from the JVM")
    private static Path getLibffiTemporaryDirectory() {
        String environmentVariable = System.getenv(LIBFFI_TMPDIR_ENVIRONMENT_VARIABLE);
        if (environmentVariable == null) {
            return null;
        }
        String workingDirectory = System.getProperty("user.dir");
        if (workingDirectory == null) {
            assert (false);
            return null;
        }
        return PathUtils.get(workingDirectory, new String[0]).resolve(environmentVariable);
    }

    public static boolean dataPathUsesList(Settings settings) {
        if (!settings.hasValue(PATH_DATA_SETTING.getKey())) {
            return false;
        }
        String rawDataPath = settings.get(PATH_DATA_SETTING.getKey());
        return rawDataPath.startsWith("[") || rawDataPath.contains(",");
    }

    public static FileStore getFileStore(Path path) throws IOException {
        return new ESFileStore(Files.getFileStore(path));
    }

    public static long getUsableSpace(Path path) throws IOException {
        long freeSpaceInBytes = Environment.getFileStore(path).getUsableSpace();
        assert (freeSpaceInBytes >= 0L);
        return freeSpaceInBytes;
    }

    public static void assertEquivalent(Environment actual, Environment expected) {
        Environment.assertEquals(actual.dataFiles(), expected.dataFiles(), "dataFiles");
        Environment.assertEquals(actual.repoFiles(), expected.repoFiles(), "repoFiles");
        Environment.assertEquals(actual.configFile(), expected.configFile(), "configFile");
        Environment.assertEquals(actual.pluginsFile(), expected.pluginsFile(), "pluginsFile");
        Environment.assertEquals(actual.binFile(), expected.binFile(), "binFile");
        Environment.assertEquals(actual.libFile(), expected.libFile(), "libFile");
        Environment.assertEquals(actual.modulesFile(), expected.modulesFile(), "modulesFile");
        Environment.assertEquals(actual.logsFile(), expected.logsFile(), "logsFile");
        Environment.assertEquals(actual.tmpFile(), expected.tmpFile(), "tmpFile");
    }

    private static void assertEquals(Object actual, Object expected, String name) {
        assert (Objects.deepEquals(actual, expected)) : "actual " + name + " [" + actual + "] is different than [ " + expected + "]";
    }
}

