/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.path;

import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.function.BinaryOperator;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;
import org.elasticsearch.common.collect.Iterators;

public class PathTrie<T> {
    private static final EnumSet<TrieMatchingMode> EXPLICIT_OR_ROOT_WILDCARD = EnumSet.of(TrieMatchingMode.EXPLICIT_NODES_ONLY, TrieMatchingMode.WILDCARD_ROOT_NODES_ALLOWED);
    private final UnaryOperator<String> decoder;
    private final TrieNode root;
    private T rootValue;
    private static final String SEPARATOR = "/";
    private static final String WILDCARD = "*";

    public PathTrie(UnaryOperator<String> decoder) {
        this.decoder = decoder;
        this.root = new TrieNode(SEPARATOR, null);
    }

    public void insert(String path, T value) {
        String[] strings = path.split(SEPARATOR);
        if (strings.length == 0) {
            if (this.rootValue != null) {
                throw new IllegalArgumentException("Path [/] already has a value [" + this.rootValue + "]");
            }
            this.rootValue = value;
            return;
        }
        int index = 0;
        if (strings[0].isEmpty()) {
            index = 1;
        }
        this.root.insert(strings, index, value);
    }

    public void insertOrUpdate(String path, T value, BinaryOperator<T> updater) {
        String[] strings = path.split(SEPARATOR);
        if (strings.length == 0) {
            this.rootValue = this.rootValue != null ? updater.apply(this.rootValue, value) : value;
            return;
        }
        int index = 0;
        if (strings[0].isEmpty()) {
            index = 1;
        }
        this.root.insertOrUpdate(strings, index, value, updater);
    }

    public T retrieve(String path) {
        return this.retrieve(path, null, TrieMatchingMode.WILDCARD_NODES_ALLOWED);
    }

    public T retrieve(String path, Map<String, String> params) {
        return this.retrieve(path, params, TrieMatchingMode.WILDCARD_NODES_ALLOWED);
    }

    T retrieve(String path, Map<String, String> params, TrieMatchingMode trieMatchingMode) {
        if (path.isEmpty()) {
            return this.rootValue;
        }
        String[] strings = path.split(SEPARATOR);
        if (strings.length == 0) {
            return this.rootValue;
        }
        int index = 0;
        if (strings[0].isEmpty()) {
            index = 1;
        }
        return this.root.retrieve(strings, index, params, trieMatchingMode);
    }

    public Stream<T> retrieveAll(String path, Supplier<Map<String, String>> paramSupplier) {
        return Arrays.stream(TrieMatchingMode.values()).map(m -> this.retrieve(path, (Map)paramSupplier.get(), (TrieMatchingMode)((Object)m)));
    }

    public Iterator<T> allNodeValues() {
        return Iterators.concat(Iterators.single(this.rootValue), this.root.allNodeValues());
    }

    private class TrieNode {
        private T value;
        private String namedWildcard;
        private Map<String, TrieNode> children;

        private TrieNode(String key, T value) {
            this.value = value;
            this.children = Collections.emptyMap();
            if (TrieNode.isNamedWildcard(key)) {
                this.updateNamedWildcard(key);
            } else {
                this.namedWildcard = null;
            }
        }

        private void updateNamedWildcard(String key) {
            String newNamedWildcard = key.substring(1, key.length() - 1);
            if (!newNamedWildcard.equals(this.namedWildcard)) {
                if (this.namedWildcard != null) {
                    throw new IllegalArgumentException("Trying to use conflicting wildcard names for same path: " + this.namedWildcard + " and " + newNamedWildcard);
                }
                this.namedWildcard = newNamedWildcard;
            }
        }

        private void addInnerChild(String key, TrieNode child) {
            HashMap<String, TrieNode> newChildren = new HashMap<String, TrieNode>(this.children);
            newChildren.put(key, child);
            this.children = Collections.unmodifiableMap(newChildren);
        }

        private synchronized void insert(String[] path, int index, T value) {
            TrieNode node;
            String token;
            if (index >= path.length) {
                return;
            }
            String key = token = path[index];
            if (TrieNode.isNamedWildcard(token)) {
                key = PathTrie.WILDCARD;
            }
            if ((node = this.children.get(key)) == null) {
                Object nodeValue = index == path.length - 1 ? value : null;
                node = new TrieNode(token, nodeValue);
                this.addInnerChild(key, node);
            } else {
                if (TrieNode.isNamedWildcard(token)) {
                    node.updateNamedWildcard(token);
                }
                if (index == path.length - 1) {
                    if (node.value != null) {
                        throw new IllegalArgumentException("Path [" + String.join((CharSequence)PathTrie.SEPARATOR, path) + "] already has a value [" + node.value + "]");
                    }
                    node.value = value;
                }
            }
            node.insert(path, index + 1, value);
        }

        private synchronized void insertOrUpdate(String[] path, int index, T value, BinaryOperator<T> updater) {
            TrieNode node;
            String token;
            if (index >= path.length) {
                return;
            }
            String key = token = path[index];
            if (TrieNode.isNamedWildcard(token)) {
                key = PathTrie.WILDCARD;
            }
            if ((node = this.children.get(key)) == null) {
                Object nodeValue = index == path.length - 1 ? value : null;
                node = new TrieNode(token, nodeValue);
                this.addInnerChild(key, node);
            } else {
                if (TrieNode.isNamedWildcard(token)) {
                    node.updateNamedWildcard(token);
                }
                if (index == path.length - 1) {
                    node.value = node.value != null ? updater.apply(node.value, value) : value;
                }
            }
            node.insertOrUpdate(path, index + 1, value, updater);
        }

        private static boolean isNamedWildcard(String key) {
            return key.charAt(0) == '{' && key.charAt(key.length() - 1) == '}';
        }

        private String namedWildcard() {
            return this.namedWildcard;
        }

        private boolean isNamedWildcard() {
            return this.namedWildcard != null;
        }

        /*
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        private T retrieve(String[] path, int index, Map<String, String> params, TrieMatchingMode trieMatchingMode) {
            TrieNode wildcardNode;
            boolean usedWildcard;
            if (index >= path.length) {
                return null;
            }
            String token = path[index];
            TrieNode node = this.children.get(token);
            if (node == null) {
                if (trieMatchingMode == TrieMatchingMode.WILDCARD_NODES_ALLOWED) {
                    node = this.children.get(PathTrie.WILDCARD);
                    if (node == null) {
                        return null;
                    }
                    usedWildcard = true;
                } else if (trieMatchingMode == TrieMatchingMode.WILDCARD_ROOT_NODES_ALLOWED && index == 1) {
                    node = this.children.get(PathTrie.WILDCARD);
                    if (node == null) {
                        return null;
                    }
                    usedWildcard = true;
                } else {
                    if (trieMatchingMode != TrieMatchingMode.WILDCARD_LEAF_NODES_ALLOWED || index + 1 != path.length) return null;
                    node = this.children.get(PathTrie.WILDCARD);
                    if (node == null) {
                        return null;
                    }
                    usedWildcard = true;
                }
            } else if (index + 1 == path.length && node.value == null && !EXPLICIT_OR_ROOT_WILDCARD.contains((Object)trieMatchingMode) && (wildcardNode = this.children.get(PathTrie.WILDCARD)) != null) {
                node = wildcardNode;
                usedWildcard = true;
            } else if (index == 1 && node.value == null && trieMatchingMode == TrieMatchingMode.WILDCARD_ROOT_NODES_ALLOWED && (wildcardNode = this.children.get(PathTrie.WILDCARD)) != null) {
                node = wildcardNode;
                usedWildcard = true;
            } else {
                usedWildcard = token.equals(PathTrie.WILDCARD);
            }
            this.recordWildcardParam(params, node, token);
            if (index == path.length - 1) {
                return node.value;
            }
            Object nodeValue = node.retrieve(path, index + 1, params, trieMatchingMode);
            if (nodeValue != null || usedWildcard || trieMatchingMode == TrieMatchingMode.EXPLICIT_NODES_ONLY || (node = this.children.get(PathTrie.WILDCARD)) == null) return nodeValue;
            this.recordWildcardParam(params, node, token);
            nodeValue = node.retrieve(path, index + 1, params, trieMatchingMode);
            return nodeValue;
        }

        private void recordWildcardParam(Map<String, String> params, TrieNode node, String value) {
            if (params != null && node.isNamedWildcard()) {
                params.put(node.namedWildcard(), (String)PathTrie.this.decoder.apply(value));
            }
        }

        private Iterator<T> allNodeValues() {
            Iterator childrenIterator = Iterators.flatMap(this.children.values().iterator(), TrieNode::allNodeValues);
            if (this.value == null) {
                return childrenIterator;
            }
            return Iterators.concat(Iterators.single(this.value), childrenIterator);
        }
    }

    static enum TrieMatchingMode {
        EXPLICIT_NODES_ONLY,
        WILDCARD_ROOT_NODES_ALLOWED,
        WILDCARD_LEAF_NODES_ALLOWED,
        WILDCARD_NODES_ALLOWED;

    }
}

