/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.coordination;

import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.cluster.coordination.CoordinationStateRejectedException;
import org.elasticsearch.cluster.coordination.NodeHealthCheckFailureException;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.monitor.NodeHealthService;
import org.elasticsearch.monitor.StatusInfo;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.NodeDisconnectedException;
import org.elasticsearch.transport.ReceiveTimeoutTransportException;
import org.elasticsearch.transport.Transport;
import org.elasticsearch.transport.TransportConnectionListener;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportRequestOptions;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;

public class LeaderChecker {
    private static final Logger logger = LogManager.getLogger(LeaderChecker.class);
    static final String LEADER_CHECK_ACTION_NAME = "internal:coordination/fault_detection/leader_check";
    public static final Setting<TimeValue> LEADER_CHECK_INTERVAL_SETTING = Setting.timeSetting("cluster.fault_detection.leader_check.interval", TimeValue.timeValueMillis(1000L), TimeValue.timeValueMillis(100L), Setting.Property.NodeScope);
    public static final Setting<TimeValue> LEADER_CHECK_TIMEOUT_SETTING = Setting.timeSetting("cluster.fault_detection.leader_check.timeout", TimeValue.timeValueMillis(10000L), TimeValue.timeValueMillis(1L), Setting.Property.NodeScope);
    public static final Setting<Integer> LEADER_CHECK_RETRY_COUNT_SETTING = Setting.intSetting("cluster.fault_detection.leader_check.retry_count", 3, 1, Setting.Property.NodeScope);
    private final TimeValue leaderCheckInterval;
    private final TimeValue leaderCheckTimeout;
    private final int leaderCheckRetryCount;
    private final TransportService transportService;
    private final LeaderFailureListener leaderFailureListener;
    private final NodeHealthService nodeHealthService;
    private final AtomicReference<CheckScheduler> currentChecker = new AtomicReference();
    private volatile DiscoveryNodes discoveryNodes;
    private static final String RESTARTING_DISCOVERY_TEXT = "restarting discovery; more details may be available in the master node logs";

    LeaderChecker(Settings settings, TransportService transportService, LeaderFailureListener leaderFailureListener, NodeHealthService nodeHealthService) {
        this.leaderCheckInterval = LEADER_CHECK_INTERVAL_SETTING.get(settings);
        this.leaderCheckTimeout = LEADER_CHECK_TIMEOUT_SETTING.get(settings);
        this.leaderCheckRetryCount = LEADER_CHECK_RETRY_COUNT_SETTING.get(settings);
        this.transportService = transportService;
        this.leaderFailureListener = leaderFailureListener;
        this.nodeHealthService = nodeHealthService;
        transportService.registerRequestHandler(LEADER_CHECK_ACTION_NAME, EsExecutors.DIRECT_EXECUTOR_SERVICE, false, false, LeaderCheckRequest::new, (request, channel, task) -> {
            this.handleLeaderCheck((LeaderCheckRequest)request);
            channel.sendResponse(TransportResponse.Empty.INSTANCE);
        });
        transportService.addConnectionListener(new TransportConnectionListener(){

            @Override
            public void onNodeDisconnected(DiscoveryNode node, Transport.Connection connection) {
                LeaderChecker.this.handleDisconnectedNode(node);
            }
        });
    }

    public DiscoveryNode leader() {
        CheckScheduler checkScheduler = this.currentChecker.get();
        return checkScheduler == null ? null : checkScheduler.leader;
    }

    void updateLeader(@Nullable DiscoveryNode leader) {
        assert (!this.transportService.getLocalNode().equals(leader));
        CheckScheduler checkScheduler = leader != null ? new CheckScheduler(leader) : null;
        CheckScheduler previousChecker = this.currentChecker.getAndSet(checkScheduler);
        if (previousChecker != null) {
            previousChecker.close();
        }
        if (checkScheduler != null) {
            checkScheduler.handleWakeUp();
        }
    }

    void setCurrentNodes(DiscoveryNodes discoveryNodes) {
        logger.trace(() -> Strings.format("setCurrentNodes: %s", discoveryNodes.mastersFirstStream().toList()));
        this.discoveryNodes = discoveryNodes;
    }

    boolean currentNodeIsMaster() {
        return this.discoveryNodes.isLocalNodeElectedMaster();
    }

    private void handleLeaderCheck(LeaderCheckRequest request) {
        DiscoveryNodes discoveryNodes = this.discoveryNodes;
        assert (discoveryNodes != null);
        StatusInfo statusInfo = this.nodeHealthService.getHealth();
        if (statusInfo.getStatus() == StatusInfo.Status.UNHEALTHY) {
            logger.debug("this node is unhealthy [{}], rejecting leader check: {}", (Object)statusInfo.getInfo(), (Object)request);
            throw new NodeHealthCheckFailureException(statusInfo.getInfo(), new Object[0]);
        }
        if (!discoveryNodes.isLocalNodeElectedMaster()) {
            logger.debug("rejecting leader check on non-master: {}", (Object)request);
            throw new CoordinationStateRejectedException("no longer the elected master", new Object[0]);
        }
        if (!discoveryNodes.nodeExists(request.getSender())) {
            logger.debug("rejecting leader check from removed node: {}", (Object)request);
            throw new CoordinationStateRejectedException("rejecting check since [" + request.getSender().descriptionWithoutAttributes() + "] has been removed from the cluster", new Object[0]);
        }
        logger.trace("handling {}", (Object)request);
    }

    private void handleDisconnectedNode(DiscoveryNode discoveryNode) {
        CheckScheduler checkScheduler = this.currentChecker.get();
        if (checkScheduler != null) {
            checkScheduler.handleDisconnectedNode(discoveryNode);
        } else {
            logger.trace("disconnect event ignored for {}, no check scheduler", (Object)discoveryNode);
        }
    }

    @FunctionalInterface
    static interface LeaderFailureListener {
        public void onLeaderFailure(Supplier<String> var1, Exception var2);
    }

    private class CheckScheduler
    implements Releasable {
        private final DiscoveryNode leader;
        private final AtomicBoolean isClosed = new AtomicBoolean();
        private int rejectedCountSinceLastSuccess;
        private int timeoutCountSinceLastSuccess;

        CheckScheduler(DiscoveryNode leader) {
            this.leader = leader;
        }

        @Override
        public void close() {
            if (!this.isClosed.compareAndSet(false, true)) {
                logger.trace("already closed, doing nothing");
            } else {
                logger.debug("closed");
            }
        }

        void handleWakeUp() {
            if (this.isClosed.get()) {
                logger.trace("closed check scheduler woken up, doing nothing");
                return;
            }
            logger.trace("checking {} with [{}] = {}", (Object)this.leader, (Object)LEADER_CHECK_TIMEOUT_SETTING.getKey(), (Object)LeaderChecker.this.leaderCheckTimeout);
            LeaderChecker.this.transportService.sendRequest(this.leader, LeaderChecker.LEADER_CHECK_ACTION_NAME, (TransportRequest)new LeaderCheckRequest(LeaderChecker.this.transportService.getLocalNode()), TransportRequestOptions.of(LeaderChecker.this.leaderCheckTimeout, TransportRequestOptions.Type.PING), new TransportResponseHandler.Empty(){

                @Override
                public Executor executor() {
                    return TransportResponseHandler.TRANSPORT_WORKER;
                }

                @Override
                public void handleResponse() {
                    if (CheckScheduler.this.isClosed.get()) {
                        logger.debug("closed check scheduler received a response, doing nothing");
                        return;
                    }
                    CheckScheduler.this.rejectedCountSinceLastSuccess = 0;
                    CheckScheduler.this.timeoutCountSinceLastSuccess = 0;
                    CheckScheduler.this.scheduleNextWakeUp();
                }

                @Override
                public void handleException(TransportException exp) {
                    if (CheckScheduler.this.isClosed.get()) {
                        logger.debug("closed check scheduler received a response, doing nothing");
                        return;
                    }
                    if (exp instanceof ConnectTransportException || exp.getCause() instanceof ConnectTransportException) {
                        logger.debug(() -> "leader [" + CheckScheduler.this.leader + "] disconnected during check", (Throwable)exp);
                        CheckScheduler.this.leaderFailed(() -> Strings.format("master node [%s] disconnected, restarting discovery [%s]", CheckScheduler.this.leader.descriptionWithoutAttributes(), ExceptionsHelper.unwrapCause(exp).getMessage()), exp);
                        return;
                    }
                    if (exp.getCause() instanceof NodeHealthCheckFailureException) {
                        logger.debug(() -> "leader [" + CheckScheduler.this.leader + "] health check failed", (Throwable)exp);
                        CheckScheduler.this.leaderFailed(() -> Strings.format("master node [%s] reported itself as unhealthy [%s], %s", CheckScheduler.this.leader.descriptionWithoutAttributes(), exp.getCause().getMessage(), LeaderChecker.RESTARTING_DISCOVERY_TEXT), exp);
                        return;
                    }
                    if (exp instanceof ReceiveTimeoutTransportException) {
                        ++CheckScheduler.this.timeoutCountSinceLastSuccess;
                    } else {
                        ++CheckScheduler.this.rejectedCountSinceLastSuccess;
                    }
                    long failureCount = CheckScheduler.this.rejectedCountSinceLastSuccess + CheckScheduler.this.timeoutCountSinceLastSuccess;
                    if (failureCount >= (long)LeaderChecker.this.leaderCheckRetryCount) {
                        logger.debug(() -> Strings.format("leader [%s] failed %s consecutive checks (rejected [%s], timed out [%s], limit [%s] is %s)", CheckScheduler.this.leader, failureCount, CheckScheduler.this.rejectedCountSinceLastSuccess, CheckScheduler.this.timeoutCountSinceLastSuccess, LEADER_CHECK_RETRY_COUNT_SETTING.getKey(), LeaderChecker.this.leaderCheckRetryCount), (Throwable)exp);
                        CheckScheduler.this.leaderFailed(() -> Strings.format("[%s] consecutive checks of the master node [%s] were unsuccessful ([%s] rejected, [%s] timed out), %s [last unsuccessful check: %s]", failureCount, CheckScheduler.this.leader.descriptionWithoutAttributes(), CheckScheduler.this.rejectedCountSinceLastSuccess, CheckScheduler.this.timeoutCountSinceLastSuccess, LeaderChecker.RESTARTING_DISCOVERY_TEXT, ExceptionsHelper.unwrapCause(exp).getMessage()), exp);
                        return;
                    }
                    logger.debug(() -> Strings.format("%s consecutive failures (limit [%s] is %s) with leader [%s]", failureCount, LEADER_CHECK_RETRY_COUNT_SETTING.getKey(), LeaderChecker.this.leaderCheckRetryCount, CheckScheduler.this.leader), (Throwable)exp);
                    CheckScheduler.this.scheduleNextWakeUp();
                }
            });
        }

        void leaderFailed(final Supplier<String> messageSupplier, final Exception e) {
            if (this.isClosed.compareAndSet(false, true)) {
                LeaderChecker.this.transportService.getThreadPool().executor("cluster_coordination").execute(new AbstractRunnable(){

                    @Override
                    protected void doRun() {
                        LeaderChecker.this.leaderFailureListener.onLeaderFailure(messageSupplier, e);
                    }

                    @Override
                    public void onRejection(Exception e2) {
                        EsRejectedExecutionException esre;
                        e.addSuppressed(e2);
                        logger.debug("rejected execution of onLeaderFailure", (Throwable)e);
                        assert (e2 instanceof EsRejectedExecutionException && (esre = (EsRejectedExecutionException)e2).isExecutorShutdown()) : e;
                    }

                    @Override
                    public void onFailure(Exception e2) {
                        e2.addSuppressed(e);
                        logger.error("failed execution of onLeaderFailure", (Throwable)e2);
                        assert (false) : e2;
                    }

                    public String toString() {
                        return "notification of leader failure: " + e.getMessage();
                    }
                });
            } else {
                logger.trace("already closed, not failing leader");
            }
        }

        void handleDisconnectedNode(DiscoveryNode discoveryNode) {
            if (discoveryNode.equals(this.leader)) {
                logger.debug("leader [{}] disconnected", (Object)this.leader);
                this.leaderFailed(() -> Strings.format("master node [%s] disconnected, restarting discovery", this.leader.descriptionWithoutAttributes()), new NodeDisconnectedException(discoveryNode, "disconnected"));
            }
        }

        private void scheduleNextWakeUp() {
            logger.trace("scheduling next check of {} for [{}] = {}", (Object)this.leader, (Object)LEADER_CHECK_INTERVAL_SETTING.getKey(), (Object)LeaderChecker.this.leaderCheckInterval);
            LeaderChecker.this.transportService.getThreadPool().schedule(new Runnable(){

                @Override
                public void run() {
                    CheckScheduler.this.handleWakeUp();
                }

                public String toString() {
                    return "scheduled check of leader " + CheckScheduler.this.leader;
                }
            }, LeaderChecker.this.leaderCheckInterval, EsExecutors.DIRECT_EXECUTOR_SERVICE);
        }
    }

    static class LeaderCheckRequest
    extends TransportRequest {
        private final DiscoveryNode sender;

        LeaderCheckRequest(DiscoveryNode sender) {
            this.sender = sender;
        }

        LeaderCheckRequest(StreamInput in) throws IOException {
            super(in);
            this.sender = new DiscoveryNode(in);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.sender.writeTo(out);
        }

        public DiscoveryNode getSender() {
            return this.sender;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            LeaderCheckRequest that = (LeaderCheckRequest)o;
            return Objects.equals(this.sender, that.sender);
        }

        public int hashCode() {
            return Objects.hash(this.sender);
        }

        @Override
        public String toString() {
            return "LeaderCheckRequest{sender=" + this.sender + "}";
        }
    }
}

