/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.coordination;

import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.cluster.node.hotthreads.NodeHotThreads;
import org.elasticsearch.action.admin.cluster.node.hotthreads.NodesHotThreadsRequest;
import org.elasticsearch.action.admin.cluster.node.hotthreads.NodesHotThreadsResponse;
import org.elasticsearch.action.admin.cluster.node.hotthreads.TransportNodesHotThreadsAction;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.ReferenceDocs;
import org.elasticsearch.common.logging.ChunkedLoggingStream;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.PrioritizedThrottledTaskRunner;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Releasables;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.TransportService;

public class LagDetector {
    private static final Logger logger = LogManager.getLogger(LagDetector.class);
    public static final Setting<TimeValue> CLUSTER_FOLLOWER_LAG_TIMEOUT_SETTING = Setting.timeSetting("cluster.follower_lag.timeout", TimeValue.timeValueMillis(90000L), TimeValue.timeValueMillis(1L), Setting.Property.NodeScope);
    private final TimeValue clusterStateApplicationTimeout;
    private final LagListener lagListener;
    private final Supplier<DiscoveryNode> localNodeSupplier;
    private final ThreadPool threadPool;
    private final Executor clusterCoordinationExecutor;
    private final Map<DiscoveryNode, NodeAppliedStateTracker> appliedStateTrackersByNode = ConcurrentCollections.newConcurrentMap();

    public LagDetector(Settings settings, ThreadPool threadPool, LagListener lagListener, Supplier<DiscoveryNode> localNodeSupplier) {
        this.threadPool = threadPool;
        this.clusterCoordinationExecutor = threadPool.executor("cluster_coordination");
        this.clusterStateApplicationTimeout = CLUSTER_FOLLOWER_LAG_TIMEOUT_SETTING.get(settings);
        this.lagListener = lagListener;
        this.localNodeSupplier = localNodeSupplier;
    }

    public void setTrackedNodes(Iterable<DiscoveryNode> discoveryNodes) {
        HashSet discoveryNodeSet = new HashSet();
        discoveryNodes.forEach(discoveryNodeSet::add);
        discoveryNodeSet.remove(this.localNodeSupplier.get());
        this.appliedStateTrackersByNode.keySet().retainAll(discoveryNodeSet);
        discoveryNodeSet.forEach(node -> this.appliedStateTrackersByNode.putIfAbsent((DiscoveryNode)node, new NodeAppliedStateTracker((DiscoveryNode)node)));
    }

    public void clearTrackedNodes() {
        this.appliedStateTrackersByNode.clear();
    }

    public void setAppliedVersion(DiscoveryNode discoveryNode, long appliedVersion) {
        NodeAppliedStateTracker nodeAppliedStateTracker = this.appliedStateTrackersByNode.get(discoveryNode);
        if (nodeAppliedStateTracker == null) {
            logger.trace("node {} applied version {} but this node's version is not being tracked", (Object)discoveryNode, (Object)appliedVersion);
        } else {
            nodeAppliedStateTracker.increaseAppliedVersion(appliedVersion);
        }
    }

    public void startLagDetector(final long version) {
        final List<NodeAppliedStateTracker> laggingTrackers = this.appliedStateTrackersByNode.values().stream().filter(t -> t.appliedVersionLessThan(version)).toList();
        if (laggingTrackers.isEmpty()) {
            logger.trace("lag detection for version {} is unnecessary: {}", (Object)version, this.appliedStateTrackersByNode.values());
        } else {
            logger.debug("starting lag detector for version {}: {}", (Object)version, laggingTrackers);
            this.threadPool.scheduleUnlessShuttingDown(this.clusterStateApplicationTimeout, this.clusterCoordinationExecutor, new Runnable(){

                @Override
                public void run() {
                    laggingTrackers.forEach(t -> t.checkForLag(version));
                }

                public String toString() {
                    return "lag detector for version " + version + " on " + laggingTrackers;
                }
            });
        }
    }

    public String toString() {
        return "LagDetector{clusterStateApplicationTimeout=" + this.clusterStateApplicationTimeout + ", appliedStateTrackersByNode=" + this.appliedStateTrackersByNode.values() + "}";
    }

    Set<DiscoveryNode> getTrackedNodes() {
        return Collections.unmodifiableSet(this.appliedStateTrackersByNode.keySet());
    }

    public static interface LagListener {
        public void onLagDetected(DiscoveryNode var1, long var2, long var4);
    }

    private class NodeAppliedStateTracker {
        private final DiscoveryNode discoveryNode;
        private final AtomicLong appliedVersion = new AtomicLong();

        NodeAppliedStateTracker(DiscoveryNode discoveryNode) {
            this.discoveryNode = discoveryNode;
        }

        void increaseAppliedVersion(long appliedVersion) {
            long maxAppliedVersion = this.appliedVersion.accumulateAndGet(appliedVersion, Math::max);
            logger.trace("{} applied version {}, max now {}", (Object)this, (Object)appliedVersion, (Object)maxAppliedVersion);
        }

        boolean appliedVersionLessThan(long version) {
            return this.appliedVersion.get() < version;
        }

        public String toString() {
            return "NodeAppliedStateTracker{discoveryNode=" + this.discoveryNode + ", appliedVersion=" + this.appliedVersion + "}";
        }

        void checkForLag(long version) {
            if (LagDetector.this.appliedStateTrackersByNode.get(this.discoveryNode) != this) {
                logger.trace("{} no longer active when checking version {}", (Object)this, (Object)version);
                return;
            }
            long appliedVersion = this.appliedVersion.get();
            if (version <= appliedVersion) {
                logger.trace("{} satisfied when checking version {}, node applied version {}", (Object)this, (Object)version, (Object)appliedVersion);
                return;
            }
            logger.warn("node [{}] is lagging at cluster state version [{}], although publication of cluster state version [{}] completed [{}] ago", (Object)this.discoveryNode, (Object)appliedVersion, (Object)version, (Object)LagDetector.this.clusterStateApplicationTimeout);
            LagDetector.this.lagListener.onLagDetected(this.discoveryNode, appliedVersion, version);
        }
    }

    static class HotThreadsLoggingTask
    extends AbstractRunnable
    implements Comparable<HotThreadsLoggingTask> {
        private final String nodeHotThreads;
        private final Releasable releasable;
        private final String prefix;

        HotThreadsLoggingTask(DiscoveryNode discoveryNode, long appliedVersion, long expectedVersion, String nodeHotThreads, Releasable releasable) {
            this.nodeHotThreads = nodeHotThreads;
            this.releasable = releasable;
            this.prefix = org.elasticsearch.common.Strings.format("hot threads from node [%s] lagging at version [%d] despite commit of cluster state version [%d]", discoveryNode.descriptionWithoutAttributes(), appliedVersion, expectedVersion);
        }

        @Override
        public void onFailure(Exception e) {
            logger.error(org.elasticsearch.common.Strings.format("unexpected exception reporting %s", this.prefix), (Throwable)e);
        }

        @Override
        protected void doRun() throws Exception {
            try (OutputStreamWriter writer = new OutputStreamWriter(ChunkedLoggingStream.create(logger, Level.DEBUG, this.prefix, ReferenceDocs.LAGGING_NODE_TROUBLESHOOTING), StandardCharsets.UTF_8);){
                writer.write(this.nodeHotThreads);
            }
        }

        @Override
        public void onAfter() {
            Releasables.closeExpectNoException(this.releasable);
        }

        @Override
        public int compareTo(HotThreadsLoggingTask o) {
            return 0;
        }
    }

    static class HotThreadsLoggingLagListener
    implements LagListener {
        private final TransportService transportService;
        private final Client client;
        private final LagListener delegate;
        private final PrioritizedThrottledTaskRunner<HotThreadsLoggingTask> loggingTaskRunner;

        HotThreadsLoggingLagListener(TransportService transportService, Client client, LagListener delegate) {
            this.transportService = transportService;
            this.client = client;
            this.delegate = delegate;
            this.loggingTaskRunner = new PrioritizedThrottledTaskRunner("hot_threads", 1, transportService.getThreadPool().generic());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void onLagDetected(final DiscoveryNode discoveryNode, final long appliedVersion, final long expectedVersion) {
            try {
                if (!logger.isDebugEnabled()) {
                    return;
                }
                if (this.client == null) {
                    return;
                }
                final ActionListener<NodesHotThreadsResponse> debugListener = new ActionListener<NodesHotThreadsResponse>(){

                    @Override
                    public void onResponse(NodesHotThreadsResponse nodesHotThreadsResponse) {
                        if (nodesHotThreadsResponse.getNodes().size() == 0) {
                            assert (nodesHotThreadsResponse.failures().size() == 1);
                            this.onFailure(nodesHotThreadsResponse.failures().get(0));
                            return;
                        }
                        nodesHotThreadsResponse.mustIncRef();
                        loggingTaskRunner.enqueueTask(new HotThreadsLoggingTask(discoveryNode, appliedVersion, expectedVersion, ((NodeHotThreads)nodesHotThreadsResponse.getNodes().get(0)).getHotThreads(), Releasables.assertOnce(nodesHotThreadsResponse::decRef)));
                    }

                    @Override
                    public void onFailure(Exception e) {
                        logger.debug(() -> Strings.format("failed to get hot threads from node [%s] lagging at version %s despite commit of cluster state version [%s]", discoveryNode.descriptionWithoutAttributes(), appliedVersion, expectedVersion), (Throwable)e);
                    }
                };
                this.transportService.connectToNode(discoveryNode, new ActionListener<Releasable>(){

                    /*
                     * WARNING - Removed try catching itself - possible behaviour change.
                     */
                    @Override
                    public void onResponse(Releasable releasable) {
                        boolean success = false;
                        ThreadContext threadContext = transportService.getThreadPool().getThreadContext();
                        try (ThreadContext.StoredContext ignored = threadContext.stashContext();){
                            threadContext.markAsSystemContext();
                            client.execute(TransportNodesHotThreadsAction.TYPE, new NodesHotThreadsRequest(discoveryNode).threads(500), ActionListener.runBefore(debugListener, () -> Releasables.close(releasable)));
                            success = true;
                        }
                        finally {
                            if (!success) {
                                Releasables.close(releasable);
                            }
                        }
                    }

                    @Override
                    public void onFailure(Exception e) {
                        debugListener.onFailure(e);
                    }
                });
            }
            finally {
                this.delegate.onLagDetected(discoveryNode, appliedVersion, expectedVersion);
            }
        }
    }
}

