/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.coordination;

import java.io.IOException;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRunnable;
import org.elasticsearch.action.support.ChannelActionListener;
import org.elasticsearch.cluster.coordination.CoordinationStateRejectedException;
import org.elasticsearch.cluster.coordination.Coordinator;
import org.elasticsearch.cluster.coordination.NodeHealthCheckFailureException;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.core.CheckedRunnable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.monitor.NodeHealthService;
import org.elasticsearch.monitor.StatusInfo;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.ReceiveTimeoutTransportException;
import org.elasticsearch.transport.Transport;
import org.elasticsearch.transport.TransportConnectionListener;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportRequestOptions;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;

public final class FollowersChecker {
    private static final Logger logger = LogManager.getLogger(FollowersChecker.class);
    public static final String FOLLOWER_CHECK_ACTION_NAME = "internal:coordination/fault_detection/follower_check";
    public static final Setting<TimeValue> FOLLOWER_CHECK_INTERVAL_SETTING = Setting.timeSetting("cluster.fault_detection.follower_check.interval", TimeValue.timeValueMillis(1000L), TimeValue.timeValueMillis(100L), Setting.Property.NodeScope);
    public static final Setting<TimeValue> FOLLOWER_CHECK_TIMEOUT_SETTING = Setting.timeSetting("cluster.fault_detection.follower_check.timeout", TimeValue.timeValueMillis(10000L), TimeValue.timeValueMillis(1L), Setting.Property.NodeScope);
    public static final Setting<Integer> FOLLOWER_CHECK_RETRY_COUNT_SETTING = Setting.intSetting("cluster.fault_detection.follower_check.retry_count", 3, 1, Setting.Property.NodeScope);
    private final TimeValue followerCheckInterval;
    private final TimeValue followerCheckTimeout;
    private final int followerCheckRetryCount;
    private final BiConsumer<DiscoveryNode, String> onNodeFailure;
    private final Consumer<FollowerCheckRequest> handleRequestAndUpdateState;
    private final Object mutex = new Object();
    private final Map<DiscoveryNode, FollowerChecker> followerCheckers = ConcurrentCollections.newConcurrentMap();
    private final Set<DiscoveryNode> faultyNodes = new HashSet<DiscoveryNode>();
    private final TransportService transportService;
    private final NodeHealthService nodeHealthService;
    private final Executor clusterCoordinationExecutor;
    private volatile FastResponseState fastResponseState;

    public FollowersChecker(Settings settings, TransportService transportService, Consumer<FollowerCheckRequest> handleRequestAndUpdateState, BiConsumer<DiscoveryNode, String> onNodeFailure, NodeHealthService nodeHealthService) {
        this.transportService = transportService;
        this.handleRequestAndUpdateState = handleRequestAndUpdateState;
        this.onNodeFailure = onNodeFailure;
        this.nodeHealthService = nodeHealthService;
        this.clusterCoordinationExecutor = transportService.getThreadPool().executor("cluster_coordination");
        this.followerCheckInterval = FOLLOWER_CHECK_INTERVAL_SETTING.get(settings);
        this.followerCheckTimeout = FOLLOWER_CHECK_TIMEOUT_SETTING.get(settings);
        this.followerCheckRetryCount = FOLLOWER_CHECK_RETRY_COUNT_SETTING.get(settings);
        this.updateFastResponseState(0L, Coordinator.Mode.CANDIDATE);
        transportService.registerRequestHandler(FOLLOWER_CHECK_ACTION_NAME, EsExecutors.DIRECT_EXECUTOR_SERVICE, false, false, FollowerCheckRequest::new, (request, transportChannel, task) -> this.handleFollowerCheck((FollowerCheckRequest)request, new ChannelActionListener<TransportResponse>(transportChannel).map(ignored -> TransportResponse.Empty.INSTANCE)));
        transportService.addConnectionListener(new TransportConnectionListener(){

            @Override
            public void onNodeDisconnected(DiscoveryNode node, Transport.Connection connection) {
                FollowersChecker.this.handleDisconnectedNode(node);
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setCurrentNodes(DiscoveryNodes discoveryNodes) {
        Object object = this.mutex;
        synchronized (object) {
            Predicate<DiscoveryNode> isUnknownNode = n -> !discoveryNodes.nodeExists((DiscoveryNode)n);
            this.followerCheckers.keySet().removeIf(isUnknownNode);
            this.faultyNodes.removeIf(isUnknownNode);
            discoveryNodes.mastersFirstStream().forEach(discoveryNode -> {
                if (!(discoveryNode.equals(discoveryNodes.getLocalNode()) || this.followerCheckers.containsKey(discoveryNode) || this.faultyNodes.contains(discoveryNode))) {
                    FollowerChecker followerChecker = new FollowerChecker((DiscoveryNode)discoveryNode);
                    this.followerCheckers.put((DiscoveryNode)discoveryNode, followerChecker);
                    followerChecker.start();
                }
            });
        }
    }

    public void clearCurrentNodes() {
        this.setCurrentNodes(DiscoveryNodes.EMPTY_NODES);
    }

    public void updateFastResponseState(long term, Coordinator.Mode mode) {
        this.fastResponseState = new FastResponseState(term, mode);
    }

    private void handleFollowerCheck(final FollowerCheckRequest request, ActionListener<Void> listener) {
        StatusInfo statusInfo = this.nodeHealthService.getHealth();
        if (statusInfo.getStatus() == StatusInfo.Status.UNHEALTHY) {
            String message = "handleFollowerCheck: node is unhealthy [" + statusInfo.getInfo() + "], rejecting " + statusInfo.getInfo();
            logger.debug(message);
            throw new NodeHealthCheckFailureException(message, new Object[0]);
        }
        FastResponseState responder = this.fastResponseState;
        if (responder.mode == Coordinator.Mode.FOLLOWER && responder.term == request.term) {
            logger.trace("responding to {} on fast path", (Object)request);
            listener.onResponse(null);
            return;
        }
        if (request.term < responder.term) {
            throw new CoordinationStateRejectedException("rejecting " + request + " since local state is " + this, new Object[0]);
        }
        this.clusterCoordinationExecutor.execute(ActionRunnable.run(listener, new CheckedRunnable<Exception>(){

            @Override
            public void run() {
                logger.trace("responding to {} on slow path", (Object)request);
                FollowersChecker.this.handleRequestAndUpdateState.accept(request);
            }

            public String toString() {
                return "responding to [" + request + "] on slow path";
            }
        }));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<DiscoveryNode> getFaultyNodes() {
        Object object = this.mutex;
        synchronized (object) {
            return new HashSet<DiscoveryNode>(this.faultyNodes);
        }
    }

    public String toString() {
        return "FollowersChecker{followerCheckInterval=" + this.followerCheckInterval + ", followerCheckTimeout=" + this.followerCheckTimeout + ", followerCheckRetryCount=" + this.followerCheckRetryCount + ", followerCheckers=" + this.followerCheckers + ", faultyNodes=" + this.faultyNodes + ", fastResponseState=" + this.fastResponseState + "}";
    }

    FastResponseState getFastResponseState() {
        return this.fastResponseState;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Set<DiscoveryNode> getKnownFollowers() {
        Object object = this.mutex;
        synchronized (object) {
            HashSet<DiscoveryNode> knownFollowers = new HashSet<DiscoveryNode>(this.faultyNodes);
            knownFollowers.addAll(this.followerCheckers.keySet());
            return knownFollowers;
        }
    }

    private void handleDisconnectedNode(DiscoveryNode discoveryNode) {
        FollowerChecker followerChecker = this.followerCheckers.get(discoveryNode);
        if (followerChecker != null) {
            followerChecker.failNode("disconnected");
        }
    }

    record FastResponseState(long term, Coordinator.Mode mode) {
    }

    public static class FollowerCheckRequest
    extends TransportRequest {
        private final long term;
        private final DiscoveryNode sender;

        public long getTerm() {
            return this.term;
        }

        public DiscoveryNode getSender() {
            return this.sender;
        }

        public FollowerCheckRequest(long term, DiscoveryNode sender) {
            this.term = term;
            this.sender = sender;
        }

        public FollowerCheckRequest(StreamInput in) throws IOException {
            super(in);
            this.term = in.readLong();
            this.sender = new DiscoveryNode(in);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeLong(this.term);
            this.sender.writeTo(out);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            FollowerCheckRequest that = (FollowerCheckRequest)o;
            return this.term == that.term && Objects.equals(this.sender, that.sender);
        }

        @Override
        public String toString() {
            return "FollowerCheckRequest{term=" + this.term + ", sender=" + this.sender + "}";
        }

        public int hashCode() {
            return Objects.hash(this.term, this.sender);
        }
    }

    private class FollowerChecker {
        private final DiscoveryNode discoveryNode;
        private int failureCountSinceLastSuccess;
        private int timeoutCountSinceLastSuccess;

        FollowerChecker(DiscoveryNode discoveryNode) {
            this.discoveryNode = discoveryNode;
        }

        private boolean running() {
            return this == FollowersChecker.this.followerCheckers.get(this.discoveryNode);
        }

        void start() {
            assert (this.running());
            this.handleWakeUp();
        }

        private void handleWakeUp() {
            if (!this.running()) {
                logger.trace("handleWakeUp: not running");
                return;
            }
            FollowerCheckRequest request = new FollowerCheckRequest(FollowersChecker.this.fastResponseState.term, FollowersChecker.this.transportService.getLocalNode());
            logger.trace("handleWakeUp: checking {} with {}", (Object)this.discoveryNode, (Object)request);
            FollowersChecker.this.transportService.sendRequest(this.discoveryNode, FollowersChecker.FOLLOWER_CHECK_ACTION_NAME, (TransportRequest)request, TransportRequestOptions.of(FollowersChecker.this.followerCheckTimeout, TransportRequestOptions.Type.PING), new TransportResponseHandler.Empty(){

                @Override
                public Executor executor() {
                    return TransportResponseHandler.TRANSPORT_WORKER;
                }

                @Override
                public void handleResponse() {
                    if (!FollowerChecker.this.running()) {
                        logger.trace("{} no longer running", (Object)FollowerChecker.this);
                        return;
                    }
                    FollowerChecker.this.failureCountSinceLastSuccess = 0;
                    FollowerChecker.this.timeoutCountSinceLastSuccess = 0;
                    logger.trace("{} check successful", (Object)FollowerChecker.this);
                    FollowerChecker.this.scheduleNextWakeUp();
                }

                @Override
                public void handleException(TransportException exp) {
                    Object reason;
                    if (!FollowerChecker.this.running()) {
                        logger.debug(() -> Strings.format("%s no longer running", FollowerChecker.this), (Throwable)exp);
                        return;
                    }
                    if (exp instanceof ReceiveTimeoutTransportException) {
                        ++FollowerChecker.this.timeoutCountSinceLastSuccess;
                    } else {
                        ++FollowerChecker.this.failureCountSinceLastSuccess;
                    }
                    if (exp instanceof ConnectTransportException || exp.getCause() instanceof ConnectTransportException) {
                        logger.debug(() -> Strings.format("%s disconnected", FollowerChecker.this), (Throwable)exp);
                        reason = "disconnected";
                    } else if (exp.getCause() instanceof NodeHealthCheckFailureException) {
                        logger.debug(() -> Strings.format("%s health check failed", FollowerChecker.this), (Throwable)exp);
                        reason = "health check failed";
                    } else if (FollowerChecker.this.failureCountSinceLastSuccess + FollowerChecker.this.timeoutCountSinceLastSuccess >= FollowersChecker.this.followerCheckRetryCount) {
                        logger.debug(() -> Strings.format("%s failed too many times", FollowerChecker.this), (Throwable)exp);
                        reason = "followers check retry count exceeded [timeouts=" + FollowerChecker.this.timeoutCountSinceLastSuccess + ", failures=" + FollowerChecker.this.failureCountSinceLastSuccess + "]";
                    } else {
                        logger.debug(() -> Strings.format("%s failed, retrying", FollowerChecker.this), (Throwable)exp);
                        FollowerChecker.this.scheduleNextWakeUp();
                        return;
                    }
                    FollowerChecker.this.failNode((String)reason);
                }
            });
        }

        void failNode(final String reason) {
            FollowersChecker.this.clusterCoordinationExecutor.execute(new AbstractRunnable(){

                @Override
                public void onRejection(Exception e) {
                    logger.debug(() -> Strings.format("rejected task to fail node [%s] with reason [%s]", FollowerChecker.this.discoveryNode, reason), (Throwable)e);
                    if (e instanceof EsRejectedExecutionException) {
                        EsRejectedExecutionException esRejectedExecutionException = (EsRejectedExecutionException)e;
                        assert (esRejectedExecutionException.isExecutorShutdown());
                    } else assert (false) : e;
                }

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                protected void doRun() {
                    Object object = FollowersChecker.this.mutex;
                    synchronized (object) {
                        if (!FollowerChecker.this.running()) {
                            logger.trace("{} no longer running, not marking faulty", (Object)FollowerChecker.this);
                            return;
                        }
                        logger.debug("{} marking node as faulty", (Object)FollowerChecker.this);
                        FollowersChecker.this.faultyNodes.add(FollowerChecker.this.discoveryNode);
                        FollowersChecker.this.followerCheckers.remove(FollowerChecker.this.discoveryNode);
                    }
                    FollowersChecker.this.onNodeFailure.accept(FollowerChecker.this.discoveryNode, reason);
                }

                @Override
                public void onFailure(Exception e) {
                    assert (false) : e;
                    logger.error(() -> Strings.format("unexpected failure when failing node [%s] with reason [%s]", FollowerChecker.this.discoveryNode, reason), (Throwable)e);
                }

                public String toString() {
                    return "detected failure of " + FollowerChecker.this.discoveryNode;
                }
            });
        }

        private void scheduleNextWakeUp() {
            FollowersChecker.this.transportService.getThreadPool().schedule(new Runnable(){

                @Override
                public void run() {
                    FollowerChecker.this.handleWakeUp();
                }

                public String toString() {
                    return FollowerChecker.this + "::handleWakeUp";
                }
            }, FollowersChecker.this.followerCheckInterval, EsExecutors.DIRECT_EXECUTOR_SERVICE);
        }

        public String toString() {
            return "FollowerChecker{discoveryNode=" + this.discoveryNode + ", failureCountSinceLastSuccess=" + this.failureCountSinceLastSuccess + ", timeoutCountSinceLastSuccess=" + this.timeoutCountSinceLastSuccess + ", [" + FOLLOWER_CHECK_RETRY_COUNT_SETTING.getKey() + "]=" + FollowersChecker.this.followerCheckRetryCount + "}";
        }
    }
}

