/*
 * Decompiled with CFR 0.152.
 */
package org.logstash.common.io;

import com.google.common.annotations.VisibleForTesting;
import java.io.Closeable;
import java.io.IOException;
import java.nio.channels.FileLock;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.TemporalAmount;
import java.util.Locale;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.logstash.DLQEntry;
import org.logstash.Event;
import org.logstash.FieldReference;
import org.logstash.FileLockFactory;
import org.logstash.Timestamp;
import org.logstash.common.io.DeadLetterQueueUtils;
import org.logstash.common.io.QueueStorageType;
import org.logstash.common.io.RecordIOReader;
import org.logstash.common.io.RecordIOWriter;

public final class DeadLetterQueueWriter
implements Closeable {
    @VisibleForTesting
    static final String SEGMENT_FILE_PATTERN = "%d.log";
    private static final Logger logger = LogManager.getLogger(DeadLetterQueueWriter.class);
    private static final String TEMP_FILE_PATTERN = "%d.log.tmp";
    private static final String LOCK_FILE = ".lock";
    private static final FieldReference DEAD_LETTER_QUEUE_METADATA_KEY = FieldReference.from(String.format("%s[dead_letter_queue]", "[@metadata]"));
    private final ReentrantLock lock = new ReentrantLock();
    private final long maxSegmentSize;
    private final long maxQueueSize;
    private final QueueStorageType storageType;
    private final AtomicLong currentQueueSize;
    private final Path queuePath;
    private final FileLock fileLock;
    private volatile RecordIOWriter currentWriter;
    private volatile int currentSegmentIndex;
    private volatile Timestamp lastEntryTimestamp;
    private final Duration flushInterval;
    private Instant lastWrite;
    private final AtomicBoolean open = new AtomicBoolean(true);
    private ScheduledExecutorService flushScheduler;
    private final LongAdder droppedEvents = new LongAdder();
    private final LongAdder expiredEvents = new LongAdder();
    private volatile String lastError = "no errors";
    private final Clock clock;
    private volatile Optional<Timestamp> oldestSegmentTimestamp;
    private volatile Optional<Path> oldestSegmentPath = Optional.empty();
    private final TemporalAmount retentionTime;
    private final SchedulerService flusherService;

    public static Builder newBuilder(Path queuePath, long maxSegmentSize, long maxQueueSize, Duration flushInterval) {
        return new Builder(queuePath, maxSegmentSize, maxQueueSize, flushInterval);
    }

    @VisibleForTesting
    static Builder newBuilderWithoutFlusher(Path queuePath, long maxSegmentSize, long maxQueueSize) {
        return new Builder(queuePath, maxSegmentSize, maxQueueSize, Duration.ZERO, false);
    }

    private DeadLetterQueueWriter(Path queuePath, long maxSegmentSize, long maxQueueSize, Duration flushInterval, QueueStorageType storageType, Duration retentionTime, Clock clock, SchedulerService flusherService) throws IOException {
        this.clock = clock;
        this.fileLock = FileLockFactory.obtainLock(queuePath, LOCK_FILE);
        this.queuePath = queuePath;
        this.maxSegmentSize = maxSegmentSize;
        this.maxQueueSize = maxQueueSize;
        this.storageType = storageType;
        this.flushInterval = flushInterval;
        this.currentQueueSize = new AtomicLong(this.computeQueueSize());
        this.retentionTime = retentionTime;
        this.cleanupTempFiles();
        this.updateOldestSegmentReference();
        this.currentSegmentIndex = DeadLetterQueueUtils.listSegmentPaths(queuePath).map(s -> s.getFileName().toString().split("\\.")[0]).mapToInt(Integer::parseInt).max().orElse(0);
        this.nextWriter();
        this.lastEntryTimestamp = Timestamp.now();
        this.flusherService = flusherService;
        this.flusherService.repeatedAction(this::scheduledFlushCheck);
    }

    public boolean isOpen() {
        return this.open.get();
    }

    public Path getPath() {
        return this.queuePath;
    }

    public long getCurrentQueueSize() {
        return this.currentQueueSize.longValue();
    }

    public String getStoragePolicy() {
        return this.storageType.name().toLowerCase(Locale.ROOT);
    }

    public long getDroppedEvents() {
        return this.droppedEvents.longValue();
    }

    public long getExpiredEvents() {
        return this.expiredEvents.longValue();
    }

    public String getLastError() {
        return this.lastError;
    }

    public void writeEntry(Event event, String pluginName, String pluginId, String reason) throws IOException {
        this.writeEntry(new DLQEntry(event, pluginName, pluginId, reason));
    }

    @Override
    public void close() {
        if (this.open.compareAndSet(true, false)) {
            try {
                this.finalizeSegment(FinalizeWhen.ALWAYS, SealReason.DLQ_CLOSE);
            }
            catch (Exception e) {
                logger.warn("Unable to close dlq writer, ignoring", (Throwable)e);
            }
            try {
                this.releaseFileLock();
            }
            catch (Exception e) {
                logger.warn("Unable to release fileLock, ignoring", (Throwable)e);
            }
            try {
                if (this.flushScheduler != null) {
                    this.flushScheduler.shutdown();
                }
            }
            catch (Exception e) {
                logger.warn("Unable shutdown flush scheduler, ignoring", (Throwable)e);
            }
        }
    }

    @VisibleForTesting
    void writeEntry(DLQEntry entry) throws IOException {
        this.lock.lock();
        try {
            Timestamp entryTimestamp = Timestamp.now();
            if (entryTimestamp.compareTo(this.lastEntryTimestamp) < 0) {
                entryTimestamp = this.lastEntryTimestamp;
            }
            this.innerWriteEntry(entry);
            this.lastEntryTimestamp = entryTimestamp;
        }
        finally {
            this.lock.unlock();
        }
    }

    private void innerWriteEntry(DLQEntry entry) throws IOException {
        Event event = entry.getEvent();
        if (DeadLetterQueueWriter.alreadyProcessed(event)) {
            logger.warn("Event previously submitted to dead letter queue. Skipping...");
            return;
        }
        byte[] record = entry.serialize();
        int eventPayloadSize = 13 + record.length;
        this.executeAgeRetentionPolicy();
        boolean skipWrite = this.executeStoragePolicy(eventPayloadSize);
        if (skipWrite) {
            return;
        }
        if (this.exceedSegmentSize(eventPayloadSize)) {
            this.finalizeSegment(FinalizeWhen.ALWAYS, SealReason.SEGMENT_FULL);
        }
        long writtenBytes = this.currentWriter.writeEvent(record);
        this.currentQueueSize.getAndAdd(writtenBytes);
        this.lastWrite = Instant.now();
    }

    private boolean exceedSegmentSize(int eventPayloadSize) throws IOException {
        return this.currentWriter.getPosition() + (long)eventPayloadSize > this.maxSegmentSize;
    }

    private void executeAgeRetentionPolicy() {
        if (this.isOldestSegmentExpired()) {
            try {
                this.deleteExpiredSegments();
            }
            catch (IOException ex) {
                logger.error("Can't remove some DLQ files while cleaning expired segments", (Throwable)ex);
            }
        }
    }

    private boolean executeStoragePolicy(int eventPayloadSize) {
        if (!this.exceedMaxQueueSize(eventPayloadSize)) {
            return false;
        }
        try {
            this.currentQueueSize.set(this.computeQueueSize());
        }
        catch (IOException ex) {
            logger.warn("Unable to determine DLQ size, skipping storage policy check", (Throwable)ex);
            return false;
        }
        if (!this.exceedMaxQueueSize(eventPayloadSize)) {
            return false;
        }
        if (this.storageType == QueueStorageType.DROP_NEWER) {
            this.lastError = String.format("Cannot write event to DLQ(path: %s): reached maxQueueSize of %d", this.queuePath, this.maxQueueSize);
            logger.error(this.lastError);
            this.droppedEvents.add(1L);
            return true;
        }
        try {
            do {
                this.dropTailSegment();
            } while (this.exceedMaxQueueSize(eventPayloadSize));
        }
        catch (IOException ex) {
            logger.error("Can't remove some DLQ files while removing older segments", (Throwable)ex);
        }
        return false;
    }

    private boolean exceedMaxQueueSize(int eventPayloadSize) {
        return this.currentQueueSize.longValue() + (long)eventPayloadSize > this.maxQueueSize;
    }

    private boolean isOldestSegmentExpired() {
        if (this.retentionTime == null) {
            return false;
        }
        Instant now = this.clock.instant();
        return this.oldestSegmentTimestamp.map(t -> t.toInstant().isBefore(now.minus(this.retentionTime))).orElse(false);
    }

    private void deleteExpiredSegments() throws IOException {
        boolean cleanNextSegment;
        do {
            if (this.oldestSegmentPath.isPresent()) {
                Path beheadedSegment = this.oldestSegmentPath.get();
                this.expiredEvents.add(this.deleteTailSegment(beheadedSegment, "age retention policy"));
            }
            this.updateOldestSegmentReference();
        } while (cleanNextSegment = this.isOldestSegmentExpired());
        this.currentQueueSize.set(this.computeQueueSize());
    }

    private long deleteTailSegment(Path segment, String motivation) throws IOException {
        try {
            long eventsInSegment = DeadLetterQueueUtils.countEventsInSegment(segment);
            Files.delete(segment);
            logger.debug("Removed segment file {} due to {}", (Object)segment, (Object)motivation);
            return eventsInSegment;
        }
        catch (NoSuchFileException nsfex) {
            logger.debug("File not found {}, maybe removed by the reader pipeline", (Object)segment);
            return 0L;
        }
    }

    void updateOldestSegmentReference() throws IOException {
        Optional<Timestamp> foundTimestamp;
        boolean previousPathEqualsToCurrent;
        Optional<Path> previousOldestSegmentPath = this.oldestSegmentPath;
        this.oldestSegmentPath = DeadLetterQueueUtils.listSegmentPathsSortedBySegmentId(this.queuePath).filter(p -> p.toFile().length() > 1L).findFirst();
        if (!this.oldestSegmentPath.isPresent()) {
            this.oldestSegmentTimestamp = Optional.empty();
            return;
        }
        boolean bl = previousPathEqualsToCurrent = previousOldestSegmentPath.isPresent() && previousOldestSegmentPath.get().equals(this.oldestSegmentPath.get());
        if (!previousPathEqualsToCurrent) {
            logger.debug("Oldest segment is {}", (Object)this.oldestSegmentPath.get());
        }
        if (!(foundTimestamp = DeadLetterQueueWriter.readTimestampOfLastEventInSegment(this.oldestSegmentPath.get())).isPresent()) {
            this.oldestSegmentPath = Optional.empty();
        }
        this.oldestSegmentTimestamp = foundTimestamp;
    }

    Optional<Path> getOldestSegmentPath() {
        return this.oldestSegmentPath;
    }

    static Optional<Timestamp> readTimestampOfLastEventInSegment(Path segmentPath) throws IOException {
        byte[] eventBytes = null;
        try (RecordIOReader recordReader = new RecordIOReader(segmentPath);){
            for (int blockId = (int)Math.ceil((double)(Files.size(segmentPath) - 1L) / 32768.0) - 1; eventBytes == null && blockId >= 0; --blockId) {
                recordReader.seekToBlock(blockId);
                eventBytes = recordReader.readEvent();
            }
        }
        catch (NoSuchFileException nsfex) {
            return Optional.empty();
        }
        if (eventBytes == null) {
            logger.warn("Cannot find a complete event into the segment file [{}], this is a DLQ segment corruption", (Object)segmentPath);
            return Optional.empty();
        }
        return Optional.of(DLQEntry.deserialize(eventBytes).getEntryTime());
    }

    void dropTailSegment() throws IOException {
        Optional<Path> oldestSegment = DeadLetterQueueUtils.listSegmentPathsSortedBySegmentId(this.queuePath).findFirst();
        if (oldestSegment.isPresent()) {
            Path beheadedSegment = oldestSegment.get();
            this.deleteTailSegment(beheadedSegment, "dead letter queue size exceeded dead_letter_queue.max_bytes size(" + this.maxQueueSize + ")");
        } else {
            logger.info("Queue size {} exceeded, but no complete DLQ segments found", (Object)this.maxQueueSize);
        }
        this.currentQueueSize.set(this.computeQueueSize());
    }

    private static boolean alreadyProcessed(Event event) {
        return event.includes(DEAD_LETTER_QUEUE_METADATA_KEY);
    }

    private void scheduledFlushCheck() {
        logger.trace("Running scheduled check");
        this.lock.lock();
        try {
            this.finalizeSegment(FinalizeWhen.ONLY_IF_STALE, SealReason.SCHEDULED_FLUSH);
            this.updateOldestSegmentReference();
            this.executeAgeRetentionPolicy();
        }
        catch (Exception e) {
            logger.warn("Unable to finalize segment", (Throwable)e);
        }
        finally {
            this.lock.unlock();
        }
    }

    private boolean isCurrentWriterStale() {
        return this.currentWriter.isStale(this.flushInterval);
    }

    private void finalizeSegment(FinalizeWhen finalizeWhen, SealReason sealReason) throws IOException {
        this.lock.lock();
        try {
            if (!this.isCurrentWriterStale() && finalizeWhen == FinalizeWhen.ONLY_IF_STALE) {
                return;
            }
            if (this.currentWriter != null) {
                if (this.currentWriter.hasWritten()) {
                    this.currentWriter.close();
                    this.sealSegment(this.currentSegmentIndex, sealReason);
                }
                this.updateOldestSegmentReference();
                this.executeAgeRetentionPolicy();
                if (this.isOpen() && this.currentWriter.hasWritten()) {
                    this.nextWriter();
                }
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void sealSegment(int segmentIndex, SealReason motivation) throws IOException {
        Files.move(this.queuePath.resolve(String.format(TEMP_FILE_PATTERN, segmentIndex)), this.queuePath.resolve(String.format(SEGMENT_FILE_PATTERN, segmentIndex)), StandardCopyOption.ATOMIC_MOVE);
        logger.debug("Sealed segment with index {} because {}", (Object)segmentIndex, (Object)motivation);
    }

    private void createFlushScheduler() {
        this.flushScheduler = Executors.newScheduledThreadPool(1, r -> {
            Thread t = new Thread(r);
            t.setDaemon(true);
            t.setName("dlq-flush-check");
            return t;
        });
        this.flushScheduler.scheduleAtFixedRate(this::scheduledFlushCheck, 1L, 1L, TimeUnit.SECONDS);
    }

    private long computeQueueSize() throws IOException {
        return DeadLetterQueueUtils.listSegmentPaths(this.queuePath).mapToLong(DeadLetterQueueWriter::safeFileSize).sum();
    }

    private static long safeFileSize(Path p) {
        try {
            return Files.size(p);
        }
        catch (IOException e) {
            return 0L;
        }
    }

    private void releaseFileLock() {
        try {
            FileLockFactory.releaseLock(this.fileLock);
        }
        catch (IOException e) {
            logger.debug("Unable to release fileLock", (Throwable)e);
        }
        try {
            Files.deleteIfExists(this.queuePath.resolve(LOCK_FILE));
        }
        catch (IOException e) {
            logger.debug("Unable to delete fileLock file", (Throwable)e);
        }
    }

    private void nextWriter() throws IOException {
        Path nextSegmentPath = this.queuePath.resolve(String.format(TEMP_FILE_PATTERN, ++this.currentSegmentIndex));
        this.currentWriter = new RecordIOWriter(nextSegmentPath);
        this.currentQueueSize.incrementAndGet();
        logger.debug("Created new head segment {}", (Object)nextSegmentPath);
    }

    private void cleanupTempFiles() throws IOException {
        DeadLetterQueueUtils.listFiles(this.queuePath, ".log.tmp").forEach(this::cleanupTempFile);
    }

    private void cleanupTempFile(Path tempFile) {
        String segmentName = tempFile.getFileName().toString().split("\\.")[0];
        Path segmentFile = this.queuePath.resolve(String.format("%s.log", segmentName));
        try {
            if (Files.exists(segmentFile, new LinkOption[0])) {
                Files.delete(tempFile);
                logger.debug("Deleted temporary file {}", (Object)tempFile);
            } else {
                RecordIOReader.SegmentStatus segmentStatus = RecordIOReader.getSegmentStatus(tempFile);
                switch (segmentStatus) {
                    case VALID: {
                        logger.debug("Moving temp file {} to segment file {}", (Object)tempFile, (Object)segmentFile);
                        Files.move(tempFile, segmentFile, StandardCopyOption.ATOMIC_MOVE);
                        break;
                    }
                    case EMPTY: {
                        this.deleteTemporaryEmptyFile(tempFile, segmentName);
                        break;
                    }
                    case INVALID: {
                        Path errorFile = this.queuePath.resolve(String.format("%s.err", segmentName));
                        logger.warn("Segment file {} is in an error state, saving as {}", (Object)segmentFile, (Object)errorFile);
                        Files.move(tempFile, errorFile, StandardCopyOption.ATOMIC_MOVE);
                        break;
                    }
                    default: {
                        throw new IllegalStateException("Unexpected value: " + RecordIOReader.getSegmentStatus(tempFile));
                    }
                }
            }
        }
        catch (IOException e) {
            throw new IllegalStateException("Unable to clean up temp file: " + tempFile, e);
        }
    }

    private void deleteTemporaryEmptyFile(Path tempFile, String segmentName) throws IOException {
        Path deleteTarget;
        if (DeadLetterQueueWriter.isWindows()) {
            Path deletedFile = this.queuePath.resolve(String.format("%s.del", segmentName));
            logger.debug("Moving temp file {} to {}", (Object)tempFile, (Object)deletedFile);
            deleteTarget = deletedFile;
            Files.move(tempFile, deletedFile, StandardCopyOption.ATOMIC_MOVE);
        } else {
            deleteTarget = tempFile;
        }
        Files.delete(deleteTarget);
        logger.debug("Deleted temporary empty file {}", (Object)deleteTarget);
    }

    private static boolean isWindows() {
        return System.getProperty("os.name").startsWith("Windows");
    }

    public static final class Builder {
        private final Path queuePath;
        private final long maxSegmentSize;
        private final long maxQueueSize;
        private final Duration flushInterval;
        private boolean startScheduledFlusher;
        private QueueStorageType storageType = QueueStorageType.DROP_NEWER;
        private Duration retentionTime = null;
        private Clock clock = Clock.systemDefaultZone();
        private SchedulerService customSchedulerService = null;

        private Builder(Path queuePath, long maxSegmentSize, long maxQueueSize, Duration flushInterval) {
            this(queuePath, maxSegmentSize, maxQueueSize, flushInterval, true);
        }

        private Builder(Path queuePath, long maxSegmentSize, long maxQueueSize, Duration flushInterval, boolean startScheduledFlusher) {
            this.queuePath = queuePath;
            this.maxSegmentSize = maxSegmentSize;
            this.maxQueueSize = maxQueueSize;
            this.flushInterval = flushInterval;
            this.startScheduledFlusher = startScheduledFlusher;
        }

        public Builder storageType(QueueStorageType storageType) {
            this.storageType = storageType;
            return this;
        }

        public Builder retentionTime(Duration retentionTime) {
            this.retentionTime = retentionTime;
            return this;
        }

        @VisibleForTesting
        Builder clock(Clock clock) {
            this.clock = clock;
            return this;
        }

        @VisibleForTesting
        Builder flusherService(SchedulerService service) {
            this.customSchedulerService = service;
            return this;
        }

        public DeadLetterQueueWriter build() throws IOException {
            if (this.customSchedulerService != null && this.startScheduledFlusher) {
                throw new IllegalArgumentException("Both default scheduler and custom scheduler were defined, ");
            }
            SchedulerService schedulerService = this.customSchedulerService != null ? this.customSchedulerService : (this.startScheduledFlusher ? new FixedRateScheduler() : new NoopScheduler());
            return new DeadLetterQueueWriter(this.queuePath, this.maxSegmentSize, this.maxQueueSize, this.flushInterval, this.storageType, this.retentionTime, this.clock, schedulerService);
        }
    }

    static interface SchedulerService {
        public void repeatedAction(Runnable var1);
    }

    private static enum FinalizeWhen {
        ALWAYS,
        ONLY_IF_STALE;

    }

    private static enum SealReason {
        DLQ_CLOSE("Dead letter queue is closing"),
        SCHEDULED_FLUSH("the segment has expired 'flush_interval'"),
        SEGMENT_FULL("the segment has reached its maximum size");

        final String motivation;

        private SealReason(String motivation) {
            this.motivation = motivation;
        }

        public String toString() {
            return this.motivation;
        }
    }

    private static class NoopScheduler
    implements SchedulerService {
        private NoopScheduler() {
        }

        @Override
        public void repeatedAction(Runnable action) {
        }
    }

    private static class FixedRateScheduler
    implements SchedulerService {
        private final ScheduledExecutorService scheduledExecutor = Executors.newScheduledThreadPool(1, r -> {
            Thread t = new Thread(r);
            t.setDaemon(true);
            t.setName("dlq-flush-check");
            return t;
        });

        FixedRateScheduler() {
        }

        @Override
        public void repeatedAction(Runnable action) {
            this.scheduledExecutor.scheduleAtFixedRate(action, 1L, 1L, TimeUnit.SECONDS);
        }
    }
}

