/*
 * Decompiled with CFR 0.152.
 */
package org.logstash.common.io;

import com.google.common.annotations.VisibleForTesting;
import java.io.Closeable;
import java.io.IOException;
import java.nio.channels.FileLock;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.logstash.DLQEntry;
import org.logstash.Event;
import org.logstash.FieldReference;
import org.logstash.FileLockFactory;
import org.logstash.Timestamp;
import org.logstash.common.io.RecordIOReader;
import org.logstash.common.io.RecordIOWriter;

public final class DeadLetterQueueWriter
implements Closeable {
    @VisibleForTesting
    static final String SEGMENT_FILE_PATTERN = "%d.log";
    private static final Logger logger = LogManager.getLogger(DeadLetterQueueWriter.class);
    private static final String TEMP_FILE_PATTERN = "%d.log.tmp";
    private static final String LOCK_FILE = ".lock";
    private final ReentrantLock lock = new ReentrantLock();
    private static final FieldReference DEAD_LETTER_QUEUE_METADATA_KEY = FieldReference.from(String.format("%s[dead_letter_queue]", "[@metadata]"));
    private final long maxSegmentSize;
    private final long maxQueueSize;
    private LongAdder currentQueueSize;
    private final Path queuePath;
    private final FileLock fileLock;
    private volatile RecordIOWriter currentWriter;
    private int currentSegmentIndex;
    private Timestamp lastEntryTimestamp;
    private Duration flushInterval;
    private Instant lastWrite;
    private final AtomicBoolean open = new AtomicBoolean(true);
    private ScheduledExecutorService flushScheduler;

    public DeadLetterQueueWriter(Path queuePath, long maxSegmentSize, long maxQueueSize, Duration flushInterval) throws IOException {
        this.fileLock = FileLockFactory.obtainLock(queuePath, LOCK_FILE);
        this.queuePath = queuePath;
        this.maxSegmentSize = maxSegmentSize;
        this.maxQueueSize = maxQueueSize;
        this.flushInterval = flushInterval;
        this.currentQueueSize = new LongAdder();
        this.currentQueueSize.add(this.getStartupQueueSize());
        this.cleanupTempFiles();
        this.currentSegmentIndex = DeadLetterQueueWriter.getSegmentPaths(queuePath).map(s -> s.getFileName().toString().split("\\.")[0]).mapToInt(Integer::parseInt).max().orElse(0);
        this.nextWriter();
        this.lastEntryTimestamp = Timestamp.now();
        this.createFlushScheduler();
    }

    public boolean isOpen() {
        return this.open.get();
    }

    public Path getPath() {
        return this.queuePath;
    }

    public long getCurrentQueueSize() {
        return this.currentQueueSize.longValue();
    }

    public void writeEntry(Event event, String pluginName, String pluginId, String reason) throws IOException {
        this.writeEntry(new DLQEntry(event, pluginName, pluginId, reason));
    }

    @Override
    public void close() {
        if (this.open.compareAndSet(true, false)) {
            try {
                this.finalizeSegment(FinalizeWhen.ALWAYS);
            }
            catch (Exception e) {
                logger.warn("Unable to close dlq writer, ignoring", (Throwable)e);
            }
            try {
                this.releaseFileLock();
            }
            catch (Exception e) {
                logger.warn("Unable to release fileLock, ignoring", (Throwable)e);
            }
            try {
                this.flushScheduler.shutdown();
            }
            catch (Exception e) {
                logger.warn("Unable shutdown flush scheduler, ignoring", (Throwable)e);
            }
        }
    }

    static Stream<Path> getSegmentPaths(Path path) throws IOException {
        return DeadLetterQueueWriter.listFiles(path, ".log");
    }

    @VisibleForTesting
    void writeEntry(DLQEntry entry) throws IOException {
        this.lock.lock();
        try {
            Timestamp entryTimestamp = Timestamp.now();
            if (entryTimestamp.compareTo(this.lastEntryTimestamp) < 0) {
                entryTimestamp = this.lastEntryTimestamp;
            }
            this.innerWriteEntry(entry);
            this.lastEntryTimestamp = entryTimestamp;
        }
        finally {
            this.lock.unlock();
        }
    }

    private void innerWriteEntry(DLQEntry entry) throws IOException {
        Event event = entry.getEvent();
        if (DeadLetterQueueWriter.alreadyProcessed(event)) {
            logger.warn("Event previously submitted to dead letter queue. Skipping...");
            return;
        }
        byte[] record = entry.serialize();
        int eventPayloadSize = 13 + record.length;
        if (this.currentQueueSize.longValue() + (long)eventPayloadSize > this.maxQueueSize) {
            logger.error("cannot write event to DLQ(path: " + this.queuePath + "): reached maxQueueSize of " + this.maxQueueSize);
            return;
        }
        if (this.currentWriter.getPosition() + (long)eventPayloadSize > this.maxSegmentSize) {
            this.finalizeSegment(FinalizeWhen.ALWAYS);
        }
        this.currentQueueSize.add(this.currentWriter.writeEvent(record));
        this.lastWrite = Instant.now();
    }

    private static boolean alreadyProcessed(Event event) {
        return event.includes(DEAD_LETTER_QUEUE_METADATA_KEY);
    }

    private void flushCheck() {
        try {
            this.finalizeSegment(FinalizeWhen.ONLY_IF_STALE);
        }
        catch (Exception e) {
            logger.warn("unable to finalize segment", (Throwable)e);
        }
    }

    private boolean isCurrentWriterStale() {
        return this.currentWriter.isStale(this.flushInterval);
    }

    private void finalizeSegment(FinalizeWhen finalizeWhen) throws IOException {
        this.lock.lock();
        try {
            if (!this.isCurrentWriterStale() && finalizeWhen == FinalizeWhen.ONLY_IF_STALE) {
                return;
            }
            if (this.currentWriter != null && this.currentWriter.hasWritten()) {
                this.currentWriter.close();
                Files.move(this.queuePath.resolve(String.format(TEMP_FILE_PATTERN, this.currentSegmentIndex)), this.queuePath.resolve(String.format(SEGMENT_FILE_PATTERN, this.currentSegmentIndex)), StandardCopyOption.ATOMIC_MOVE);
                if (this.isOpen()) {
                    this.nextWriter();
                }
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void createFlushScheduler() {
        this.flushScheduler = Executors.newScheduledThreadPool(1, r -> {
            Thread t = new Thread(r);
            t.setDaemon(true);
            t.setName("dlq-flush-check");
            return t;
        });
        this.flushScheduler.scheduleAtFixedRate(this::flushCheck, 1L, 1L, TimeUnit.SECONDS);
    }

    private long getStartupQueueSize() throws IOException {
        return DeadLetterQueueWriter.getSegmentPaths(this.queuePath).mapToLong(p -> {
            try {
                return Files.size(p);
            }
            catch (IOException e) {
                throw new IllegalStateException(e);
            }
        }).sum();
    }

    private void releaseFileLock() {
        try {
            FileLockFactory.releaseLock(this.fileLock);
        }
        catch (IOException e) {
            logger.debug("Unable to release fileLock", (Throwable)e);
        }
        try {
            Files.deleteIfExists(this.queuePath.resolve(LOCK_FILE));
        }
        catch (IOException e) {
            logger.debug("Unable to delete fileLock file", (Throwable)e);
        }
    }

    private void nextWriter() throws IOException {
        this.currentWriter = new RecordIOWriter(this.queuePath.resolve(String.format(TEMP_FILE_PATTERN, ++this.currentSegmentIndex)));
        this.currentQueueSize.increment();
    }

    private void cleanupTempFiles() throws IOException {
        DeadLetterQueueWriter.listFiles(this.queuePath, ".log.tmp").forEach(this::cleanupTempFile);
    }

    private static Stream<Path> listFiles(Path path, String suffix) throws IOException {
        try (Stream<Path> files = Files.list(path);){
            Stream<Path> stream = files.filter(p -> p.toString().endsWith(suffix)).collect(Collectors.toList()).stream();
            return stream;
        }
    }

    private void cleanupTempFile(Path tempFile) {
        String segmentName = tempFile.getFileName().toString().split("\\.")[0];
        Path segmentFile = this.queuePath.resolve(String.format("%s.log", segmentName));
        try {
            if (Files.exists(segmentFile, new LinkOption[0])) {
                Files.delete(tempFile);
            } else {
                RecordIOReader.SegmentStatus segmentStatus = RecordIOReader.getSegmentStatus(tempFile);
                switch (segmentStatus) {
                    case VALID: {
                        logger.debug("Moving temp file {} to segment file {}", (Object)tempFile, (Object)segmentFile);
                        Files.move(tempFile, segmentFile, StandardCopyOption.ATOMIC_MOVE);
                        break;
                    }
                    case EMPTY: {
                        this.deleteTemporaryFile(tempFile, segmentName);
                        break;
                    }
                    case INVALID: {
                        Path errorFile = this.queuePath.resolve(String.format("%s.err", segmentName));
                        logger.warn("Segment file {} is in an error state, saving as {}", (Object)segmentFile, (Object)errorFile);
                        Files.move(tempFile, errorFile, StandardCopyOption.ATOMIC_MOVE);
                        break;
                    }
                    default: {
                        throw new IllegalStateException("Unexpected value: " + (Object)((Object)RecordIOReader.getSegmentStatus(tempFile)));
                    }
                }
            }
        }
        catch (IOException e) {
            throw new IllegalStateException("Unable to clean up temp file: " + tempFile, e);
        }
    }

    private void deleteTemporaryFile(Path tempFile, String segmentName) throws IOException {
        Path deleteTarget;
        if (DeadLetterQueueWriter.isWindows()) {
            Path deletedFile = this.queuePath.resolve(String.format("%s.del", segmentName));
            logger.debug("Moving temp file {} to {}", (Object)tempFile, (Object)deletedFile);
            deleteTarget = deletedFile;
            Files.move(tempFile, deletedFile, StandardCopyOption.ATOMIC_MOVE);
        } else {
            deleteTarget = tempFile;
        }
        Files.delete(deleteTarget);
    }

    private static boolean isWindows() {
        return System.getProperty("os.name").startsWith("Windows");
    }

    private static enum FinalizeWhen {
        ALWAYS,
        ONLY_IF_STALE;

    }
}

