/*
 * Decompiled with CFR 0.152.
 */
package org.logstash.common.io;

import java.io.Closeable;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.concurrent.atomic.LongAdder;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.ReadableInstant;
import org.logstash.DLQEntry;
import org.logstash.Event;
import org.logstash.Timestamp;
import org.logstash.common.io.RecordIOWriter;

public final class DeadLetterQueueWriter
implements Closeable {
    private static final Logger logger = LogManager.getLogger(DeadLetterQueueWriter.class);
    private static final long MAX_SEGMENT_SIZE_BYTES = 0xA00000L;
    static final String SEGMENT_FILE_PATTERN = "%d.log";
    static final String LOCK_FILE = ".lock";
    public static final String DEAD_LETTER_QUEUE_METADATA_KEY = "dead_letter_queue";
    private final long maxSegmentSize;
    private final long maxQueueSize;
    private LongAdder currentQueueSize;
    private final Path queuePath;
    private final FileLock lock;
    private RecordIOWriter currentWriter;
    private int currentSegmentIndex;
    private Timestamp lastEntryTimestamp;
    private boolean open;

    public DeadLetterQueueWriter(Path queuePath, long maxSegmentSize, long maxQueueSize) throws IOException {
        Files.createDirectories(queuePath, new FileAttribute[0]);
        Path lockFilePath = queuePath.resolve(LOCK_FILE);
        boolean isNewlyCreated = lockFilePath.toFile().createNewFile();
        FileChannel channel = FileChannel.open(lockFilePath, StandardOpenOption.WRITE);
        try {
            this.lock = channel.lock();
        }
        catch (OverlappingFileLockException e) {
            if (isNewlyCreated) {
                logger.warn("Previous Dead Letter Queue Writer was not closed safely.");
            }
            throw new RuntimeException("uh oh, someone else is writing to this dead-letter queue");
        }
        this.queuePath = queuePath;
        this.maxSegmentSize = maxSegmentSize;
        this.maxQueueSize = maxQueueSize;
        this.currentQueueSize = new LongAdder();
        this.currentQueueSize.add(this.getStartupQueueSize());
        this.currentSegmentIndex = DeadLetterQueueWriter.getSegmentPaths(queuePath).map(s -> s.getFileName().toString().split("\\.")[0]).mapToInt(Integer::parseInt).max().orElse(0);
        this.currentWriter = this.nextWriter();
        this.lastEntryTimestamp = Timestamp.now();
        this.open = true;
    }

    public DeadLetterQueueWriter(String queuePath) throws IOException {
        this(Paths.get(queuePath, new String[0]), 0xA00000L, Long.MAX_VALUE);
    }

    private long getStartupQueueSize() throws IOException {
        return DeadLetterQueueWriter.getSegmentPaths(this.queuePath).mapToLong(p -> {
            try {
                return Files.size(p);
            }
            catch (IOException e) {
                return 0L;
            }
        }).sum();
    }

    private RecordIOWriter nextWriter() throws IOException {
        RecordIOWriter recordIOWriter = new RecordIOWriter(this.queuePath.resolve(String.format(SEGMENT_FILE_PATTERN, ++this.currentSegmentIndex)));
        this.currentQueueSize.increment();
        return recordIOWriter;
    }

    static Stream<Path> getSegmentPaths(Path path) throws IOException {
        return Files.list(path).filter(p -> p.toString().endsWith(".log"));
    }

    public synchronized void writeEntry(DLQEntry entry) throws IOException {
        this.innerWriteEntry(entry);
    }

    public synchronized void writeEntry(Event event, String pluginName, String pluginId, String reason) throws IOException {
        Timestamp entryTimestamp = Timestamp.now();
        if (entryTimestamp.getTime().isBefore((ReadableInstant)this.lastEntryTimestamp.getTime())) {
            entryTimestamp = this.lastEntryTimestamp;
        }
        DLQEntry entry = new DLQEntry(event, pluginName, pluginId, reason);
        this.innerWriteEntry(entry);
        this.lastEntryTimestamp = entryTimestamp;
    }

    private void innerWriteEntry(DLQEntry entry) throws IOException {
        Event event = entry.getEvent();
        if (this.alreadyProcessed(event)) {
            logger.warn("Event previously submitted to dead letter queue. Skipping...");
            return;
        }
        byte[] record = entry.serialize();
        int eventPayloadSize = 13 + record.length;
        if (this.currentQueueSize.longValue() + (long)eventPayloadSize > this.maxQueueSize) {
            logger.error("cannot write event to DLQ: reached maxQueueSize of " + this.maxQueueSize);
            return;
        }
        if (this.currentWriter.getPosition() + (long)eventPayloadSize > this.maxSegmentSize) {
            this.currentWriter.close();
            this.currentWriter = this.nextWriter();
        }
        this.currentQueueSize.add(this.currentWriter.writeEvent(record));
    }

    private boolean alreadyProcessed(Event event) {
        return event.getMetadata() != null && event.getMetadata().containsKey(DEAD_LETTER_QUEUE_METADATA_KEY);
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.currentWriter != null) {
            try {
                this.currentWriter.close();
                this.open = false;
            }
            catch (Exception e) {
                logger.debug("Unable to close dlq writer", (Throwable)e);
            }
        }
        this.releaseLock();
    }

    private void releaseLock() {
        if (this.lock != null) {
            try {
                this.lock.release();
                if (this.lock.channel() != null && this.lock.channel().isOpen()) {
                    this.lock.channel().close();
                }
            }
            catch (Exception e) {
                logger.debug("Unable to close lock channel", (Throwable)e);
            }
            try {
                Files.deleteIfExists(this.queuePath.resolve(LOCK_FILE));
            }
            catch (IOException e) {
                logger.debug("Unable to delete lock file", (Throwable)e);
            }
        }
    }

    public boolean isOpen() {
        return this.open;
    }

    public Path getPath() {
        return this.queuePath;
    }

    public long getCurrentQueueSize() {
        return this.currentQueueSize.longValue();
    }
}

