"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.serializeTh = exports.serializeTraceState = exports.parseOtelTraceState = exports.INVALID_TRACE_STATE = void 0;
const util_1 = require("./util");
exports.INVALID_TRACE_STATE = Object.freeze({
    randomValue: util_1.INVALID_RANDOM_VALUE,
    threshold: util_1.INVALID_THRESHOLD,
});
const TRACE_STATE_SIZE_LIMIT = 256;
const MAX_VALUE_LENGTH = 14; // 56 bits, 4 bits per hex digit
function parseOtelTraceState(traceState) {
    const ot = traceState?.get('ot');
    if (!ot || ot.length > TRACE_STATE_SIZE_LIMIT) {
        return exports.INVALID_TRACE_STATE;
    }
    let threshold = util_1.INVALID_THRESHOLD;
    let randomValue = util_1.INVALID_RANDOM_VALUE;
    // Parse based on https://opentelemetry.io/docs/specs/otel/trace/tracestate-handling/
    const members = ot.split(';');
    let rest;
    for (const member of members) {
        if (member.startsWith('th:')) {
            threshold = parseTh(member.slice('th:'.length), util_1.INVALID_THRESHOLD);
            continue;
        }
        if (member.startsWith('rv:')) {
            randomValue = parseRv(member.slice('rv:'.length), util_1.INVALID_RANDOM_VALUE);
            continue;
        }
        if (!rest) {
            rest = [];
        }
        rest.push(member);
    }
    return {
        randomValue,
        threshold,
        rest,
    };
}
exports.parseOtelTraceState = parseOtelTraceState;
function serializeTraceState(otTraceState) {
    if (!(0, util_1.isValidThreshold)(otTraceState.threshold) &&
        !(0, util_1.isValidRandomValue)(otTraceState.randomValue) &&
        !otTraceState.rest) {
        return '';
    }
    const parts = [];
    if ((0, util_1.isValidThreshold)(otTraceState.threshold) &&
        otTraceState.threshold !== util_1.MAX_THRESHOLD) {
        parts.push(`th:${serializeTh(otTraceState.threshold)}`);
    }
    if ((0, util_1.isValidRandomValue)(otTraceState.randomValue)) {
        parts.push(`rv:${serializeRv(otTraceState.randomValue)}`);
    }
    if (otTraceState.rest) {
        parts.push(...otTraceState.rest);
    }
    let res = parts.join(';');
    while (res.length > TRACE_STATE_SIZE_LIMIT) {
        const lastSemicolon = res.lastIndexOf(';');
        if (lastSemicolon === -1) {
            break;
        }
        res = res.slice(0, lastSemicolon);
    }
    return res;
}
exports.serializeTraceState = serializeTraceState;
function parseTh(value, defaultValue) {
    if (!value || value.length > MAX_VALUE_LENGTH) {
        return defaultValue;
    }
    try {
        return BigInt('0x' + value.padEnd(MAX_VALUE_LENGTH, '0'));
    }
    catch {
        return defaultValue;
    }
}
function parseRv(value, defaultValue) {
    if (!value || value.length !== MAX_VALUE_LENGTH) {
        return defaultValue;
    }
    try {
        return BigInt(`0x${value}`);
    }
    catch {
        return defaultValue;
    }
}
// hex value without trailing zeros
function serializeTh(threshold) {
    if (threshold === 0n) {
        return '0';
    }
    const value = threshold.toString(16).padStart(MAX_VALUE_LENGTH, '0');
    let idxAfterNonZero = value.length;
    for (let i = value.length - 1; i >= 0; i--) {
        if (value[i] !== '0') {
            idxAfterNonZero = i + 1;
            break;
        }
    }
    // Checked at beginning so there is definitely a nonzero.
    return value.slice(0, idxAfterNonZero);
}
exports.serializeTh = serializeTh;
function serializeRv(randomValue) {
    return randomValue.toString(16).padStart(MAX_VALUE_LENGTH, '0');
}
//# sourceMappingURL=tracestate.js.map