"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createComposableTraceIDRatioBasedSampler = void 0;
const util_1 = require("./util");
const tracestate_1 = require("./tracestate");
class ComposableTraceIDRatioBasedSampler {
    intent;
    description;
    constructor(ratio) {
        if (ratio < 0 || ratio > 1) {
            throw new Error(`Invalid sampling probability: ${ratio}. Must be between 0 and 1.`);
        }
        const threshold = calculateThreshold(ratio);
        const thresholdStr = threshold === util_1.MAX_THRESHOLD ? 'max' : (0, tracestate_1.serializeTh)(threshold);
        if (threshold !== util_1.MAX_THRESHOLD) {
            this.intent = {
                threshold: threshold,
                thresholdReliable: true,
            };
        }
        else {
            // Same as AlwaysOff, notably the threshold is not considered reliable. The spec mentions
            // returning an instance of ComposableAlwaysOffSampler in this case but it seems clearer
            // if the description of the sampler matches the user's request.
            this.intent = {
                threshold: util_1.INVALID_THRESHOLD,
                thresholdReliable: false,
            };
        }
        this.description = `ComposableTraceIDRatioBasedSampler(threshold=${thresholdStr}, ratio=${ratio})`;
    }
    getSamplingIntent() {
        return this.intent;
    }
    toString() {
        return this.description;
    }
}
/**
 * Returns a composable sampler that samples each span with a fixed ratio.
 */
function createComposableTraceIDRatioBasedSampler(ratio) {
    return new ComposableTraceIDRatioBasedSampler(ratio);
}
exports.createComposableTraceIDRatioBasedSampler = createComposableTraceIDRatioBasedSampler;
const probabilityThresholdScale = Math.pow(2, 56);
// TODO: Reduce threshold precision following spec recommendation of 4
// to reduce size of serialized tracestate.
function calculateThreshold(samplingProbability) {
    return (util_1.MAX_THRESHOLD -
        BigInt(Math.round(samplingProbability * probabilityThresholdScale)));
}
//# sourceMappingURL=traceidratio.js.map