"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createComposableParentThresholdSampler = void 0;
const api_1 = require("@opentelemetry/api");
const tracestate_1 = require("./tracestate");
const util_1 = require("./util");
class ComposableParentThresholdSampler {
    rootSampler;
    description;
    constructor(rootSampler) {
        this.rootSampler = rootSampler;
        this.description = `ComposableParentThresholdSampler(rootSampler=${rootSampler})`;
    }
    getSamplingIntent(context, traceId, spanName, spanKind, attributes, links) {
        const parentSpanContext = api_1.trace.getSpanContext(context);
        if (!parentSpanContext || !(0, api_1.isSpanContextValid)(parentSpanContext)) {
            return this.rootSampler.getSamplingIntent(context, traceId, spanName, spanKind, attributes, links);
        }
        const otTraceState = (0, tracestate_1.parseOtelTraceState)(parentSpanContext.traceState);
        if ((0, util_1.isValidThreshold)(otTraceState.threshold)) {
            return {
                threshold: otTraceState.threshold,
                thresholdReliable: true,
            };
        }
        const threshold = parentSpanContext.traceFlags & api_1.TraceFlags.SAMPLED
            ? util_1.MIN_THRESHOLD
            : util_1.INVALID_THRESHOLD;
        return {
            threshold,
            thresholdReliable: false,
        };
    }
    toString() {
        return this.description;
    }
}
/**
 * Returns a composable sampler that respects the sampling decision of the
 * parent span or falls back to the given sampler if it is a root span.
 */
function createComposableParentThresholdSampler(rootSampler) {
    return new ComposableParentThresholdSampler(rootSampler);
}
exports.createComposableParentThresholdSampler = createComposableParentThresholdSampler;
//# sourceMappingURL=parentthreshold.js.map