"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createCompositeSampler = void 0;
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const api_1 = require("@opentelemetry/api");
const core_1 = require("@opentelemetry/core");
const sdk_trace_base_1 = require("@opentelemetry/sdk-trace-base");
const tracestate_1 = require("./tracestate");
const util_1 = require("./util");
class CompositeSampler {
    delegate;
    constructor(delegate) {
        this.delegate = delegate;
    }
    shouldSample(context, traceId, spanName, spanKind, attributes, links) {
        const spanContext = api_1.trace.getSpanContext(context);
        const traceState = spanContext?.traceState;
        let otTraceState = (0, tracestate_1.parseOtelTraceState)(traceState);
        const intent = this.delegate.getSamplingIntent(context, traceId, spanName, spanKind, attributes, links);
        let adjustedCountCorrect = false;
        let sampled = false;
        if ((0, util_1.isValidThreshold)(intent.threshold)) {
            adjustedCountCorrect = intent.thresholdReliable;
            let randomness;
            if ((0, util_1.isValidRandomValue)(otTraceState.randomValue)) {
                randomness = otTraceState.randomValue;
            }
            else {
                // Use last 56 bits of trace_id as randomness.
                randomness = BigInt(`0x${traceId.slice(-14)}`);
            }
            sampled = intent.threshold <= randomness;
        }
        const decision = sampled
            ? sdk_trace_base_1.SamplingDecision.RECORD_AND_SAMPLED
            : sdk_trace_base_1.SamplingDecision.NOT_RECORD;
        if (sampled && adjustedCountCorrect) {
            otTraceState = {
                ...otTraceState,
                threshold: intent.threshold,
            };
        }
        else {
            otTraceState = {
                ...otTraceState,
                threshold: util_1.INVALID_THRESHOLD,
            };
        }
        const otts = (0, tracestate_1.serializeTraceState)(otTraceState);
        let newTraceState;
        if (traceState) {
            newTraceState = traceState;
            if (intent.updateTraceState) {
                newTraceState = intent.updateTraceState(newTraceState);
            }
        }
        if (otts) {
            if (!newTraceState) {
                newTraceState = new core_1.TraceState();
            }
            newTraceState = newTraceState.set('ot', otts);
        }
        return {
            decision,
            attributes: intent.attributes,
            traceState: newTraceState,
        };
    }
    toString() {
        return this.delegate.toString();
    }
}
/**
 * Returns a composite sampler that uses a composable sampler to make its
 * sampling decisions while handling tracestate.
 */
function createCompositeSampler(delegate) {
    return new CompositeSampler(delegate);
}
exports.createCompositeSampler = createCompositeSampler;
//# sourceMappingURL=composite.js.map