/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { ATTR_SERVICE_NAME } from '@opentelemetry/semantic-conventions';
import { ATTR_FAAS_MAX_MEMORY, ATTR_FAAS_INSTANCE, ATTR_CLOUD_PROVIDER, CLOUD_PROVIDER_VALUE_AZURE, ATTR_CLOUD_PLATFORM, CLOUD_PLATFORM_VALUE_AZURE_FUNCTIONS, ATTR_CLOUD_REGION, ATTR_PROCESS_PID, } from '../semconv';
import { WEBSITE_SITE_NAME, WEBSITE_INSTANCE_ID, FUNCTIONS_MEM_LIMIT, REGION_NAME, CLOUD_RESOURCE_ID_RESOURCE_ATTRIBUTE, } from '../types';
import { getAzureResourceUri, isAzureFunction } from '../utils';
const AZURE_FUNCTIONS_ATTRIBUTE_ENV_VARS = {
    [ATTR_SERVICE_NAME]: WEBSITE_SITE_NAME,
    [ATTR_FAAS_INSTANCE]: WEBSITE_INSTANCE_ID,
    [ATTR_FAAS_MAX_MEMORY]: FUNCTIONS_MEM_LIMIT,
};
/**
 * The AzureFunctionsDetector can be used to detect if a process is running in Azure Functions
 * @returns a {@link Resource} populated with data about the environment or an empty Resource if detection fails.
 */
class AzureFunctionsDetector {
    detect() {
        let attributes = {};
        const serviceName = process.env[WEBSITE_SITE_NAME];
        /**
         * Checks that we are operating within an Azure Function using the function version since WEBSITE_SITE_NAME
         * will exist in Azure App Service as well and detectors should be mutually exclusive.
         * If the function version is not present, we check for the website sku to determine if it is a function.
         */
        if (serviceName && isAzureFunction()) {
            const functionInstance = process.env[WEBSITE_INSTANCE_ID];
            const functionMemLimit = process.env[FUNCTIONS_MEM_LIMIT];
            attributes = {
                [ATTR_CLOUD_PROVIDER]: CLOUD_PROVIDER_VALUE_AZURE,
                [ATTR_CLOUD_PLATFORM]: CLOUD_PLATFORM_VALUE_AZURE_FUNCTIONS,
                [ATTR_CLOUD_REGION]: process.env[REGION_NAME],
                [ATTR_PROCESS_PID]: process.pid,
            };
            if (serviceName) {
                attributes = {
                    ...attributes,
                    [ATTR_SERVICE_NAME]: serviceName,
                };
            }
            if (functionInstance) {
                attributes = {
                    ...attributes,
                    [ATTR_FAAS_INSTANCE]: functionInstance,
                };
            }
            if (functionMemLimit) {
                attributes = {
                    ...attributes,
                    [ATTR_FAAS_MAX_MEMORY]: functionMemLimit,
                };
            }
            const azureResourceUri = getAzureResourceUri(serviceName);
            if (azureResourceUri) {
                attributes = {
                    ...attributes,
                    ...{ [CLOUD_RESOURCE_ID_RESOURCE_ATTRIBUTE]: azureResourceUri },
                };
            }
            for (const [key, value] of Object.entries(AZURE_FUNCTIONS_ATTRIBUTE_ENV_VARS)) {
                const envVar = process.env[value];
                if (envVar) {
                    attributes = { ...attributes, ...{ [key]: envVar } };
                }
            }
        }
        return { attributes };
    }
}
export const azureFunctionsDetector = new AzureFunctionsDetector();
//# sourceMappingURL=AzureFunctionsDetector.js.map