"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenAIInstrumentation = void 0;
// avoids a dependency on @opentelemetry/core for hrTime utilities
const perf_hooks_1 = require("perf_hooks");
const api_1 = require("@opentelemetry/api");
const instrumentation_1 = require("@opentelemetry/instrumentation");
const api_logs_1 = require("@opentelemetry/api-logs");
const semconv_1 = require("./semconv");
/** @knipignore */
const version_1 = require("./version");
const utils_1 = require("./utils");
// The JS semconv package doesn't yet emit constants for event names.
// TODO: otel-js issue for semconv pkg not including event names
const EVENT_GEN_AI_SYSTEM_MESSAGE = 'gen_ai.system.message';
const EVENT_GEN_AI_USER_MESSAGE = 'gen_ai.user.message';
const EVENT_GEN_AI_ASSISTANT_MESSAGE = 'gen_ai.assistant.message';
const EVENT_GEN_AI_TOOL_MESSAGE = 'gen_ai.tool.message';
const EVENT_GEN_AI_CHOICE = 'gen_ai.choice';
class OpenAIInstrumentation extends instrumentation_1.InstrumentationBase {
    _genaiClientOperationDuration;
    _genaiClientTokenUsage;
    constructor(config = {}) {
        super(version_1.PACKAGE_NAME, version_1.PACKAGE_VERSION, config);
        // Possible environment variable overrides for config.
        const cfg = this.getConfig();
        const envCC = (0, utils_1.getEnvBool)('OTEL_INSTRUMENTATION_GENAI_CAPTURE_MESSAGE_CONTENT', this._diag);
        if (envCC !== undefined) {
            cfg.captureMessageContent = envCC;
        }
    }
    // Override InstrumentationAbtract.setConfig so we can normalize config.
    setConfig(config = {}) {
        const { captureMessageContent, ...validConfig } = config;
        validConfig.captureMessageContent =
            !!captureMessageContent;
        super.setConfig(validConfig);
    }
    init() {
        return [
            new instrumentation_1.InstrumentationNodeModuleDefinition('openai', ['>=4.19.0 <7'], modExports => {
                this._wrap(modExports.OpenAI.Chat.Completions.prototype, 'create', this._getPatchedChatCompletionsCreate());
                this._wrap(modExports.OpenAI.Embeddings.prototype, 'create', this._getPatchedEmbeddingsCreate());
                return modExports;
            }, modExports => {
                this._unwrap(modExports.OpenAI.Chat.Completions.prototype, 'create');
                this._unwrap(modExports.OpenAI.Embeddings.prototype, 'create');
            }),
        ];
    }
    // This is a 'protected' method on class `InstrumentationAbstract`.
    _updateMetricInstruments() {
        this._genaiClientOperationDuration = this.meter.createHistogram(semconv_1.METRIC_GEN_AI_CLIENT_OPERATION_DURATION, {
            description: 'GenAI operation duration',
            unit: 's',
            advice: {
                explicitBucketBoundaries: [
                    0.01, 0.02, 0.04, 0.08, 0.16, 0.32, 0.64, 1.28, 2.56, 5.12, 10.24,
                    20.48, 40.96, 81.92,
                ],
            },
        });
        this._genaiClientTokenUsage = this.meter.createHistogram(semconv_1.METRIC_GEN_AI_CLIENT_TOKEN_USAGE, {
            description: 'Measures number of input and output tokens used',
            unit: '{token}',
            advice: {
                explicitBucketBoundaries: [
                    1, 4, 16, 64, 256, 1024, 4096, 16384, 65536, 262144, 1048576,
                    4194304, 16777216, 67108864,
                ],
            },
        });
    }
    _getPatchedChatCompletionsCreate() {
        const self = this;
        return (original) => {
            // https://platform.openai.com/docs/api-reference/chat/create
            return function patchedCreate(...args) {
                if (!self.isEnabled) {
                    return original.apply(this, args);
                }
                self._diag.debug('OpenAI.Chat.Completions.create args: %O', args);
                const params = args[0];
                const config = self.getConfig();
                const startNow = perf_hooks_1.performance.now();
                let startInfo;
                try {
                    startInfo = self._startChatCompletionsSpan(params, config, this?._client?.baseURL);
                }
                catch (err) {
                    self._diag.error('unexpected error starting span:', err);
                    return original.apply(this, args);
                }
                const { span, ctx, commonAttrs } = startInfo;
                const apiPromise = api_1.context.with(ctx, () => original.apply(this, args));
                // Streaming.
                if (isStreamPromise(params, apiPromise)) {
                    // When streaming, `apiPromise` resolves to `Stream`,
                    // an async iterable (i.e. has a `Symbol.asyncIterator` method). We
                    // want to wrap that iteration to gather telemetry. Instead of wrapping
                    // `Symbol.asyncIterator`, which would be nice, we wrap the `iterator`
                    // method because it is used internally by `Stream#tee()`.
                    return apiPromise.then(stream => {
                        // eslint-disable-next-line @typescript-eslint/no-explicit-any
                        self._wrap(stream, 'iterator', origIterator => {
                            return () => {
                                return self._onChatCompletionsStreamIterator(origIterator(), span, startNow, config, commonAttrs, ctx);
                            };
                        });
                        return stream;
                    });
                }
                // Non-streaming.
                apiPromise
                    .then(result => {
                    self._onChatCompletionsCreateResult(span, startNow, commonAttrs, result, config, ctx);
                })
                    .catch(self._createAPIPromiseRejectionHandler(startNow, span, commonAttrs));
                return apiPromise;
            };
        };
    }
    /**
     * Start a span for this chat-completion API call. This also emits log events
     * as appropriate for the request params.
     */
    _startChatCompletionsSpan(params, config, baseURL) {
        // Attributes common to span, metrics, log events.
        const commonAttrs = {
            [semconv_1.ATTR_GEN_AI_OPERATION_NAME]: 'chat',
            [semconv_1.ATTR_GEN_AI_REQUEST_MODEL]: params.model,
            [semconv_1.ATTR_GEN_AI_SYSTEM]: 'openai',
        };
        Object.assign(commonAttrs, (0, utils_1.getAttrsFromBaseURL)(baseURL, this._diag));
        // Span attributes.
        const attrs = {
            ...commonAttrs,
        };
        if (params.frequency_penalty != null) {
            attrs[semconv_1.ATTR_GEN_AI_REQUEST_FREQUENCY_PENALTY] = params.frequency_penalty;
        }
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        if (params.max_completion_tokens != null) {
            attrs[semconv_1.ATTR_GEN_AI_REQUEST_MAX_TOKENS] =
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                params.max_completion_tokens;
        }
        else if (params.max_tokens != null) {
            // `max_tokens` is deprecated in favour of `max_completion_tokens`.
            attrs[semconv_1.ATTR_GEN_AI_REQUEST_MAX_TOKENS] = params.max_tokens;
        }
        if (params.presence_penalty != null) {
            attrs[semconv_1.ATTR_GEN_AI_REQUEST_PRESENCE_PENALTY] = params.presence_penalty;
        }
        if (params.stop != null) {
            if (Array.isArray(params.stop)) {
                attrs[semconv_1.ATTR_GEN_AI_REQUEST_STOP_SEQUENCES] = params.stop;
            }
            else {
                attrs[semconv_1.ATTR_GEN_AI_REQUEST_STOP_SEQUENCES] = [params.stop];
            }
        }
        if (params.temperature != null) {
            attrs[semconv_1.ATTR_GEN_AI_REQUEST_TEMPERATURE] = params.temperature;
        }
        if (params.top_p != null) {
            attrs[semconv_1.ATTR_GEN_AI_REQUEST_TOP_P] = params.top_p;
        }
        const span = this.tracer.startSpan(`${attrs[semconv_1.ATTR_GEN_AI_OPERATION_NAME]} ${attrs[semconv_1.ATTR_GEN_AI_REQUEST_MODEL]}`, {
            kind: api_1.SpanKind.CLIENT,
            attributes: attrs,
        });
        const ctx = api_1.trace.setSpan(api_1.context.active(), span);
        // Capture prompts as log events.
        const timestamp = Date.now();
        params.messages.forEach((msg) => {
            switch (msg.role) {
                case 'system': {
                    const body = {};
                    if (config.captureMessageContent) {
                        if (Array.isArray(msg.content)) {
                            body.content = msg.content.map(p => p.text).join('');
                        }
                        else {
                            body.content = msg.content;
                        }
                    }
                    this.logger.emit({
                        timestamp,
                        context: ctx,
                        severityNumber: api_logs_1.SeverityNumber.INFO,
                        attributes: {
                            [semconv_1.ATTR_EVENT_NAME]: EVENT_GEN_AI_SYSTEM_MESSAGE,
                            [semconv_1.ATTR_GEN_AI_SYSTEM]: 'openai',
                        },
                        body,
                    });
                    break;
                }
                case 'user': {
                    const body = {};
                    if (config.captureMessageContent) {
                        if (Array.isArray(msg.content)) {
                            body.content = msg.content
                                .filter(isTextContent)
                                .map(p => p.text)
                                .join('');
                        }
                        else {
                            body.content = msg.content;
                        }
                    }
                    this.logger.emit({
                        timestamp,
                        context: ctx,
                        severityNumber: api_logs_1.SeverityNumber.INFO,
                        attributes: {
                            [semconv_1.ATTR_EVENT_NAME]: EVENT_GEN_AI_USER_MESSAGE,
                            [semconv_1.ATTR_GEN_AI_SYSTEM]: 'openai',
                        },
                        body,
                    });
                    break;
                }
                case 'assistant': {
                    const body = {};
                    if (config.captureMessageContent) {
                        if (msg.content) {
                            if (Array.isArray(msg.content)) {
                                body.content = msg.content
                                    .filter(isTextContent)
                                    .map(p => p.text)
                                    .join('');
                            }
                            else {
                                body.content = msg.content;
                            }
                        }
                        // As of openai@5.12.1, type ChatCompletionMessageToolCall can
                        // have type="custom" which has no `function` property. As well,
                        // GenAI semconv has since changed how it captures tool calls.
                        // For now we just cope: we could capture
                        // `ChatCompletionMessageCustomToolCall.Custom` properties, but we
                        // don't for now.
                        body.tool_calls = msg.tool_calls?.map(tc => {
                            const repr = {
                                id: tc.id,
                                type: tc.type,
                            };
                            if (tc.type === 'function') {
                                repr.function = {
                                    name: tc.function.name,
                                    arguments: tc.function.arguments,
                                };
                            }
                            return repr;
                        });
                    }
                    else {
                        body.tool_calls = msg.tool_calls?.map(tc => {
                            const repr = {
                                id: tc.id,
                                type: tc.type,
                            };
                            if (tc.type === 'function') {
                                repr.function = { name: tc.function.name };
                            }
                            return repr;
                        });
                    }
                    this.logger.emit({
                        timestamp,
                        context: ctx,
                        severityNumber: api_logs_1.SeverityNumber.INFO,
                        attributes: {
                            [semconv_1.ATTR_EVENT_NAME]: EVENT_GEN_AI_ASSISTANT_MESSAGE,
                            [semconv_1.ATTR_GEN_AI_SYSTEM]: 'openai',
                        },
                        body,
                    });
                    break;
                }
                case 'tool': {
                    const body = {
                        id: msg.tool_call_id,
                    };
                    if (config.captureMessageContent) {
                        if (Array.isArray(msg.content)) {
                            body.content = msg.content.map(p => p.text).join('');
                        }
                        else {
                            body.content = msg.content;
                        }
                    }
                    this.logger.emit({
                        timestamp,
                        context: ctx,
                        severityNumber: api_logs_1.SeverityNumber.INFO,
                        attributes: {
                            [semconv_1.ATTR_EVENT_NAME]: EVENT_GEN_AI_TOOL_MESSAGE,
                            [semconv_1.ATTR_GEN_AI_SYSTEM]: 'openai',
                        },
                        body,
                    });
                    break;
                }
                default:
                    this._diag.debug(`unknown message role in OpenAI.Chat.Completions.create: ${msg.role}`);
            }
        });
        return { span, ctx, commonAttrs };
    }
    /**
     * This wraps an instance of a `openai/streaming.Stream.iterator()`, an
     * async iterator. It should yield the chunks unchanged, and gather telemetry
     * data from those chunks, then end the span.
     */
    async *_onChatCompletionsStreamIterator(streamIter, span, startNow, config, commonAttrs, ctx) {
        let id;
        let model;
        const finishReasons = [];
        const choices = [];
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        for await (const chunk of streamIter) {
            yield chunk;
            // Gather telemetry from this chunk.
            this._diag.debug('OpenAI.Chat.Completions.create stream chunk: %O', chunk);
            const idx = chunk.choices[0]?.index ?? 0;
            if (!choices[idx]) {
                choices[idx] = {};
            }
            if (config.captureMessageContent) {
                const contentPart = chunk.choices[0]?.delta?.content;
                if (contentPart) {
                    if (!choices[idx].content) {
                        choices[idx].content = '';
                    }
                    choices[idx].content += contentPart;
                }
            }
            // Assume delta.tool_calls, if exists, is an array of length 1.
            const toolCallPart = chunk.choices[0]?.delta?.tool_calls?.[0];
            if (toolCallPart) {
                if (!choices[idx].toolCalls) {
                    choices[idx].toolCalls = [];
                }
                const toolCalls = choices[idx].toolCalls;
                if (toolCallPart.id) {
                    // First chunk in a tool call.
                    const repr = {
                        id: toolCallPart.id,
                        type: toolCallPart.type,
                    };
                    if (toolCallPart.type === 'function') {
                        repr.function = {
                            name: toolCallPart.function?.name,
                            arguments: toolCallPart.function?.arguments ?? '',
                        };
                    }
                    toolCalls.push(repr);
                }
                else if (toolCalls.length > 0) {
                    // A tool call chunk with more of the `function.arguments`.
                    const lastPart = toolCalls[toolCalls.length - 1];
                    if (lastPart.function !== undefined) {
                        lastPart.function.arguments +=
                            toolCallPart.function?.arguments ?? '';
                    }
                }
            }
            if (!id && chunk.id) {
                id = chunk.id;
                span.setAttribute(semconv_1.ATTR_GEN_AI_RESPONSE_ID, id);
            }
            if (!model && chunk.model) {
                model = chunk.model;
                span.setAttribute(semconv_1.ATTR_GEN_AI_RESPONSE_MODEL, model);
            }
            if (!finishReasons[idx]) {
                const finishReason = chunk.choices[0]?.finish_reason;
                if (finishReason) {
                    finishReasons[idx] = finishReason;
                }
            }
            if (chunk.usage) {
                // A final usage chunk if `stream_options.include_usage: true`.
                span.setAttribute(semconv_1.ATTR_GEN_AI_USAGE_INPUT_TOKENS, chunk.usage.prompt_tokens);
                span.setAttribute(semconv_1.ATTR_GEN_AI_USAGE_OUTPUT_TOKENS, chunk.usage.completion_tokens);
                this._genaiClientTokenUsage.record(chunk.usage.prompt_tokens, {
                    ...commonAttrs,
                    [semconv_1.ATTR_GEN_AI_RESPONSE_MODEL]: model,
                    [semconv_1.ATTR_GEN_AI_TOKEN_TYPE]: 'input',
                });
                this._genaiClientTokenUsage.record(chunk.usage.completion_tokens, {
                    ...commonAttrs,
                    [semconv_1.ATTR_GEN_AI_RESPONSE_MODEL]: model,
                    [semconv_1.ATTR_GEN_AI_TOKEN_TYPE]: 'output',
                });
            }
        }
        span.setAttribute(semconv_1.ATTR_GEN_AI_RESPONSE_FINISH_REASONS, finishReasons.filter(f => f !== undefined));
        // Capture choices as log events.
        for (const [idx, choice] of choices.entries()) {
            if (!choice) {
                continue;
            }
            const message = {};
            if (config.captureMessageContent &&
                choice.content &&
                choice.content.length > 0) {
                message.content = choice.content;
            }
            if (choice.toolCalls && choice.toolCalls.length > 0) {
                message.tool_calls = choice.toolCalls;
                if (!config.captureMessageContent) {
                    message.tool_calls.forEach(tc => {
                        delete tc.function?.arguments;
                    });
                }
            }
            this.logger.emit({
                timestamp: Date.now(),
                context: ctx,
                severityNumber: api_logs_1.SeverityNumber.INFO,
                attributes: {
                    [semconv_1.ATTR_EVENT_NAME]: EVENT_GEN_AI_CHOICE,
                    [semconv_1.ATTR_GEN_AI_SYSTEM]: 'openai',
                },
                body: {
                    finish_reason: finishReasons[idx],
                    index: idx,
                    message,
                },
            });
        }
        this._genaiClientOperationDuration.record((perf_hooks_1.performance.now() - startNow) / 1000, {
            ...commonAttrs,
            [semconv_1.ATTR_GEN_AI_RESPONSE_MODEL]: model,
        });
        span.end();
    }
    _onChatCompletionsCreateResult(span, startNow, commonAttrs, result, config, ctx) {
        this._diag.debug('OpenAI.Chat.Completions.create result: %O', result);
        try {
            span.setAttribute(semconv_1.ATTR_GEN_AI_RESPONSE_FINISH_REASONS, result.choices.map(c => c.finish_reason));
            span.setAttribute(semconv_1.ATTR_GEN_AI_RESPONSE_ID, result.id);
            span.setAttribute(semconv_1.ATTR_GEN_AI_RESPONSE_MODEL, result.model);
            if (result.usage) {
                span.setAttribute(semconv_1.ATTR_GEN_AI_USAGE_INPUT_TOKENS, result.usage.prompt_tokens);
                span.setAttribute(semconv_1.ATTR_GEN_AI_USAGE_OUTPUT_TOKENS, result.usage.completion_tokens);
            }
            // Capture choices as log events.
            result.choices.forEach(choice => {
                const message = {};
                if (config.captureMessageContent) {
                    // TODO: telemetry diff with streaming case: content=null, no 'role: assistant', 'tool calls (enableCaptureContent=true)' test case
                    if (choice.message.content) {
                        message.content = choice.message.content;
                    }
                    if (choice.message.tool_calls) {
                        message.tool_calls = choice.message.tool_calls;
                    }
                }
                else {
                    if (choice.message.tool_calls) {
                        message.tool_calls = choice.message.tool_calls.map(tc => {
                            const repr = {
                                id: tc.id,
                                type: tc.type,
                            };
                            if (tc.type === 'function') {
                                repr.function = { name: tc.function.name };
                            }
                            return repr;
                        });
                    }
                }
                this.logger.emit({
                    timestamp: Date.now(),
                    context: ctx,
                    severityNumber: api_logs_1.SeverityNumber.INFO,
                    attributes: {
                        [semconv_1.ATTR_EVENT_NAME]: EVENT_GEN_AI_CHOICE,
                        [semconv_1.ATTR_GEN_AI_SYSTEM]: 'openai',
                    },
                    body: {
                        finish_reason: choice.finish_reason,
                        index: choice.index,
                        message,
                    },
                });
            });
            this._genaiClientOperationDuration.record((perf_hooks_1.performance.now() - startNow) / 1000, {
                ...commonAttrs,
                [semconv_1.ATTR_GEN_AI_RESPONSE_MODEL]: result.model,
            });
            if (result.usage) {
                this._genaiClientTokenUsage.record(result.usage.prompt_tokens, {
                    ...commonAttrs,
                    [semconv_1.ATTR_GEN_AI_RESPONSE_MODEL]: result.model,
                    [semconv_1.ATTR_GEN_AI_TOKEN_TYPE]: 'input',
                });
                this._genaiClientTokenUsage.record(result.usage.completion_tokens, {
                    ...commonAttrs,
                    [semconv_1.ATTR_GEN_AI_RESPONSE_MODEL]: result.model,
                    [semconv_1.ATTR_GEN_AI_TOKEN_TYPE]: 'output',
                });
            }
        }
        catch (err) {
            this._diag.error('unexpected error getting telemetry from chat result:', err);
        }
        span.end();
    }
    _createAPIPromiseRejectionHandler(startNow, span, commonAttrs) {
        return (err) => {
            this._diag.debug('OpenAI APIPromise rejection: %O', err);
            // https://github.com/openai/openai-node/blob/master/src/error.ts
            // The most reliable low cardinality string for errors seems to be
            // the class name. See also:
            // https://platform.openai.com/docs/guides/error-codes
            const errorType = err?.constructor?.name;
            this._genaiClientOperationDuration.record((perf_hooks_1.performance.now() - startNow) / 1000, {
                ...commonAttrs,
                'error.type': errorType,
            });
            span.setStatus({
                code: api_1.SpanStatusCode.ERROR,
                message: err.message,
            });
            span.setAttribute('error.type', errorType);
            span.end();
        };
    }
    _getPatchedEmbeddingsCreate() {
        const self = this;
        return (original) => {
            // https://platform.openai.com/docs/api-reference/embeddings/create
            return function patchedCreate(...args) {
                if (!self.isEnabled) {
                    return original.apply(this, args);
                }
                self._diag.debug('OpenAI.Chat.Embeddings.create args: %O', args);
                const params = args[0];
                const startNow = perf_hooks_1.performance.now();
                let startInfo;
                try {
                    startInfo = self._startEmbeddingsSpan(params, this?._client?.baseURL);
                }
                catch (err) {
                    self._diag.error('unexpected error starting span:', err);
                    return original.apply(this, args);
                }
                const { span, ctx, commonAttrs } = startInfo;
                const apiPromise = api_1.context.with(ctx, () => original.apply(this, args));
                apiPromise
                    .then(result => {
                    self._onEmbeddingsCreateResult(span, startNow, commonAttrs, result);
                })
                    .catch(self._createAPIPromiseRejectionHandler(startNow, span, commonAttrs));
                return apiPromise;
            };
        };
    }
    /**
     * Start a span for this chat-completion API call. This also emits log events
     * as appropriate for the request params.
     */
    _startEmbeddingsSpan(params, baseURL) {
        // Attributes common to span, metrics, log events.
        const commonAttrs = {
            [semconv_1.ATTR_GEN_AI_OPERATION_NAME]: 'embeddings',
            [semconv_1.ATTR_GEN_AI_REQUEST_MODEL]: params.model,
            [semconv_1.ATTR_GEN_AI_SYSTEM]: 'openai',
        };
        Object.assign(commonAttrs, (0, utils_1.getAttrsFromBaseURL)(baseURL, this._diag));
        // Span attributes.
        const attrs = {
            ...commonAttrs,
        };
        if (params.encoding_format != null) {
            attrs[semconv_1.ATTR_GEN_AI_REQUEST_ENCODING_FORMATS] = [params.encoding_format];
        }
        const span = this.tracer.startSpan(`${attrs[semconv_1.ATTR_GEN_AI_OPERATION_NAME]} ${attrs[semconv_1.ATTR_GEN_AI_REQUEST_MODEL]}`, {
            kind: api_1.SpanKind.CLIENT,
            attributes: attrs,
        });
        const ctx = api_1.trace.setSpan(api_1.context.active(), span);
        return { span, ctx, commonAttrs };
    }
    _onEmbeddingsCreateResult(span, startNow, commonAttrs, result) {
        this._diag.debug('OpenAI.Embeddings.create result: %O', result);
        try {
            span.setAttribute(semconv_1.ATTR_GEN_AI_RESPONSE_MODEL, result.model);
            this._genaiClientOperationDuration.record((perf_hooks_1.performance.now() - startNow) / 1000, {
                ...commonAttrs,
                [semconv_1.ATTR_GEN_AI_RESPONSE_MODEL]: result.model,
            });
            span.setAttribute(semconv_1.ATTR_GEN_AI_USAGE_INPUT_TOKENS, result.usage.prompt_tokens);
            this._genaiClientTokenUsage.record(result.usage.prompt_tokens, {
                ...commonAttrs,
                [semconv_1.ATTR_GEN_AI_RESPONSE_MODEL]: result.model,
                [semconv_1.ATTR_GEN_AI_TOKEN_TYPE]: 'input',
            });
        }
        catch (err) {
            this._diag.error('unexpected error getting telemetry from embeddings result:', err);
        }
        span.end();
    }
}
exports.OpenAIInstrumentation = OpenAIInstrumentation;
function isTextContent(value) {
    return value.type === 'text';
}
function isStreamPromise(params, value) {
    if (params && params.stream) {
        return true;
    }
    return false;
}
//# sourceMappingURL=instrumentation.js.map