"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.jsonSchemaToZod = exports.WatsonxToolsOutputParser = exports._convertToolCallIdToMistralCompatible = exports._isValidMistralToolCallId = exports.authenticateAndSetInstance = void 0;
/* eslint-disable @typescript-eslint/no-explicit-any */
const watsonx_ai_1 = require("@ibm-cloud/watsonx-ai");
const ibm_cloud_sdk_core_1 = require("ibm-cloud-sdk-core");
const openai_tools_1 = require("@langchain/core/output_parsers/openai_tools");
const output_parsers_1 = require("@langchain/core/output_parsers");
const zod_1 = require("zod");
const authenticateAndSetInstance = ({ watsonxAIApikey, watsonxAIAuthType, watsonxAIBearerToken, watsonxAIUsername, watsonxAIPassword, watsonxAIUrl, version, serviceUrl, }) => {
    if (watsonxAIAuthType === "iam" && watsonxAIApikey) {
        return watsonx_ai_1.WatsonXAI.newInstance({
            version,
            serviceUrl,
            authenticator: new ibm_cloud_sdk_core_1.IamAuthenticator({
                apikey: watsonxAIApikey,
            }),
        });
    }
    else if (watsonxAIAuthType === "bearertoken" && watsonxAIBearerToken) {
        return watsonx_ai_1.WatsonXAI.newInstance({
            version,
            serviceUrl,
            authenticator: new ibm_cloud_sdk_core_1.BearerTokenAuthenticator({
                bearerToken: watsonxAIBearerToken,
            }),
        });
    }
    else if (watsonxAIAuthType === "cp4d" && watsonxAIUrl) {
        if (watsonxAIUsername && watsonxAIPassword && watsonxAIApikey)
            return watsonx_ai_1.WatsonXAI.newInstance({
                version,
                serviceUrl,
                authenticator: new ibm_cloud_sdk_core_1.CloudPakForDataAuthenticator({
                    username: watsonxAIUsername,
                    password: watsonxAIPassword,
                    url: watsonxAIUrl,
                    apikey: watsonxAIApikey,
                }),
            });
    }
    else
        return watsonx_ai_1.WatsonXAI.newInstance({
            version,
            serviceUrl,
        });
    return undefined;
};
exports.authenticateAndSetInstance = authenticateAndSetInstance;
// Mistral enforces a specific pattern for tool call IDs
// Thanks to Mistral for implementing this, I was unable to import which is why this is copied 1:1
const TOOL_CALL_ID_PATTERN = /^[a-zA-Z0-9]{9}$/;
function _isValidMistralToolCallId(toolCallId) {
    return TOOL_CALL_ID_PATTERN.test(toolCallId);
}
exports._isValidMistralToolCallId = _isValidMistralToolCallId;
function _base62Encode(num) {
    let numCopy = num;
    const base62 = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
    if (numCopy === 0)
        return base62[0];
    const arr = [];
    const base = base62.length;
    while (numCopy) {
        arr.push(base62[numCopy % base]);
        numCopy = Math.floor(numCopy / base);
    }
    return arr.reverse().join("");
}
function _simpleHash(str) {
    let hash = 0;
    for (let i = 0; i < str.length; i += 1) {
        const char = str.charCodeAt(i);
        hash = (hash << 5) - hash + char;
        hash &= hash; // Convert to 32-bit integer
    }
    return Math.abs(hash);
}
function _convertToolCallIdToMistralCompatible(toolCallId) {
    if (_isValidMistralToolCallId(toolCallId)) {
        return toolCallId;
    }
    else {
        const hash = _simpleHash(toolCallId);
        const base62Str = _base62Encode(hash);
        if (base62Str.length >= 9) {
            return base62Str.slice(0, 9);
        }
        else {
            return base62Str.padStart(9, "0");
        }
    }
}
exports._convertToolCallIdToMistralCompatible = _convertToolCallIdToMistralCompatible;
class WatsonxToolsOutputParser extends openai_tools_1.JsonOutputToolsParser {
    static lc_name() {
        return "WatsonxToolsOutputParser";
    }
    constructor(params) {
        super(params);
        Object.defineProperty(this, "lc_namespace", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: ["langchain", "watsonx", "output_parsers"]
        });
        Object.defineProperty(this, "returnId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(this, "keyName", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "returnSingle", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(this, "zodSchema", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "latestCorrect", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.keyName = params.keyName;
        this.returnSingle = params.returnSingle ?? this.returnSingle;
        this.zodSchema = params.zodSchema;
    }
    async _validateResult(result) {
        let parsedResult = result;
        if (typeof result === "string") {
            try {
                parsedResult = JSON.parse(result);
            }
            catch (e) {
                throw new output_parsers_1.OutputParserException(`Failed to parse. Text: "${JSON.stringify(result, null, 2)}". Error: ${JSON.stringify(e.message)}`, result);
            }
        }
        else {
            parsedResult = result;
        }
        if (this.zodSchema === undefined) {
            return parsedResult;
        }
        const zodParsedResult = await this.zodSchema.safeParseAsync(parsedResult);
        if (zodParsedResult.success) {
            return zodParsedResult.data;
        }
        else {
            throw new output_parsers_1.OutputParserException(`Failed to parse. Text: "${JSON.stringify(result, null, 2)}". Error: ${JSON.stringify(zodParsedResult.error.errors)}`, JSON.stringify(result, null, 2));
        }
    }
    async parsePartialResult(generations) {
        const tools = generations.flatMap((generation) => {
            const message = generation.message;
            if (!Array.isArray(message.tool_calls)) {
                return [];
            }
            const tool = message.tool_calls;
            return tool;
        });
        if (tools[0] === undefined) {
            if (this.latestCorrect) {
                tools.push(this.latestCorrect);
            }
            else {
                const toolCall = { name: "", args: {} };
                tools.push(toolCall);
            }
        }
        const [tool] = tools;
        tool.name = "";
        this.latestCorrect = tool;
        return tool.args;
    }
}
exports.WatsonxToolsOutputParser = WatsonxToolsOutputParser;
function jsonSchemaToZod(obj) {
    if (obj?.properties && obj.type === "object") {
        const shape = {};
        Object.keys(obj.properties).forEach((key) => {
            if (obj.properties) {
                const prop = obj.properties[key];
                let zodType;
                if (prop.type === "string") {
                    zodType = zod_1.z.string();
                    if (prop?.pattern) {
                        zodType = zodType.regex(prop.pattern, "Invalid pattern");
                    }
                }
                else if (prop.type === "number" ||
                    prop.type === "integer" ||
                    prop.type === "float") {
                    zodType = zod_1.z.number();
                    if (typeof prop?.minimum === "number") {
                        zodType = zodType.min(prop.minimum, {
                            message: `${key} must be at least ${prop.minimum}`,
                        });
                    }
                    if (prop?.maximum)
                        zodType = zodType.lte(prop.maximum, {
                            message: `${key} must be maximum of ${prop.maximum}`,
                        });
                }
                else if (prop.type === "boolean")
                    zodType = zod_1.z.boolean();
                else if (prop.type === "array")
                    zodType = zod_1.z.array(jsonSchemaToZod(prop.items));
                else if (prop.type === "object") {
                    zodType = jsonSchemaToZod(prop);
                }
                else
                    throw new Error(`Unsupported type: ${prop.type}`);
                if (prop.description) {
                    zodType = zodType.describe(prop.description);
                }
                if (!obj.required?.includes(key)) {
                    zodType = zodType.optional();
                }
                shape[key] = zodType;
            }
        });
        return zod_1.z.object(shape);
    }
    throw new Error("Unsupported root schema type");
}
exports.jsonSchemaToZod = jsonSchemaToZod;
