"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkflowGraph = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _dagre = require("@dagrejs/dagre");
var _build_execution_graph = require("../build_execution_graph/build_execution_graph");
var _create_typed_graph = require("./create_typed_graph");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * A class that encapsulates the logic of workflow graph operations and provides
 * a specific API to work with directed graphs representing workflow definitions.
 *
 * This class wraps the graphlib.Graph functionality and provides workflow-specific
 * methods for traversing, analyzing, and extracting subgraphs from workflow definitions.
 *
 * @example
 * ```typescript
 * const workflowGraph = WorkflowGraph.fromWorkflowDefinition(workflowDef);
 * const nodes = workflowGraph.getAllNodes();
 * const stepGraph = workflowGraph.getStepGraph('step-id');
 * ```
 */
class WorkflowGraph {
  constructor(graph) {
    (0, _defineProperty2.default)(this, "graph", void 0);
    (0, _defineProperty2.default)(this, "__topologicalOrder", null);
    (0, _defineProperty2.default)(this, "stepIdsSet", null);
    this.graph = graph;
  }
  static fromWorkflowDefinition(workflowDefinition) {
    return new WorkflowGraph((0, _build_execution_graph.convertToWorkflowGraph)(workflowDefinition));
  }
  get topologicalOrder() {
    if (!this.__topologicalOrder) {
      this.__topologicalOrder = _dagre.graphlib.alg.topsort(this.graph);
    }
    return this.__topologicalOrder;
  }
  getNode(nodeId) {
    return this.graph.node(nodeId);
  }
  getNodeStack(nodeId) {
    const predecessors = this.getAllPredecessors(nodeId).toReversed();
    const stack = [];
    for (const node of predecessors) {
      if (node.type.startsWith('enter-')) {
        stack.push(node.id);
      }
      if (node.type.startsWith('exit-')) {
        stack.pop();
      }
    }
    return stack;
  }
  getAllNodes() {
    return this.graph.nodes().map(nodeId => this.graph.node(nodeId));
  }
  getEdges() {
    return this.graph.edges().map(edge => ({
      v: edge.v,
      w: edge.w
    }));
  }
  hasStep(stepId) {
    if (!this.stepIdsSet) {
      this.stepIdsSet = new Set(this.getAllNodes().map(node => node.stepId));
    }
    return this.stepIdsSet.has(stepId);
  }
  getStepGraph(stepId) {
    // Find the boundaries of the step in topological order
    const beginNodeIndex = this.topologicalOrder.findIndex(id => this.getNode(id).stepId === stepId);
    if (beginNodeIndex === -1) {
      throw new Error(`Step with id ${stepId} not found in the workflow graph.`);
    }

    // Find the last node of the target step
    let endNodeIndex = -1;
    for (let i = this.topologicalOrder.length - 1; i >= beginNodeIndex; i--) {
      const nodeId = this.topologicalOrder[i];
      if (this.getNode(nodeId).stepId === stepId) {
        endNodeIndex = i;
        break;
      }
    }
    if (endNodeIndex === -1) {
      throw new Error(`Step with id ${stepId} not found in the workflow graph.`);
    }

    // Extract all nodes between begin and end (inclusive) - this includes child steps
    const subGraphNodeIds = this.topologicalOrder.slice(beginNodeIndex, endNodeIndex + 1);
    const subGraph = (0, _create_typed_graph.createTypedGraph)({
      directed: true
    });

    // Add all nodes in the range to subgraph
    for (const nodeId of subGraphNodeIds) {
      subGraph.setNode(nodeId, this.graph.node(nodeId));
    }

    // Add edges between nodes that are both in the subgraph
    const nodeIdSet = new Set(subGraphNodeIds);
    for (const nodeId of subGraphNodeIds) {
      const successors = this.graph.successors(nodeId) || [];
      for (const succId of successors) {
        // Only add edge if both nodes are in the subgraph
        if (nodeIdSet.has(succId)) {
          subGraph.setEdge(nodeId, succId);
        }
      }
    }
    return new WorkflowGraph(subGraph);
  }
  getDirectSuccessors(nodeId) {
    const successors = this.graph.successors(nodeId) || [];
    return successors.map(id => this.graph.node(id));
  }
  getAllPredecessors(nodeId) {
    const visited = new Set();
    const collectPredecessors = predNodeId => {
      if (visited.has(predNodeId)) {
        return;
      }
      visited.add(predNodeId);
      const preds = this.graph.predecessors(predNodeId) || [];
      preds.forEach(predId => collectPredecessors(predId));
    };
    const directPredecessors = this.graph.predecessors(nodeId) || [];
    directPredecessors.forEach(predId => collectPredecessors(predId));
    return Array.from(visited).map(id => this.graph.node(id));
  }
}
exports.WorkflowGraph = WorkflowGraph;