"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.extractPropertyPathsFromKql = extractPropertyPathsFromKql;
var _esQuery = require("@kbn/es-query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Extracts property paths from a KQL (Kibana Query Language) query string.
 *
 * This function parses a KQL query and extracts field names that appear before
 * colons in field:value expressions, ignoring quoted strings and handling
 * logical operators (AND, OR, NOT) and parentheses.
 *
 * @param kql - The KQL query string to parse
 * @returns Array of unique property paths found in the query
 *
 * @example
 * ```typescript
 * extractPropertyPathsFromKql('foo.bar:this and steps.analysis:foo')
 * // Returns: ['foo.bar', 'steps.analysis']
 *
 * extractPropertyPathsFromKql('name:"John Doe" and age:30 or status:active')
 * // Returns: ['name', 'age', 'status']
 * ```
 */
function extractPropertyPathsFromKql(kql) {
  if (!kql || typeof kql !== 'string') {
    return [];
  }
  const ast = (0, _esQuery.fromKueryExpression)(kql);
  const accumulator = new Set();
  visitRecursive(ast, accumulator);
  // Filter out special KQL wildcards and sort the results
  return Array.from(accumulator);
}
function visitRecursive(node, accumulator) {
  if (node.type === 'function') {
    const functionNode = node;
    switch (functionNode.function) {
      case 'and':
        functionNode.arguments.forEach(arg => visitRecursive(arg, accumulator));
        break;
      case 'or':
        functionNode.arguments.forEach(arg => visitRecursive(arg, accumulator));
        break;
      case 'not':
        visitRecursive(functionNode.arguments[0], accumulator);
        break;
      case 'range':
        const [leftRangeLiteral] = functionNode.arguments;
        if (!leftRangeLiteral.value) {
          break;
        }
        accumulator.add(String(leftRangeLiteral.value));
        break;
      case 'is':
        const [leftLiteral, _] = node.arguments;
        if (!leftLiteral.value) {
          break;
        }
        accumulator.add(String(leftLiteral.value));
        break;
    }
  }
}