"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkflowsManagementApi = void 0;
var _i18n = require("@kbn/i18n");
var _workflows = require("@kbn/workflows");
var _errors = require("../../common/lib/errors");
var _validate_step_names = require("../../common/lib/validate_step_names");
var _yaml_utils = require("../../common/lib/yaml_utils");
var _schema = require("../../common/schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class WorkflowsManagementApi {
  constructor(workflowsService, getWorkflowsExecutionEngine) {
    this.workflowsService = workflowsService;
    this.getWorkflowsExecutionEngine = getWorkflowsExecutionEngine;
  }
  async getWorkflows(params, spaceId) {
    return await this.workflowsService.getWorkflows(params, spaceId);
  }
  async getWorkflow(id, spaceId) {
    return await this.workflowsService.getWorkflow(id, spaceId);
  }
  async createWorkflow(workflow, spaceId, request) {
    return await this.workflowsService.createWorkflow(workflow, spaceId, request);
  }
  async cloneWorkflow(workflow, spaceId, request) {
    // Parse and update the YAML to change the name
    const parsedYaml = (0, _yaml_utils.parseWorkflowYamlToJSON)(workflow.yaml, _schema.WORKFLOW_ZOD_SCHEMA_LOOSE);
    if (parsedYaml.error) {
      throw parsedYaml.error;
    }
    const updatedYaml = {
      ...parsedYaml.data,
      name: `${workflow.name} ${_i18n.i18n.translate('workflowsManagement.cloneSuffix', {
        defaultMessage: 'Copy'
      })}`
    };

    // Convert back to YAML string
    const clonedYaml = `name: ${updatedYaml.name}\n${workflow.yaml.split('\n').slice(1).join('\n')}`;
    return await this.workflowsService.createWorkflow({
      yaml: clonedYaml
    }, spaceId, request);
  }
  async updateWorkflow(id, workflow, spaceId, request) {
    const originalWorkflow = await this.workflowsService.getWorkflow(id, spaceId);
    if (!originalWorkflow) {
      throw new Error(`Workflow with id ${id} not found`);
    }
    return await this.workflowsService.updateWorkflow(id, workflow, spaceId, request);
  }
  async deleteWorkflows(workflowIds, spaceId, request) {
    return await this.workflowsService.deleteWorkflows(workflowIds, spaceId);
  }
  async runWorkflow(workflow, spaceId, inputs, request) {
    const {
      event,
      ...manualInputs
    } = inputs;
    const context = {
      event,
      spaceId,
      inputs: manualInputs
    };
    const workflowsExecutionEngine = await this.getWorkflowsExecutionEngine();
    const executeResponse = await workflowsExecutionEngine.executeWorkflow(workflow, context, request);
    return executeResponse.workflowExecutionId;
  }
  async testWorkflow(workflowYaml, inputs, spaceId, request) {
    const parsedYaml = (0, _yaml_utils.parseWorkflowYamlToJSON)(workflowYaml, (0, _schema.getWorkflowZodSchemaLoose)());
    if (parsedYaml.error) {
      // TODO: handle error properly
      // It should throw BadRequestError in the API
      throw parsedYaml.error;
    }

    // Validate step name uniqueness
    const stepValidation = (0, _validate_step_names.validateStepNameUniqueness)(parsedYaml.data);
    if (!stepValidation.isValid) {
      const errorMessages = stepValidation.errors.map(error => error.message);
      throw new _errors.WorkflowValidationError('Workflow validation failed: Step names must be unique throughout the workflow.', errorMessages);
    }
    const workflowToCreate = (0, _workflows.transformWorkflowYamlJsontoEsWorkflow)(parsedYaml.data);
    const context = {
      ...inputs,
      spaceId
    };
    const workflowsExecutionEngine = await this.getWorkflowsExecutionEngine();
    const executeResponse = await workflowsExecutionEngine.executeWorkflow({
      id: 'test-workflow',
      name: workflowToCreate.name,
      enabled: workflowToCreate.enabled,
      definition: workflowToCreate.definition,
      yaml: workflowYaml,
      isTestRun: true
    }, context, request);
    return executeResponse.workflowExecutionId;
  }
  async testStep(workflowYaml, stepId, contextOverride, spaceId) {
    const parsedYaml = (0, _yaml_utils.parseWorkflowYamlToJSON)(workflowYaml, (0, _schema.getWorkflowZodSchemaLoose)());
    if (parsedYaml.error) {
      throw parsedYaml.error;
    }
    const workflowToCreate = (0, _workflows.transformWorkflowYamlJsontoEsWorkflow)(parsedYaml.data);
    const workflowsExecutionEngine = await this.getWorkflowsExecutionEngine();
    const executeResponse = await workflowsExecutionEngine.executeWorkflowStep({
      id: 'test-workflow',
      name: workflowToCreate.name,
      enabled: workflowToCreate.enabled,
      definition: workflowToCreate.definition,
      yaml: workflowYaml,
      isTestRun: true,
      spaceId
    }, stepId, contextOverride);
    return executeResponse.workflowExecutionId;
  }
  async getWorkflowExecutions(params, spaceId) {
    return await this.workflowsService.getWorkflowExecutions(params, spaceId);
  }
  async getWorkflowExecution(workflowExecutionId, spaceId) {
    return await this.workflowsService.getWorkflowExecution(workflowExecutionId, spaceId);
  }
  async getWorkflowExecutionLogs(params, spaceId) {
    let result;
    if (params.stepExecutionId) {
      result = await this.workflowsService.getStepLogs({
        executionId: params.executionId,
        stepExecutionId: params.stepExecutionId,
        limit: params.limit,
        offset: params.offset,
        sortField: params.sortField,
        sortOrder: params.sortOrder
      }, spaceId);
    } else {
      result = await this.workflowsService.getExecutionLogs(params, spaceId);
    }

    // Transform the logs to match our API format
    return {
      logs: result.logs.filter(log => log) // Filter out undefined/null logs
      .map(log => {
        var _log$workflow, _log$workflow2, _log$workflow3, _log$workflow4, _log$workflow5, _log$event, _log$workflow6, _log$workflow7, _log$workflow8;
        return {
          id: log.id || `${log['@timestamp']}-${(_log$workflow = log.workflow) === null || _log$workflow === void 0 ? void 0 : _log$workflow.execution_id}-${((_log$workflow2 = log.workflow) === null || _log$workflow2 === void 0 ? void 0 : _log$workflow2.step_id) || 'workflow'}`,
          timestamp: log['@timestamp'],
          level: log.level,
          message: log.message,
          stepId: (_log$workflow3 = log.workflow) === null || _log$workflow3 === void 0 ? void 0 : _log$workflow3.step_id,
          stepName: (_log$workflow4 = log.workflow) === null || _log$workflow4 === void 0 ? void 0 : _log$workflow4.step_name,
          connectorType: (_log$workflow5 = log.workflow) === null || _log$workflow5 === void 0 ? void 0 : _log$workflow5.step_type,
          duration: (_log$event = log.event) === null || _log$event === void 0 ? void 0 : _log$event.duration,
          additionalData: {
            workflowId: (_log$workflow6 = log.workflow) === null || _log$workflow6 === void 0 ? void 0 : _log$workflow6.id,
            workflowName: (_log$workflow7 = log.workflow) === null || _log$workflow7 === void 0 ? void 0 : _log$workflow7.name,
            executionId: (_log$workflow8 = log.workflow) === null || _log$workflow8 === void 0 ? void 0 : _log$workflow8.execution_id,
            event: log.event,
            tags: log.tags,
            error: log.error
          }
        };
      }),
      total: result.total,
      limit: params.limit || 100,
      offset: params.offset || 0
    };
  }
  async getStepExecution(params, spaceId) {
    return await this.workflowsService.getStepExecution(params, spaceId);
  }
  async cancelWorkflowExecution(workflowExecutionId, spaceId) {
    const workflowsExecutionEngine = await this.getWorkflowsExecutionEngine();
    return await workflowsExecutionEngine.cancelWorkflowExecution(workflowExecutionId, spaceId);
  }
  async getWorkflowStats(spaceId) {
    return await this.workflowsService.getWorkflowStats(spaceId);
  }
  async getWorkflowAggs(fields = [], spaceId) {
    return await this.workflowsService.getWorkflowAggs(fields, spaceId);
  }
}
exports.WorkflowsManagementApi = WorkflowsManagementApi;