"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useYamlValidation = useYamlValidation;
var _monaco = require("@kbn/monaco");
var _react = require("react");
var _yaml = require("yaml");
var _graph = require("@kbn/workflows/graph");
var _parse_variable_path = require("../../../../common/lib/parse_variable_path");
var _yaml_utils = require("../../../../common/lib/yaml_utils");
var _regex = require("../../../../common/lib/regex");
var _zod = require("../../../../common/lib/zod");
var _get_context_for_path = require("../../../features/workflow_context/lib/get_context_for_path");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const SEVERITY_MAP = {
  error: _utils.MarkerSeverity.Error,
  warning: _utils.MarkerSeverity.Warning,
  info: _utils.MarkerSeverity.Hint
};
const collectAllStepNames = yamlDocument => {
  const stepNames = [];
  if (!(yamlDocument !== null && yamlDocument !== void 0 && yamlDocument.contents)) return stepNames;
  (0, _yaml.visit)(yamlDocument, {
    Scalar(key, node, ancestors) {
      if (!node.range) {
        return;
      }
      const lastAncestor = ancestors === null || ancestors === void 0 ? void 0 : ancestors[ancestors.length - 1];
      const isNameProp = (0, _yaml.isPair)(lastAncestor) && (0, _yaml.isScalar)(lastAncestor.key) && lastAncestor.key.value === 'name';
      if (!isNameProp || !node.value) {
        return;
      }

      // Make sure we're looking at the VALUE of a name property, not the key itself
      // The key "name" will also be a scalar, but it will be the key of the pair, not the value
      const isNameValue = (0, _yaml.isPair)(lastAncestor) && lastAncestor.value === node;
      if (!isNameValue) {
        return;
      }

      // Use the same logic as getStepNode to identify step names
      const path = (0, _yaml_utils.getPathFromAncestors)(ancestors);
      const isInSteps = path.length >= 3 && (path[path.length - 3] === 'steps' || path[path.length - 3] === 'else');
      if (isInSteps) {
        const [startOffset, endOffset] = node.range;

        // Convert byte offsets to line/column positions
        const text = yamlDocument.toString();
        let line = 1;
        let column = 1;
        let startLine = 1;
        let startCol = 1;
        let endLine = 1;
        let endCol = 1;
        for (let i = 0; i < text.length; i++) {
          if (i === startOffset) {
            startLine = line;
            startCol = column;
          }
          if (i === endOffset) {
            endLine = line;
            endCol = column;
            break;
          }
          if (text[i] === '\n') {
            line++;
            column = 1;
          } else {
            column++;
          }
        }
        stepNames.push({
          name: node.value,
          node,
          startLineNumber: startLine,
          startColumn: startCol,
          endLineNumber: endLine,
          endColumn: endCol
        });
      }
    }
  });
  return stepNames;
};
function useYamlValidation({
  workflowYamlSchema,
  onValidationErrors
}) {
  const [error, setError] = (0, _react.useState)(null);
  const [validationErrors, setValidationErrors] = (0, _react.useState)(null);
  const decorationsCollection = (0, _react.useRef)(null);

  // Function to validate mustache expressions and apply decorations
  const validateVariables = (0, _react.useCallback)(editor => {
    const model = editor.getModel();
    if (!model) {
      return;
    }
    if ('original' in model) {
      // TODO: validate diff editor
      return;
    }
    editor = editor;
    const decorations = [];
    try {
      var _workflowGraph$getAll;
      const text = model.getValue();

      // Parse the YAML to JSON to get the workflow definition
      const result = (0, _yaml_utils.parseWorkflowYamlToJSON)(text, workflowYamlSchema);
      const yamlDocument = (0, _yaml.parseDocument)(text);
      const workflowGraph = result.success ? _graph.WorkflowGraph.fromWorkflowDefinition(result.data) : null;

      // Collect markers to add to the model
      const markers = [];

      // Validate step name uniqueness
      const stepNames = collectAllStepNames(yamlDocument);
      const stepNameCounts = new Map();

      // Group step names by their values
      for (const stepInfo of stepNames) {
        const existing = stepNameCounts.get(stepInfo.name);
        if (existing) {
          existing.push(stepInfo);
        } else {
          stepNameCounts.set(stepInfo.name, [stepInfo]);
        }
      }

      // Add markers for duplicate step names
      for (const [stepName, occurrences] of stepNameCounts) {
        if (occurrences.length > 1) {
          for (const occurrence of occurrences) {
            markers.push({
              severity: SEVERITY_MAP.error,
              message: `Step name "${stepName}" is not unique. Found ${occurrences.length} steps with this name.`,
              startLineNumber: occurrence.startLineNumber,
              startColumn: occurrence.startColumn,
              endLineNumber: occurrence.endLineNumber,
              endColumn: occurrence.endColumn,
              source: 'step-name-validation'
            });

            // Add full line highlighting with red background
            decorations.push({
              range: new _monaco.monaco.Range(occurrence.startLineNumber, 1, occurrence.startLineNumber, model.getLineMaxColumn(occurrence.startLineNumber)),
              options: {
                className: 'duplicate-step-name-error',
                marginClassName: 'duplicate-step-name-error-margin',
                isWholeLine: true,
                stickiness: _monaco.monaco.editor.TrackedRangeStickiness.NeverGrowsWhenTypingAtEdges
              }
            });
          }
        }
      }

      // Currently, foreach doesn't use mustache expressions, so we need to handle it separately
      // TODO: remove if/when foreach uses mustache expressions
      const foreachVariableItems = ((_workflowGraph$getAll = workflowGraph === null || workflowGraph === void 0 ? void 0 : workflowGraph.getAllNodes()) !== null && _workflowGraph$getAll !== void 0 ? _workflowGraph$getAll : []).map(node => {
        if ((0, _graph.isEnterForeach)(node)) {
          const yamlNode = (0, _yaml_utils.getStepNode)(yamlDocument, node.stepId);
          const foreachValueNode = yamlNode === null || yamlNode === void 0 ? void 0 : yamlNode.get('foreach', true);
          if (foreachValueNode && foreachValueNode.range) {
            return {
              start: foreachValueNode.range[0],
              end: foreachValueNode.range[2],
              key: node.configuration.foreach
            };
          }
        }
        return null;
      }).filter(foreach => foreach);
      const textMatches = [...text.matchAll(_regex.VARIABLE_REGEX_GLOBAL)].map(match => {
        var _match$index, _match$index2, _match$groups$key, _match$groups;
        return {
          start: (_match$index = match.index) !== null && _match$index !== void 0 ? _match$index : 0,
          end: ((_match$index2 = match.index) !== null && _match$index2 !== void 0 ? _match$index2 : 0) + match[0].length,
          // match[0] is the entire {{...}} expression
          key: (_match$groups$key = (_match$groups = match.groups) === null || _match$groups === void 0 ? void 0 : _match$groups.key) !== null && _match$groups$key !== void 0 ? _match$groups$key : null
        };
      });
      const variableItems = [...textMatches, ...foreachVariableItems];
      // TODO: check if the variable is inside quouted string or yaml | or > string section
      for (const variableItem of variableItems) {
        const {
          start,
          end,
          key
        } = variableItem;
        // Get the position (line, column) for the match
        const startPos = model.getPositionAt(start);
        const endPos = model.getPositionAt(end);
        let errorMessage = null;
        let hoverMessage = null;
        let severity = 'error';
        const path = (0, _yaml_utils.getCurrentPath)(yamlDocument, start);
        let context = null;
        if (result.success) {
          try {
            context = (0, _get_context_for_path.getContextSchemaForPath)(result.data, workflowGraph, path);
          } catch (e) {
            // Fallback to the main workflow schema if context detection fails
            context = null;
            errorMessage = e.message;
          }
        }
        if (!key) {
          errorMessage = `Variable is not defined`;
        } else {
          const parsedPath = (0, _parse_variable_path.parseVariablePath)(key);
          if (!parsedPath) {
            errorMessage = `Invalid variable path: ${key}`;
          } else if (parsedPath.errors) {
            errorMessage = parsedPath.errors.join(', ');
          } else if (parsedPath !== null && parsedPath !== void 0 && parsedPath.propertyPath && !errorMessage) {
            if (!context) {
              severity = 'warning';
              errorMessage = `Variable ${parsedPath.propertyPath} cannot be validated, because the workflow schema is invalid`;
            } else {
              const refSchema = (0, _zod.getSchemaAtPath)(context, parsedPath.propertyPath);
              if (!refSchema) {
                errorMessage = `Variable ${parsedPath.propertyPath} is invalid`;
              } else if ((0, _zod.getZodTypeName)(refSchema) === 'unknown') {
                hoverMessage = `<pre>(property) ${parsedPath.propertyPath}: ${(0, _zod.getDetailedTypeDescription)(refSchema)}</pre>`;
                severity = 'warning';
                errorMessage = `Variable ${parsedPath.propertyPath} cannot be validated, because it's type is unknown`;
              }
            }
          }
        }

        // Add marker for validation issues
        if (errorMessage) {
          markers.push({
            severity: SEVERITY_MAP[severity],
            message: errorMessage,
            startLineNumber: startPos.lineNumber,
            startColumn: startPos.column,
            endLineNumber: endPos.lineNumber,
            endColumn: endPos.column,
            source: 'variable-validation'
          });
          decorations.push({
            range: new _monaco.monaco.Range(startPos.lineNumber, startPos.column, endPos.lineNumber, endPos.column),
            options: {
              inlineClassName: 'template-variable-error',
              stickiness: _monaco.monaco.editor.TrackedRangeStickiness.NeverGrowsWhenTypingAtEdges,
              hoverMessage: hoverMessage ? createMarkdownContent(hoverMessage) : null
            }
          });
        } else {
          decorations.push({
            range: new _monaco.monaco.Range(startPos.lineNumber, startPos.column, endPos.lineNumber, endPos.column),
            options: {
              inlineClassName: 'template-variable-valid',
              stickiness: _monaco.monaco.editor.TrackedRangeStickiness.NeverGrowsWhenTypingAtEdges,
              hoverMessage: hoverMessage ? createMarkdownContent(hoverMessage) : null
            }
          });
        }
      }
      if (decorationsCollection.current) {
        decorationsCollection.current.clear();
      }
      decorationsCollection.current = editor.createDecorationsCollection(decorations);

      // Set markers on the model for the problems panel
      _monaco.monaco.editor.setModelMarkers(model, 'variable-validation', markers.filter(m => m.source === 'variable-validation'));
      _monaco.monaco.editor.setModelMarkers(model, 'step-name-validation', markers.filter(m => m.source === 'step-name-validation'));
      setError(null);
    } catch (e) {
      setError(new Error('Error validating variables'));
    }
  }, [workflowYamlSchema]);
  const handleMarkersChanged = (0, _react.useCallback)((editor, modelUri, markers, owner) => {
    var _editor$getModel;
    const editorUri = (_editor$getModel = editor.getModel()) === null || _editor$getModel === void 0 ? void 0 : _editor$getModel.uri;
    if (modelUri.path !== (editorUri === null || editorUri === void 0 ? void 0 : editorUri.path)) {
      return;
    }
    const errors = [];
    for (const marker of markers) {
      let formattedMessage = marker.message;

      // Apply custom formatting to schema validation errors (from monaco-yaml)
      if (owner === 'yaml' && marker.message) {
        var _receivedValue;
        // Extract the actual value from the editor at the error position
        const model = editor.getModel();
        let receivedValue;
        if (model) {
          try {
            // Get the text at the error position
            const range = {
              startLineNumber: marker.startLineNumber,
              startColumn: marker.startColumn,
              endLineNumber: marker.endLineNumber || marker.startLineNumber,
              endColumn: marker.endColumn || marker.startColumn + 10 // fallback range
            };
            const textAtError = model.getValueInRange(range);

            // Try to extract the value (remove quotes if present)
            const valueMatch = textAtError.match(/^\s*([^:\s]+)/);
            if (valueMatch) {
              receivedValue = valueMatch[1].replace(/['"]/g, '');
            }
          } catch (e) {
            // Fallback to parsing the message
            receivedValue = extractReceivedValue(marker.message);
          }
        }

        // Create a mock error object that matches our formatter's expected structure
        const mockError = {
          message: marker.message,
          issues: [{
            code: marker.message.includes('Value must be') ? 'invalid_literal' : 'unknown',
            message: marker.message,
            path: ['type'],
            // Assume it's a type field error for now
            received: (_receivedValue = receivedValue) !== null && _receivedValue !== void 0 ? _receivedValue : ''
          }]
        };
        const {
          message
        } = (0, _yaml_utils.formatValidationError)(mockError, workflowYamlSchema);
        formattedMessage = message;
      }
      errors.push({
        message: formattedMessage,
        severity: (0, _utils.getSeverityString)(marker.severity),
        lineNumber: marker.startLineNumber,
        column: marker.startColumn,
        owner
      });
    }
    const errorsUpdater = prevErrors => {
      const prevOtherOwners = prevErrors === null || prevErrors === void 0 ? void 0 : prevErrors.filter(e => e.owner !== owner);
      return [...(prevOtherOwners !== null && prevOtherOwners !== void 0 ? prevOtherOwners : []), ...errors];
    };
    setValidationErrors(errorsUpdater);
    onValidationErrors === null || onValidationErrors === void 0 ? void 0 : onValidationErrors(errorsUpdater);
  }, [onValidationErrors, workflowYamlSchema]);

  // Helper function to extract the received value from Monaco's error message
  function extractReceivedValue(message) {
    // Try different patterns to extract the received value

    // Pattern 1: "Value must be one of: ... Received: 'value'"
    let receivedMatch = message.match(/Received:\s*['"]([^'"]+)['"]/);
    if (receivedMatch) {
      return receivedMatch[1];
    }

    // Pattern 2: "Value must be one of: ... Received: value" (without quotes)
    receivedMatch = message.match(/Received:\s*([^\s,]+)/);
    if (receivedMatch) {
      return receivedMatch[1];
    }

    // Pattern 3: Look for the actual value in the editor at the error position
    // This is more complex but might be needed if Monaco doesn't include the value in the message

    // For now, return undefined if we can't extract it
    return undefined;
  }
  return {
    error,
    validationErrors,
    validateVariables,
    handleMarkersChanged
  };
}
function createMarkdownContent(content) {
  return {
    value: content,
    isTrusted: true,
    supportHtml: true
  };
}