"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findAllSteps = findAllSteps;
exports.findStepAtPosition = findStepAtPosition;
exports.getStepRange = getStepRange;
exports.isPositionInStep = isPositionInStep;
var _monaco = require("@kbn/monaco");
var _yaml_utils = require("../../../../common/lib/yaml_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Unified step detection utility that finds all steps in a YAML document
 * Uses the same logic as the unified actions provider for consistency
 */
function findAllSteps(yamlDocument, model) {
  const steps = [];
  const processedSteps = new Set();
  if (!yamlDocument || !model) {
    return steps;
  }
  const totalLines = model.getLineCount();

  // Iterate through all lines looking for "type:" declarations
  for (let lineNumber = 1; lineNumber <= totalLines; lineNumber++) {
    const lineContent = model.getLineContent(lineNumber);

    // Look for "type:" lines that might be connector types
    if (lineContent.trim().startsWith('type:')) {
      const typeColonIndex = lineContent.indexOf('type:');
      const position = new _monaco.monaco.Position(lineNumber, typeColonIndex + 6); // After "type:"
      const absolutePosition = model.getOffsetAt(position);
      try {
        const yamlPath = (0, _yaml_utils.getCurrentPath)(yamlDocument, absolutePosition);

        // Check if this is within a step using the same logic as unified actions provider
        const stepsIndex = yamlPath.findIndex(segment => segment === 'steps');
        if (stepsIndex !== -1 && yamlPath[stepsIndex + 1] !== undefined) {
          const stepIndex = parseInt(String(yamlPath[stepsIndex + 1]), 10);
          if (!isNaN(stepIndex)) {
            const stepKey = `step_${stepIndex}`;

            // Skip if we've already processed this step
            if (processedSteps.has(stepKey)) {
              continue;
            }
            processedSteps.add(stepKey);

            // Get step node using the same path logic
            const stepPath = yamlPath.slice(0, stepsIndex + 2);
            const stepNode = yamlDocument.getIn(stepPath, true);
            if (stepNode) {
              var _get, _ref, _get2, _ref2, _get2$call;
              const typeNode = stepNode === null || stepNode === void 0 ? void 0 : (_get = (_ref = stepNode).get) === null || _get === void 0 ? void 0 : _get.call(_ref, 'type', true);
              const stepType = typeNode === null || typeNode === void 0 ? void 0 : typeNode.value;
              const stepName = (stepNode === null || stepNode === void 0 ? void 0 : (_get2 = (_ref2 = stepNode).get) === null || _get2 === void 0 ? void 0 : (_get2$call = _get2.call(_ref2, 'name')) === null || _get2$call === void 0 ? void 0 : _get2$call.value) || `step_${stepIndex}`;
              if (stepType) {
                steps.push({
                  stepIndex,
                  stepNode,
                  stepName,
                  stepType,
                  typeNode,
                  stepPath
                });
              }
            }
          }
        }
      } catch (error) {
        // Skip this position if there's an error
        continue;
      }
    }
  }
  return steps;
}

/**
 * Get the Monaco range for a step node
 * Uses the YAML parser's actual range for accuracy
 */
function getStepRange(stepNode, model) {
  try {
    if (!stepNode.range || !model) {
      return null;
    }
    const [startOffset,, endOffset] = stepNode.range;
    const startPos = model.getPositionAt(startOffset);
    const endPos = model.getPositionAt(endOffset);

    // Use the YAML node's actual range, but stop before the next step
    let adjustedEndLine = endPos.lineNumber;
    let adjustedEndColumn = endPos.column;

    // Walk backwards from endPos to find the last line that belongs to this step
    while (adjustedEndLine > startPos.lineNumber) {
      const lineContent = model.getLineContent(adjustedEndLine);
      const trimmedContent = lineContent.trim();

      // If this line starts with '-', it's the beginning of the next step
      if (trimmedContent.startsWith('-')) {
        // Use the previous line as the end
        adjustedEndLine--;
        if (adjustedEndLine >= startPos.lineNumber) {
          adjustedEndColumn = model.getLineMaxColumn(adjustedEndLine);
        }
        break;
      }

      // If this line has content and doesn't start with '-', use it as the end
      if (trimmedContent.length > 0) {
        adjustedEndColumn = model.getLineMaxColumn(adjustedEndLine);
        break;
      }

      // If it's empty, move to the previous line
      adjustedEndLine--;
    }

    // Safety check: ensure we don't go beyond the start line
    if (adjustedEndLine < startPos.lineNumber) {
      adjustedEndLine = startPos.lineNumber;
      adjustedEndColumn = model.getLineMaxColumn(adjustedEndLine);
    }
    return new _monaco.monaco.Range(startPos.lineNumber, startPos.column, adjustedEndLine, adjustedEndColumn);
  } catch (error) {
    // console.warn('getStepRange: Error calculating step range', error);
    return null;
  }
}

/**
 * Check if a position is within a step's range
 */
function isPositionInStep(position, stepRange) {
  return stepRange.containsPosition(position);
}

/**
 * Find the step that contains a given position
 */
function findStepAtPosition(position, steps, model) {
  for (const step of steps) {
    const stepRange = getStepRange(step.stepNode, model);
    if (stepRange && isPositionInStep(position, stepRange)) {
      return step;
    }
  }
  return null;
}