"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KibanaMonacoConnectorHandler = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _base_monaco_connector_handler = require("./base_monaco_connector_handler");
var _schema = require("../../../../../common/schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Cache for connectors (they don't change during runtime)
let allConnectorsCache = null;
function getCachedAllConnectors() {
  if (allConnectorsCache === null) {
    allConnectorsCache = (0, _schema.getAllConnectors)(); // Now uses lazy loading with require()
  }
  return allConnectorsCache;
}

/**
 * Monaco connector handler for Kibana APIs
 * Provides Monaco editor extensions (hover, actions, etc.) for Kibana connector types
 */
class KibanaMonacoConnectorHandler extends _base_monaco_connector_handler.BaseMonacoConnectorHandler {
  constructor(options = {}) {
    super('KibanaMonacoConnectorHandler', 90, ['kibana.']);
    (0, _defineProperty2.default)(this, "http", void 0);
    (0, _defineProperty2.default)(this, "notifications", void 0);
    // private readonly kibanaHost?: string;
    (0, _defineProperty2.default)(this, "connectorExamples", void 0);
    this.http = options.http;
    this.notifications = options.notifications;
    // this.kibanaHost = options.kibanaHost;
    this.connectorExamples = new Map();

    // Process generated connectors to extract examples
    if (options.generatedConnectors) {
      this.processGeneratedConnectors(options.generatedConnectors);
    }
  }

  /**
   * Generate hover content for Kibana connectors
   */
  async generateHoverContent(context) {
    try {
      const {
        connectorType,
        stepContext
      } = context;
      if (!stepContext) {
        return null;
      }

      // Get connector information from schema
      const connector = this.getConnectorInfo(connectorType);
      const withParams = this.extractWithParams(stepContext.stepNode);

      // Extract API information from connector type
      const apiInfo = this.parseKibanaConnectorType(connectorType, connector);
      if (!apiInfo) {
        return null;
      }

      // Get documentation URL from connector definition
      const documentationUrl = this.getDocumentationUrl(connectorType);

      // Get examples for this connector
      const examples = this.getExamples(connectorType);

      // Generate request example
      const requestExample = this.generateRequestExample(apiInfo, withParams);

      // Create enhanced hover content with enhanced formatting and shadowed icons
      const content = [`**Endpoint**: \`${apiInfo.method} ${apiInfo.path}\``, '', `**Description**: ${apiInfo.description || `Execute ${apiInfo.method} request to ${apiInfo.path}`}`, '', documentationUrl ? `<span style="text-shadow: 0 0 6px rgba(255,165,0,0.6); opacity: 0.8;">📖</span> **[View API Documentation](${documentationUrl})** - Opens in new tab` : '', documentationUrl ? '' : '', `### <span style="text-shadow: 0 0 4px rgba(0,200,0,0.4); opacity: 0.8;">⚡</span> Request Example`, '```http', requestExample, '```', '', examples ? this.generateKibanaApiDocumentation(apiInfo, examples) : '', examples ? '' : '', '---', `_<span style="text-shadow: 0 0 3px rgba(255,255,0,0.4); opacity: 0.7;">💡</span> Use Ctrl+Space for parameter autocomplete_`].filter(line => line !== null).join('\n');
      return this.createMarkdownContent(content);
    } catch (error) {
      // console.warn('KibanaMonacoConnectorHandler: Error generating hover content', error);
      return null;
    }
  }

  /**
   * Generate floating action buttons for Kibana connectors
   */
  async generateActions(context) {
    const actions = [];

    // Add actions if we have the necessary services
    if (this.http && this.notifications) {
      actions.push(this.createActionInfo('copy-step', 'Copy Step', () => this.copyStep(context), {
        icon: 'copy',
        tooltip: 'Copy workflow step',
        priority: 12
      }));
    }
    return actions;
  }

  /**
   * Get examples for Kibana connector types
   */
  getExamples(connectorType) {
    return this.connectorExamples.get(connectorType) || null;
  }

  /**
   * Process generated connectors to extract examples
   */
  processGeneratedConnectors(generatedConnectors) {
    for (const connector of generatedConnectors) {
      if (connector.examples) {
        this.connectorExamples.set(connector.type, connector.examples);
      }
    }
    // console.log(
    //   `KibanaMonacoConnectorHandler: Processed ${this.connectorExamples.size} connector examples`
    // );
  }

  /**
   * Parse Kibana connector type to extract API information
   */
  parseKibanaConnectorType(connectorType, connector) {
    var _connector$methods, _connector$patterns;
    // Extract operation ID from connector type (kibana.operationId)
    const operationId = connectorType.replace('kibana.', '');

    // Use connector information if available
    const method = (connector === null || connector === void 0 ? void 0 : (_connector$methods = connector.methods) === null || _connector$methods === void 0 ? void 0 : _connector$methods[0]) || 'POST';
    const path = (connector === null || connector === void 0 ? void 0 : (_connector$patterns = connector.patterns) === null || _connector$patterns === void 0 ? void 0 : _connector$patterns[0]) || `/api/${operationId.replace(/([A-Z])/g, '/$1').toLowerCase()}`;
    const description = (connector === null || connector === void 0 ? void 0 : connector.description) || `Kibana ${operationId} API endpoint`;
    return {
      method,
      path,
      operationId,
      description
    };
  }

  /**
   * Generate API documentation section
   */
  generateKibanaApiDocumentation(apiInfo, examples) {
    const lines = ['**API Information:**', `- **Method**: ${apiInfo.method}`, `- **Path**: ${apiInfo.path}`, `- **Operation**: ${apiInfo.operationId}`];
    if (examples !== null && examples !== void 0 && examples.params) {
      lines.push('', '**Example Parameters:**');
      lines.push('```yaml');
      for (const [key, value] of Object.entries(examples.params)) {
        lines.push(`${key}: ${typeof value === 'string' ? `"${value}"` : JSON.stringify(value, null, 2)}`);
      }
      lines.push('```');
    }
    if (examples !== null && examples !== void 0 && examples.snippet) {
      lines.push('', '**Example Usage:**');
      lines.push('```yaml');
      lines.push(examples.snippet);
      lines.push('```');
    }
    return lines.join('\n');
  }

  /**
   * Get connector information from schema
   */
  getConnectorInfo(connectorType) {
    try {
      const allConnectors = getCachedAllConnectors();
      if (!allConnectors) return null;
      return allConnectors.find(c => c.type === connectorType);
    } catch (error) {
      // console.warn('KibanaMonacoConnectorHandler: Error getting connector info', error);
      return null;
    }
  }

  /**
   * Get documentation URL for the connector type
   */
  getDocumentationUrl(connectorType) {
    try {
      const connector = this.getConnectorInfo(connectorType);
      if (connector !== null && connector !== void 0 && connector.documentation) {
        // Similar to Console, replace version placeholders with current version
        let docUrl = connector.documentation;

        // Replace common version placeholders with 'current' for stable links
        docUrl = docUrl.replace('/master/', '/current/').replace('/{branch}/', '/current/');
        return docUrl;
      }

      // Fallback to generic Kibana API docs
      return 'https://www.elastic.co/guide/en/kibana/current/api.html';
    } catch (error) {
      // console.warn('KibanaMonacoConnectorHandler: Error getting documentation URL', error);
      return null;
    }
  }

  /**
   * Generate request example
   */
  generateRequestExample(apiInfo, withParams) {
    const lines = [`${apiInfo.method} ${apiInfo.path}`, 'Content-Type: application/json', 'kbn-xsrf: true'];
    if (Object.keys(withParams).length > 0) {
      lines.push('');
      lines.push(JSON.stringify(withParams, null, 2));
    }
    return lines.join('\n');
  }

  /**
   * Extract 'with' parameters from step node
   */
  extractWithParams(stepNode) {
    const withParams = {};
    try {
      if (stepNode !== null && stepNode !== void 0 && stepNode.items) {
        for (const item of stepNode.items) {
          var _item$key, _item$value;
          if (((_item$key = item.key) === null || _item$key === void 0 ? void 0 : _item$key.value) === 'with' && (_item$value = item.value) !== null && _item$value !== void 0 && _item$value.items) {
            for (const withItem of item.value.items) {
              var _withItem$key;
              if ((_withItem$key = withItem.key) !== null && _withItem$key !== void 0 && _withItem$key.value) {
                var _withItem$value, _withItem$value2, _withItem$value2$toJS, _withItem$value3;
                const key = withItem.key.value;
                const value = ((_withItem$value = withItem.value) === null || _withItem$value === void 0 ? void 0 : _withItem$value.value) || ((_withItem$value2 = withItem.value) === null || _withItem$value2 === void 0 ? void 0 : (_withItem$value2$toJS = (_withItem$value3 = _withItem$value2).toJSON) === null || _withItem$value2$toJS === void 0 ? void 0 : _withItem$value2$toJS.call(_withItem$value3)) || withItem.value;
                withParams[key] = value;
              }
            }
            break;
          }
        }
      }
    } catch (error) {
      // console.warn('KibanaMonacoConnectorHandler: Error extracting with params', error);
    }
    return withParams;
  }

  /**
   * Copy entire workflow step
   */
  async copyStep(context) {
    try {
      var _stepContext$stepNode;
      const {
        stepContext
      } = context;
      if (!stepContext) return;

      // Get the entire step YAML
      const stepYaml = ((_stepContext$stepNode = stepContext.stepNode) === null || _stepContext$stepNode === void 0 ? void 0 : _stepContext$stepNode.toString()) || '';
      await navigator.clipboard.writeText(stepYaml);
      if (this.notifications) {
        this.notifications.toasts.addSuccess({
          title: 'Copied to clipboard',
          text: 'Workflow step copied successfully'
        });
      }
    } catch (error) {
      // console.error('KibanaMonacoConnectorHandler: Error copying step', error);
      if (this.notifications) {
        this.notifications.toasts.addError(error, {
          title: 'Failed to copy step'
        });
      }
    }
  }
}
exports.KibanaMonacoConnectorHandler = KibanaMonacoConnectorHandler;