"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ElasticsearchMonacoConnectorHandler = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _base_monaco_connector_handler = require("./base_monaco_connector_handler");
var _elasticsearch_step_utils = require("../elasticsearch_step_utils");
var _schema = require("../../../../../common/schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Cache for connectors (they don't change during runtime)
let allConnectorsCache = null;
function getCachedAllConnectors() {
  if (allConnectorsCache === null) {
    allConnectorsCache = (0, _schema.getAllConnectors)(); // Now uses lazy loading with require()
  }
  return allConnectorsCache;
}

/**
 * Monaco connector handler for Elasticsearch APIs
 * Provides Monaco editor extensions (hover, actions, etc.) for Elasticsearch connector types
 */
class ElasticsearchMonacoConnectorHandler extends _base_monaco_connector_handler.BaseMonacoConnectorHandler {
  // private readonly esHost?: string;
  // private readonly kibanaHost?: string;

  constructor(options = {}) {
    super('ElasticsearchMonacoConnectorHandler', 100, ['elasticsearch.']);
    (0, _defineProperty2.default)(this, "http", void 0);
    (0, _defineProperty2.default)(this, "notifications", void 0);
    this.http = options.http;
    this.notifications = options.notifications;
    // this.esHost = options.esHost;
    // this.kibanaHost = options.kibanaHost;
  }

  /**
   * Generate hover content for Elasticsearch connectors
   */
  async generateHoverContent(context) {
    try {
      const {
        connectorType,
        stepContext
      } = context;
      if (!stepContext) {
        return null;
      }

      // Get Elasticsearch request information
      const withParams = this.extractWithParams(stepContext.stepNode);
      const requestInfo = (0, _elasticsearch_step_utils.getElasticsearchRequestInfo)(connectorType, withParams);

      // Get documentation URL from connector definition
      const documentationUrl = this.getDocumentationUrl(connectorType);

      // Generate console format
      const consoleFormat = this.generateConsoleFormat(requestInfo, withParams);

      // Create hover content with enhanced formatting and shadowed icons
      const content = [`**Endpoint**: \`${requestInfo.method} ${requestInfo.url}\``, '', `**Description**: Execute ${requestInfo.method} request to ${requestInfo.url}`, '', documentationUrl ? `<span style="text-shadow: 0 0 6px rgba(255,165,0,0.6); opacity: 0.8;">📖</span> **[View API Documentation](${documentationUrl})** - Opens in new tab` : '', documentationUrl ? '' : '', `### <span style="text-shadow: 0 0 4px rgba(0,200,0,0.4); opacity: 0.8;">⚡</span> Console Format`, '```http', consoleFormat, '```', '', '', '---', `<span style="text-shadow: 0 0 3px rgba(255,255,0,0.4); opacity: 0.7;">💡</span> _Click Ctrl+Space (Ctrl+I on Mac) inside the "with:" block to see all available options_`].filter(line => line !== null).join('\n');
      return this.createMarkdownContent(content);
    } catch (error) {
      // console.warn('ElasticsearchMonacoConnectorHandler: Error generating hover content', error);
      return null;
    }
  }

  /**
   * Generate floating action buttons for Elasticsearch connectors
   */
  async generateActions(context) {
    const actions = [];

    // Add "Copy as Console" action if we have the necessary services
    if (this.http && this.notifications) {
      actions.push(this.createActionInfo('copy-as-console', 'Copy as Console', () => this.copyAsConsole(context), {
        icon: 'copy',
        tooltip: 'Copy this step as Console command',
        priority: 10
      }));
    }
    return actions;
  }

  /**
   * Get examples for Elasticsearch connector types
   */
  getExamples(connectorType) {
    // Extract API name for examples
    const apiName = connectorType.replace('elasticsearch.', '');

    // Common Elasticsearch examples
    const commonExamples = {
      search: {
        params: {
          index: 'logs-*',
          query: {
            range: {
              '@timestamp': {
                gte: 'now-1h'
              }
            }
          },
          size: 10
        },
        snippet: `- name: search_recent_logs
  type: elasticsearch.search
  with:
    index: "logs-*"
    query:
      range:
        "@timestamp":
          gte: "now-1h"
    size: 10`
      },
      'indices.create': {
        params: {
          index: 'my-new-index',
          body: {
            settings: {
              number_of_shards: 1,
              number_of_replicas: 0
            }
          }
        },
        snippet: `- name: create_index
  type: elasticsearch.indices.create
  with:
    index: "my-new-index"
    body:
      settings:
        number_of_shards: 1
        number_of_replicas: 0`
      },
      index: {
        params: {
          index: 'my-index',
          id: 'doc-1',
          body: {
            title: 'Example Document',
            content: 'This is an example document',
            timestamp: '{{ now }}'
          }
        },
        snippet: `- name: index_document
  type: elasticsearch.index
  with:
    index: "my-index"
    id: "doc-1"
    body:
      title: "Example Document"
      content: "This is an example document"
      timestamp: "{{ now }}"`
      }
    };
    return commonExamples[apiName] || null;
  }

  /**
   * Extract 'with' parameters from step node
   */
  extractWithParams(stepNode) {
    const withParams = {};
    try {
      if (stepNode !== null && stepNode !== void 0 && stepNode.items) {
        for (const item of stepNode.items) {
          var _item$key, _item$value;
          if (((_item$key = item.key) === null || _item$key === void 0 ? void 0 : _item$key.value) === 'with' && (_item$value = item.value) !== null && _item$value !== void 0 && _item$value.items) {
            for (const withItem of item.value.items) {
              var _withItem$key;
              if ((_withItem$key = withItem.key) !== null && _withItem$key !== void 0 && _withItem$key.value) {
                var _withItem$value, _withItem$value2, _withItem$value2$toJS, _withItem$value3;
                const key = withItem.key.value;
                const value = ((_withItem$value = withItem.value) === null || _withItem$value === void 0 ? void 0 : _withItem$value.value) || ((_withItem$value2 = withItem.value) === null || _withItem$value2 === void 0 ? void 0 : (_withItem$value2$toJS = (_withItem$value3 = _withItem$value2).toJSON) === null || _withItem$value2$toJS === void 0 ? void 0 : _withItem$value2$toJS.call(_withItem$value3)) || withItem.value;
                withParams[key] = value;
              }
            }
            break;
          }
        }
      }
    } catch (error) {
      // console.warn('ElasticsearchMonacoConnectorHandler: Error extracting with params', error);
    }
    return withParams;
  }

  /**
   * Generate console format for the request
   */
  generateConsoleFormat(requestInfo, withParams) {
    const lines = [`${requestInfo.method} ${requestInfo.url}`];
    if (requestInfo.data && requestInfo.data.length > 0) {
      lines.push(...requestInfo.data);
    }
    return lines.join('\n');
  }

  /**
   * Copy step as Console command
   */
  async copyAsConsole(context) {
    try {
      const {
        connectorType,
        stepContext
      } = context;
      if (!stepContext) return;
      const withParams = this.extractWithParams(stepContext.stepNode);
      const requestInfo = (0, _elasticsearch_step_utils.getElasticsearchRequestInfo)(connectorType, withParams);
      const consoleFormat = this.generateConsoleFormat(requestInfo, withParams);
      await navigator.clipboard.writeText(consoleFormat);
      if (this.notifications) {
        this.notifications.toasts.addSuccess({
          title: 'Copied to clipboard',
          text: 'Console command copied successfully'
        });
      }
    } catch (error) {
      // console.error('ElasticsearchMonacoConnectorHandler: Error copying as console', error);
      if (this.notifications) {
        this.notifications.toasts.addError(error, {
          title: 'Failed to copy'
        });
      }
    }
  }

  /**
   * Get documentation URL for the connector type
   */
  getDocumentationUrl(connectorType) {
    try {
      const allConnectors = getCachedAllConnectors();
      if (!allConnectors) return null;
      const connector = allConnectors.find(c => c.type === connectorType);
      if (connector !== null && connector !== void 0 && connector.documentation) {
        // Similar to Console, replace version placeholders with current version
        let docUrl = connector.documentation;

        // Replace common version placeholders with 'current' for stable links
        docUrl = docUrl.replace('/master/', '/current/').replace('/{branch}/', '/current/');
        return docUrl;
      }
      return null;
    } catch (error) {
      // console.warn('ElasticsearchMonacoConnectorHandler: Error getting documentation URL', error);
      return null;
    }
  }
}
exports.ElasticsearchMonacoConnectorHandler = ElasticsearchMonacoConnectorHandler;