"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRequiredParamsForConnector = getRequiredParamsForConnector;
var _zod = require("@kbn/zod");
var _connectors_cache = require("./connectors_cache");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Get required parameters for a connector type from generated schemas
 */
function getRequiredParamsForConnector(connectorType) {
  // Get all connectors (both static and generated)
  const allConnectors = (0, _connectors_cache.getCachedAllConnectors)();

  // Find the connector by type
  const connector = allConnectors.find(c => c.type === connectorType);
  if (connector && connector.paramsSchema) {
    try {
      var _examples;
      // Check if this connector has enhanced examples
      const hasEnhancedExamples = (_examples = connector.examples) === null || _examples === void 0 ? void 0 : _examples.params;

      // Processing enhanced examples for connector

      if (hasEnhancedExamples) {
        // Use examples directly from enhanced connector
        const exampleParams = connector.examples.params;
        // Using enhanced examples
        const result = [];
        for (const [key, value] of Object.entries(exampleParams)) {
          // Include common important parameters for ES APIs
          if (['index', 'id', 'body', 'query', 'size', 'from', 'sort', 'aggs', 'aggregations', 'format'].includes(key)) {
            result.push({
              name: key,
              example: value
            });
            // Added enhanced example
          }
        }
        if (result.length > 0) {
          // Returning enhanced examples
          return result;
        }
      }

      // Fallback to extracting from schema
      const params = extractRequiredParamsFromSchema(connector.paramsSchema);

      // Return only required parameters, or most important ones if no required ones
      const requiredParams = params.filter(p => p.required);
      if (requiredParams.length > 0) {
        return requiredParams.map(p => ({
          name: p.name,
          example: p.example
        }));
      }

      // If no required params, return the most important ones for ES APIs
      const importantParams = params.filter(p => ['index', 'id', 'body', 'query', 'size', 'from', 'sort', 'aggs', 'aggregations', 'format'].includes(p.name));
      if (importantParams.length > 0) {
        return importantParams.slice(0, 3).map(p => ({
          name: p.name,
          example: p.example
        }));
      }
    } catch (error) {
      // Silently continue with fallback parameters
    }
  }

  // Fallback to basic hardcoded ones for non-ES connectors
  const basicConnectorParams = {
    console: [{
      name: 'message',
      example: 'Hello World'
    }],
    slack: [{
      name: 'message',
      example: 'Hello Slack'
    }],
    http: [{
      name: 'url',
      example: 'https://api.example.com'
    }, {
      name: 'method',
      example: 'GET'
    }],
    wait: [{
      name: 'duration',
      example: '5s'
    }]
  };
  return basicConnectorParams[connectorType] || [];
}

/**
 * Extract example for body parameter based on its schema
 */
function extractBodyExample(bodySchema) {
  try {
    // Handle ZodOptional wrapper
    let schema = bodySchema;
    if (bodySchema instanceof _zod.z.ZodOptional) {
      schema = bodySchema._def.innerType;
    }

    // If it's a ZodObject, try to extract its shape and build YAML-compatible example
    if (schema instanceof _zod.z.ZodObject) {
      const shape = schema._def.shape();
      const example = {};

      // Extract examples from each field
      for (const [key, fieldSchema] of Object.entries(shape)) {
        var _def;
        const field = fieldSchema;
        const description = (field === null || field === void 0 ? void 0 : (_def = field._def) === null || _def === void 0 ? void 0 : _def.description) || '';

        // Extract example from description if available
        const stringExampleMatch = description.match(/e\.g\.,?\s*"([^"]+)"/);
        const objectExampleMatch = description.match(/e\.g\.,?\s*(\{[^}]+\})/);
        if (stringExampleMatch) {
          example[key] = stringExampleMatch[1];
        } else if (objectExampleMatch) {
          try {
            example[key] = JSON.parse(objectExampleMatch[1]);
          } catch {
            // If JSON parse fails, use as string
            example[key] = objectExampleMatch[1];
          }
        }
        // No fallback - only use examples explicitly defined in enhanced connectors
      }
      if (Object.keys(example).length > 0) {
        return example; // Return object, not JSON string
      }
    }
  } catch (error) {
    // Fallback to empty object
  }
  return {};
}

/**
 * Extract required parameters from a Zod schema
 */
function extractRequiredParamsFromSchema(schema) {
  const params = [];
  if (schema instanceof _zod.z.ZodObject) {
    const shape = schema.shape;
    for (const [key, fieldSchema] of Object.entries(shape)) {
      const zodField = fieldSchema;

      // Skip common non-parameter fields
      if (['pretty', 'human', 'error_trace', 'source', 'filter_path'].includes(key)) {
        continue;
      }

      // Check if field is required (not optional)
      const isOptional = zodField instanceof _zod.z.ZodOptional;
      const isRequired = !isOptional;

      // Extract description for examples
      let description = '';
      let example = '';
      if ('description' in zodField && typeof zodField.description === 'string') {
        description = zodField.description;
        // Try to extract example from description
        const exampleMatch = description.match(/example[:\s]+['"]*([^'"]+)['"]*|default[:\s]+['"]*([^'"]+)['"]*/i);
        if (exampleMatch) {
          example = exampleMatch[1] || exampleMatch[2] || '';
        }
      }

      // Add some default examples based on common parameter names
      if (!example) {
        if (key === 'index') {
          example = 'my-index';
        } else if (key === 'id') {
          example = 'doc-id';
        } else if (key === 'body') {
          // Try to extract body structure from schema
          example = extractBodyExample(zodField);
        } else if (key === 'query') {
          example = '{}';
        } else if (key.includes('name')) {
          example = 'my-name';
        }
      }

      // Only include required parameters or very common ones
      if (isRequired || ['index', 'id', 'body'].includes(key)) {
        params.push({
          name: key,
          example,
          required: isRequired
        });
      }
    }
  }
  return params;
}