"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findElasticsearchStepAtPosition = findElasticsearchStepAtPosition;
exports.getConsoleFormat = getConsoleFormat;
exports.getElasticsearchRequestInfo = getElasticsearchRequestInfo;
exports.getElasticsearchSteps = getElasticsearchSteps;
exports.isElasticsearchStep = isElasticsearchStep;
exports.stepToConsoleRequest = stepToConsoleRequest;
var _yaml = require("yaml");
var _workflows = require("@kbn/workflows");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Checks if a workflow step type starts with 'elasticsearch'
 * Accepts undefined or null as input.
 */
function isElasticsearchStep(stepType) {
  return typeof stepType === 'string' && stepType.startsWith('elasticsearch');
}

/**
 * Converts an Elasticsearch step type to HTTP method and URL using the EXACT same logic as the execution engine
 */
function getElasticsearchRequestInfo(stepType, withParams) {
  // Import and reuse the EXACT same logic as ElasticsearchActionStepImpl
  const {
    method,
    path,
    body
  } = (0, _workflows.buildRequestFromConnector)(stepType, withParams || {});
  return {
    method,
    url: path,
    data: body ? [JSON.stringify(body, null, 2)] : undefined
  };
}

/**
 * Extracts Elasticsearch step information from a YAML document
 */
function getElasticsearchSteps(yamlDocument) {
  const elasticsearchSteps = [];
  if (!(yamlDocument !== null && yamlDocument !== void 0 && yamlDocument.contents)) {
    // console.log('getElasticsearchSteps: No document contents');
    return elasticsearchSteps;
  }

  // console.log('getElasticsearchSteps: Starting YAML visit');

  (0, _yaml.visit)(yamlDocument, {
    Pair(key, pair, ancestors) {
      if (!pair.key || !(0, _yaml.isScalar)(pair.key) || pair.key.value !== 'type') {
        return;
      }

      // console.log('getElasticsearchSteps: Found type pair', pair.value);

      // Check if this is a type field within a step
      const path = ancestors.slice();
      let isStepType = false;

      // Walk up the ancestors to see if we're in a steps array
      for (let i = path.length - 1; i >= 0; i--) {
        const ancestor = path[i];
        if ((0, _yaml.isPair)(ancestor) && (0, _yaml.isScalar)(ancestor.key) && ancestor.key.value === 'steps') {
          isStepType = true;
          break;
        }
      }

      // console.log('getElasticsearchSteps: isStepType', isStepType);

      if (isStepType && (0, _yaml.isScalar)(pair.value)) {
        const stepType = pair.value.value;
        // console.log('getElasticsearchSteps: Found step type', stepType);

        if (isElasticsearchStep(stepType)) {
          // console.log('getElasticsearchSteps: Is Elasticsearch step', stepType);
          // Find the parent step node that contains this type
          const stepNode = ancestors[ancestors.length - 1];

          // Extract 'with' parameters from the step
          let withParams = {};
          if (stepNode && (0, _yaml.isMap)(stepNode)) {
            for (const item of stepNode.items) {
              if ((0, _yaml.isPair)(item) && (0, _yaml.isScalar)(item.key) && item.key.value === 'with') {
                if (item.value && (0, _yaml.isMap)(item.value)) {
                  withParams = {};
                  for (const withItem of item.value.items) {
                    if ((0, _yaml.isPair)(withItem) && (0, _yaml.isScalar)(withItem.key)) {
                      const paramKey = withItem.key.value;
                      // Handle both scalar values and nested objects
                      if ((0, _yaml.isScalar)(withItem.value)) {
                        withParams[paramKey] = withItem.value.value;
                      } else {
                        var _withItem$value, _withItem$value$toJSO, _withItem$value2;
                        // For nested objects (like query), convert the YAML node to a JS object
                        withParams[paramKey] = ((_withItem$value = withItem.value) === null || _withItem$value === void 0 ? void 0 : (_withItem$value$toJSO = (_withItem$value2 = _withItem$value).toJSON) === null || _withItem$value$toJSO === void 0 ? void 0 : _withItem$value$toJSO.call(_withItem$value2)) || withItem.value;
                      }
                    }
                  }
                }
                break;
              }
            }
          }

          // console.log('getElasticsearchSteps: With params', withParams);

          const requestInfo = getElasticsearchRequestInfo(stepType, withParams);

          // console.log('getElasticsearchSteps: Request info', requestInfo);

          elasticsearchSteps.push({
            type: stepType,
            method: requestInfo.method,
            url: requestInfo.url,
            data: requestInfo.data,
            stepNode,
            typeNode: pair
          });
        }
      }
    }
  });

  // console.log('getElasticsearchSteps: Found', elasticsearchSteps.length, 'elasticsearch steps');
  return elasticsearchSteps;
}

/**
 * Finds an Elasticsearch step at a specific Monaco position
 */
function findElasticsearchStepAtPosition(model, position, yamlDocument) {
  const elasticsearchSteps = getElasticsearchSteps(yamlDocument);

  /*
  console.log(
    'findElasticsearchStepAtPosition: Found',
    elasticsearchSteps.length,
    'elasticsearch steps'
  );
  */

  const offset = model.getOffsetAt(position);
  // console.log('findElasticsearchStepAtPosition: Position offset', offset);

  for (const step of elasticsearchSteps) {
    // Check if the position is within the step node range
    const stepRange = step.stepNode.range;
    // console.log('findElasticsearchStepAtPosition: Checking step', step.type, 'range', stepRange);
    if (stepRange && offset >= stepRange[0] && offset <= stepRange[1]) {
      // console.log('findElasticsearchStepAtPosition: Found matching step', step);
      return step;
    }
  }

  // console.log('findElasticsearchStepAtPosition: No matching step found');
  return null;
}

/**
 * Converts workflow step data to Console-compatible request format
 */
function stepToConsoleRequest(step) {
  return {
    method: step.method,
    url: step.url,
    data: step.data || []
  };
}

/**
 * Converts an Elasticsearch workflow step to Console format string
 */
function getConsoleFormat(step) {
  const request = stepToConsoleRequest(step);
  let consoleFormat = `${request.method} ${request.url}`;
  if (request.data && request.data.length > 0) {
    consoleFormat += '\n' + request.data.join('\n');
  }
  return consoleFormat;
}