"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JSONDataTable = JSONDataTable;
var _react = _interopRequireWildcard(require("react"));
var _eui = require("@elastic/eui");
var _use_memo_css = require("@kbn/css-utils/public/use_memo_css");
var _react2 = require("@emotion/react");
var _i18nReact = require("@kbn/i18n-react");
var _discoverUtils = require("@kbn/discover-utils");
var _table_field_value = require("./table_field_value");
var _kibana_flatten = require("../../lib/kibana_flatten");
var _field_name = require("./field_name");
var _infer_field_type = require("./infer_field_type");
var _format_value = require("./format_value");
var _use_formatted_date = require("../use_formatted_date");
var _jsxFileName = "/opt/buildkite-agent/builds/bk-agent-prod-gcp-1767906851655284711/elastic/kibana-artifacts-staging/kibana/src/platform/plugins/shared/workflows_management/public/shared/ui/json_data_view/json_data_table.tsx";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
const MIN_NAME_COLUMN_WIDTH = 120;
const MAX_NAME_COLUMN_WIDTH = 300;

/**
 * Props for the JSONDataTable component
 */

/**
 * JSONDataTable component that displays arbitrary JSON data using Kibana's unified data table.
 *
 * This component converts JSON objects into a format compatible with UnifiedDataTable,
 * providing a rich tabular view with support for nested objects, arrays, and various data types.
 *
 * @example
 * ```tsx
 * const jsonData = {
 *   id: '123',
 *   name: 'John Doe',
 *   email: 'john@example.com',
 *   metadata: { role: 'admin', lastLogin: '2024-01-15T10:30:00Z' }
 * };
 *
 * <JSONDataTable
 *   data={jsonData}
 *   title="User Data"
 *   columns={['id', 'name', 'email']}
 * />
 * ```
 */
function JSONDataTable({
  data: jsonObject,
  title = 'JSON Data',
  columns,
  searchTerm,
  'data-test-subj': dataTestSubj = 'jsonDataTable'
}) {
  const styles = (0, _use_memo_css.useMemoCss)(componentStyles);
  const containerRef = (0, _react.useRef)(null);
  const getFormattedDateTime = (0, _use_formatted_date.useGetFormattedDateTime)();

  // Create DataTableRecord from JSON - each field becomes a row
  const dataTableRecords = (0, _react.useMemo)(() => {
    // Flatten nested objects for better display

    const flattened = (0, _kibana_flatten.kibanaFlatten)(jsonObject);

    // Filter fields if columns prop is provided
    const fieldsToShow = columns ? Object.keys(flattened).filter(key => columns.includes(key)) : Object.keys(flattened);

    // Create a row for each field-value pair
    return fieldsToShow.map((fieldName, index) => {
      var _getFormattedDateTime;
      const value = flattened[fieldName];
      const fieldType = (0, _infer_field_type.inferFieldType)(value);
      const displayValue = fieldType === 'date' ? (_getFormattedDateTime = getFormattedDateTime(new Date(value))) !== null && _getFormattedDateTime !== void 0 ? _getFormattedDateTime : '' : (0, _format_value.formatValue)(value);
      return {
        id: `field-${index}`,
        raw: {
          _id: `field-${index}`,
          _index: title.toLowerCase().replace(/\s+/g, '_'),
          _source: {
            field: fieldName,
            value: displayValue,
            fieldType
          }
        },
        flattened: {
          field: fieldName,
          value: displayValue,
          fieldType // Store the field type for the cell renderer
        }
      };
    });
  }, [jsonObject, columns, getFormattedDateTime, title]);
  const filteredDataTableRecords = (0, _react.useMemo)(() => {
    return dataTableRecords.filter(record => {
      return record.flattened.field.toLowerCase().includes((searchTerm === null || searchTerm === void 0 ? void 0 : searchTerm.toLowerCase()) || '') || record.flattened.value.toLowerCase().includes((searchTerm === null || searchTerm === void 0 ? void 0 : searchTerm.toLowerCase()) || '');
    });
  }, [dataTableRecords, searchTerm]);
  const {
    width: containerWidth
  } = (0, _eui.useResizeObserver)(containerRef.current);
  const {
    curPageIndex,
    pageSize,
    changePageIndex,
    changePageSize
  } = (0, _discoverUtils.usePager)({
    initialPageSize: 20,
    totalItems: filteredDataTableRecords.length
  });

  // Grid columns configuration
  const gridColumns = (0, _react.useMemo)(() => [{
    id: 'name',
    displayAsText: 'Field',
    initialWidth: Math.min(Math.max(Math.round(containerWidth * 0.3), MIN_NAME_COLUMN_WIDTH), MAX_NAME_COLUMN_WIDTH),
    actions: false
  }, {
    id: 'value',
    displayAsText: 'Value',
    actions: false
  }], [containerWidth]);

  // Cell renderer for the data grid
  const renderCellValue = (0, _react.useMemo)(() => {
    return ({
      rowIndex,
      columnId
    }) => {
      const row = filteredDataTableRecords[rowIndex];
      if (!row) return null;
      if (columnId === 'name') {
        const fieldName = row.flattened.field;
        const fieldType = row.flattened.fieldType;
        return /*#__PURE__*/_react.default.createElement(_field_name.FieldName, {
          fieldName: fieldName,
          fieldType: fieldType,
          highlight: searchTerm,
          __self: this,
          __source: {
            fileName: _jsxFileName,
            lineNumber: 204,
            columnNumber: 16
          }
        });
      }
      if (columnId === 'value') {
        return /*#__PURE__*/_react.default.createElement(_table_field_value.TableFieldValue, {
          formattedValue: row.flattened.value,
          field: row.flattened.field,
          rawValue: row.flattened.value,
          isHighlighted: Boolean(searchTerm && row.flattened.value.toLowerCase().includes(searchTerm.toLowerCase())),
          __self: this,
          __source: {
            fileName: _jsxFileName,
            lineNumber: 209,
            columnNumber: 11
          }
        });
      }
      return null;
    };
  }, [filteredDataTableRecords, searchTerm]);
  if (filteredDataTableRecords.length === 0) {
    return /*#__PURE__*/_react.default.createElement(_eui.EuiEmptyPrompt, {
      title: /*#__PURE__*/_react.default.createElement("h2", {
        __self: this,
        __source: {
          fileName: _jsxFileName,
          lineNumber: 228,
          columnNumber: 11
        }
      }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "workflows.jsonDataTable.noData",
        defaultMessage: "No data to display",
        __self: this,
        __source: {
          fileName: _jsxFileName,
          lineNumber: 229,
          columnNumber: 13
        }
      })),
      iconType: "empty",
      __self: this,
      __source: {
        fileName: _jsxFileName,
        lineNumber: 226,
        columnNumber: 7
      }
    });
  }
  return /*#__PURE__*/_react.default.createElement(_eui.EuiDataGrid, {
    className: "kbnDocViewer__fieldsGrid",
    css: styles.fieldsGrid,
    "aria-label": title || 'JSON Data Table',
    columns: gridColumns,
    columnVisibility: {
      visibleColumns: ['name', 'value'],
      setVisibleColumns: () => {}
    },
    rowCount: filteredDataTableRecords.length,
    renderCellValue: renderCellValue,
    toolbarVisibility: false,
    sorting: {
      columns: [],
      onSort: () => {}
    },
    rowHeightsOptions: {
      defaultHeight: 'auto'
    },
    gridStyle: {
      header: 'underline',
      border: 'horizontal',
      fontSize: 's',
      stripes: true
    },
    pagination: {
      pageSizeOptions: [20, 50, 100, 200],
      pageIndex: curPageIndex,
      pageSize,
      onChangeItemsPerPage: changePageSize,
      onChangePage: changePageIndex
    },
    __self: this,
    __source: {
      fileName: _jsxFileName,
      lineNumber: 241,
      columnNumber: 5
    }
  });
}
const componentStyles = {
  fieldsGrid: themeContext => {
    const {
      euiTheme
    } = themeContext;
    const {
      fontSize
    } = (0, _eui.euiFontSize)(themeContext, 's');

    // taken from src/platform/plugins/shared/unified_doc_viewer/public/components/doc_viewer_table/table.tsx
    return (0, _react2.css)({
      '&.euiDataGrid--noControls.euiDataGrid--bordersHorizontal .euiDataGridHeader': {
        borderTop: 'none'
      },
      '&.euiDataGrid--headerUnderline .euiDataGridHeader': {
        borderBottom: euiTheme.border.thin
      },
      '& [data-gridcell-column-id="name"] .euiDataGridRowCell__content': {
        paddingTop: 0,
        paddingBottom: 0
      },
      '.kbnDocViewer__fieldName': {
        padding: euiTheme.size.xs,
        paddingLeft: 0,
        lineHeight: euiTheme.font.lineHeightMultiplier,
        '.euiDataGridRowCell__popover &': {
          fontSize
        }
      },
      '.kbnDocViewer__fieldName_icon': {
        paddingTop: `calc(${euiTheme.size.xs} * 1.5)`,
        lineHeight: euiTheme.font.lineHeightMultiplier
      }
    });
  }
};