"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.flowNodeTypes = void 0;
exports.getLayoutedNodesAndEdges = getLayoutedNodesAndEdges;
exports.transformYamlToNodesAndEdges = transformYamlToNodesAndEdges;
var _dagre = _interopRequireWildcard(require("@dagrejs/dagre"));
var _react = require("@xyflow/react");
var _graph_utils = require("../../../shared/lib/graph_utils");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const flowNodeTypes = exports.flowNodeTypes = ['if', 'merge', 'parallel', 'foreach', 'atomic', 'http', 'trigger'];
function transformYamlToNodesAndEdges(triggers, steps) {
  var _steps$;
  const nodes = [];
  const edges = [];
  const firstStepId = steps === null || steps === void 0 ? void 0 : (_steps$ = steps[0]) === null || _steps$ === void 0 ? void 0 : _steps$.name.toLowerCase().replace(/\s+/g, '-');
  for (const trigger of triggers) {
    const id = trigger.type.toLowerCase().replace(/\s+/g, '-');
    const name = trigger.type;
    nodes.push({
      id,
      label: name,
      type: 'trigger',
      data: {
        stepType: trigger.type,
        label: (0, _graph_utils.getTriggerLabel)(trigger.type)
      },
      style: {
        width: 250,
        height: 64
      }
    });
    edges.push({
      id: `${id}:${firstStepId}`,
      source: id,
      target: firstStepId
    });
  }
  for (let i = 0; i < steps.length; i++) {
    const step = steps[i];
    const id = step.name.toLowerCase().replace(/\s+/g, '-');
    const name = step.name;
    const nodeType = flowNodeTypes.includes(step.type) ? step.type : 'action';
    nodes.push({
      id,
      data: {
        label: name,
        stepType: step.type,
        step
      },
      type: nodeType,
      style: {
        width: 250,
        height: 64
      }
    });

    // Create edge to next step at the same level
    if (i < steps.length - 1) {
      const nextStep = steps[i + 1];
      const nextId = nextStep.name.toLowerCase().replace(/\s+/g, '-');
      edges.push({
        id: `${id}:${nextId}`,
        source: id,
        target: nextId
      });
    }

    // Handle recursive step types
    if (step.type === 'if' && 'steps' in step && step.steps) {
      const {
        nodes: ifNodes,
        edges: ifEdges
      } = transformYamlToNodesAndEdges([], step.steps);
      nodes.push(...ifNodes);
      edges.push(...ifEdges);

      // Create edge from if step to first nested step
      if (step.steps.length > 0) {
        const firstNestedId = step.steps[0].name.toLowerCase().replace(/\s+/g, '-');
        edges.push({
          id: `${id}:${firstNestedId}`,
          source: id,
          target: firstNestedId
        });
      }

      // Handle else branch if it exists
      if ('else' in step && step.else) {
        const {
          nodes: elseNodes,
          edges: elseEdges
        } = transformYamlToNodesAndEdges([], step.else);
        nodes.push(...elseNodes);
        edges.push(...elseEdges);

        // Create edge from if step to first else step
        if (step.else.length > 0) {
          const firstElseId = step.else[0].name.toLowerCase().replace(/\s+/g, '-');
          edges.push({
            id: `${id}:${firstElseId}-else`,
            source: id,
            target: firstElseId
          });
        }
      }
    }
    if (step.type === 'foreach' && 'steps' in step && step.steps) {
      const {
        nodes: foreachNodes,
        edges: foreachEdges
      } = transformYamlToNodesAndEdges([], step.steps);
      nodes.push(...foreachNodes);
      edges.push(...foreachEdges);

      // Create edge from foreach step to first nested step
      if (step.steps.length > 0) {
        const firstNestedId = step.steps[0].name.toLowerCase().replace(/\s+/g, '-');
        edges.push({
          id: `${id}:${firstNestedId}`,
          source: id,
          target: firstNestedId
        });
      }
    }
    if (step.type === 'atomic' && 'steps' in step && step.steps) {
      const {
        nodes: atomicNodes,
        edges: atomicEdges
      } = transformYamlToNodesAndEdges([], step.steps);
      nodes.push(...atomicNodes);
      edges.push(...atomicEdges);

      // Create edge from atomic step to first nested step
      if (step.steps.length > 0) {
        const firstNestedId = step.steps[0].name.toLowerCase().replace(/\s+/g, '-');
        edges.push({
          id: `${id}:${firstNestedId}`,
          source: id,
          target: firstNestedId
        });
      }
    }
    if (step.type === 'parallel' && 'branches' in step && step.branches) {
      for (const branch of step.branches) {
        const {
          nodes: branchNodes,
          edges: branchEdges
        } = transformYamlToNodesAndEdges([], branch.steps);
        nodes.push(...branchNodes);
        edges.push(...branchEdges);

        // Create edge from parallel step to first step in each branch
        if (branch.steps.length > 0) {
          const firstBranchId = branch.steps[0].name.toLowerCase().replace(/\s+/g, '-');
          edges.push({
            id: `${id}:${firstBranchId}`,
            source: id,
            target: firstBranchId
          });
        }
      }
    }
    if (step.type === 'merge' && 'steps' in step && step.steps) {
      const {
        nodes: mergeNodes,
        edges: mergeEdges
      } = transformYamlToNodesAndEdges([], step.steps);
      nodes.push(...mergeNodes);
      edges.push(...mergeEdges);

      // Create edge from merge step to first nested step
      if (step.steps.length > 0) {
        const firstNestedId = step.steps[0].name.toLowerCase().replace(/\s+/g, '-');
        edges.push({
          id: `${id}:${firstNestedId}`,
          source: id,
          target: firstNestedId
        });
      }
    }
  }
  return {
    nodes,
    edges
  };
}
function getLayoutedNodesAndEdges(workflowDefinition) {
  var _workflowDefinition$t, _workflowDefinition$s;
  const {
    nodes,
    edges
  } = transformYamlToNodesAndEdges((_workflowDefinition$t = workflowDefinition === null || workflowDefinition === void 0 ? void 0 : workflowDefinition.triggers) !== null && _workflowDefinition$t !== void 0 ? _workflowDefinition$t : [], (_workflowDefinition$s = workflowDefinition === null || workflowDefinition === void 0 ? void 0 : workflowDefinition.steps) !== null && _workflowDefinition$s !== void 0 ? _workflowDefinition$s : []);
  const dagreGraph = new _dagre.graphlib.Graph();
  dagreGraph.setDefaultEdgeLabel(() => ({}));

  // Set graph direction and spacing
  dagreGraph.setGraph({
    rankdir: 'TB',
    nodesep: 40,
    ranksep: 40,
    edgesep: 40
  });
  nodes.forEach(node => {
    dagreGraph.setNode(node.id, {
      label: node.label,
      width: node.style.width,
      height: node.style.height
    });
  });
  edges.forEach(edge => {
    dagreGraph.setEdge(edge.source, edge.target, {
      label: edge.id
    });
  });
  _dagre.default.layout(dagreGraph);
  const layoutedNodes = nodes.map(node => {
    const dagreNode = dagreGraph.node(node.id);
    return {
      ...node,
      targetPosition: _react.Position.Top,
      sourcePosition: _react.Position.Bottom,
      style: {
        ...node.style,
        width: dagreNode.width,
        height: dagreNode.height
      },
      // Dagre provides positions with the center of the node as origin
      position: {
        x: dagreNode.x - dagreNode.width / 2,
        y: dagreNode.y - dagreNode.height / 2
      }
    };
  });
  return {
    nodes: layoutedNodes,
    edges
  };
}