"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCompactTypeDescription = getCompactTypeDescription;
exports.getDetailedTypeDescription = getDetailedTypeDescription;
exports.getJsonSchemaDescription = getJsonSchemaDescription;
exports.getTypeScriptLikeDescription = getTypeScriptLikeDescription;
var _zod = require("@kbn/zod");
var _zodToJsonSchema = require("zod-to-json-schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Generate a detailed human-readable type description from a Zod schema
 */
function getDetailedTypeDescription(schema, options = {}) {
  const {
    detailed = true,
    maxDepth = 6,
    showOptional = true,
    includeDescriptions = true,
    singleLine = false,
    indentSpacesNumber = 2
  } = options;
  if (detailed) {
    return generateDetailedDescription(schema, 0, {
      maxDepth,
      showOptional,
      includeDescriptions,
      singleLine,
      indentSpacesNumber
    });
  } else {
    return getBasicTypeName(schema);
  }
}

/**
 * Get a simple type name (similar to existing getZodTypeName)
 */
function getBasicTypeName(schema) {
  const typedSchema = schema;
  const def = typedSchema._def;
  switch (def.typeName) {
    case 'ZodString':
      return 'string';
    case 'ZodNumber':
      return 'number';
    case 'ZodBoolean':
      return 'boolean';
    case 'ZodArray':
      const arrayElement = schema.element;
      return `${getBasicTypeName(arrayElement)}[]`;
    case 'ZodObject':
      return 'object';
    case 'ZodUnion':
      const unionTypes = schema.options;
      const unionString = unionTypes.map(t => getBasicTypeName(t)).join(' | ');
      return `(${unionString})`;
    case 'ZodOptional':
      const optionalInner = schema.unwrap();
      return `${getBasicTypeName(optionalInner)}?`;
    case 'ZodDate':
      return 'date';
    case 'ZodLiteral':
      const literalValue = schema.value;
      return typeof literalValue === 'string' ? `"${literalValue}"` : String(literalValue);
    case 'ZodEnum':
      const enumValues = schema.options;
      return enumValues.map(v => `"${v}"`).join(' | ');
    case 'ZodAny':
      return 'any';
    case 'ZodNull':
      return 'null';
    case 'ZodUndefined':
      return 'undefined';
    case 'ZodUnknown':
      return 'unknown';
    case 'ZodRecord':
      const recordSchema = schema;
      const keyType = getBasicTypeName(recordSchema.keySchema || _zod.z.any());
      const valueType = getBasicTypeName(recordSchema.valueSchema || _zod.z.any());
      return `Record<${keyType}, ${valueType}>`;
    default:
      return 'unknown';
  }
}

/**
 * Generate detailed type description with structure
 */
function generateDetailedDescription(schema, currentDepth, opts) {
  const {
    maxDepth,
    showOptional,
    includeDescriptions,
    indentSpacesNumber = 2,
    singleLine = false
  } = opts;
  if (currentDepth >= maxDepth) {
    return getBasicTypeName(schema);
  }
  const nl = singleLine ? '' : '\n';
  const typedSchema = schema;
  const def = typedSchema._def;

  // Extract description if available
  const description = includeDescriptions && 'description' in schema ? schema.description : null;
  const descriptionSuffix = description ? ` // ${description}` : '';
  switch (def.typeName) {
    case 'ZodObject':
      {
        const objectSchema = schema;
        const shape = objectSchema.shape;
        const properties = [];
        for (const [key, fieldSchema] of Object.entries(shape)) {
          const isOptional = fieldSchema instanceof _zod.z.ZodOptional;
          // If field is optional, we need to unwrap it and process the inner type
          const actualFieldSchema = isOptional ? fieldSchema.unwrap() : fieldSchema;
          const fieldType = generateDetailedDescription(actualFieldSchema, currentDepth + 1, {
            maxDepth,
            showOptional,
            // Pass showOptional for nested structures
            includeDescriptions,
            indentSpacesNumber,
            singleLine
          });
          const optionalMarker = showOptional && isOptional ? '?' : '';

          // Add proper indentation - always use 2 spaces per level
          const indent = ' '.repeat(indentSpacesNumber);
          properties.push(`${indent}${key}${optionalMarker}: ${fieldType}`);
        }
        const objectBody = properties.length > 0 ? `{${nl}${properties.join(`;${nl}`)}${nl}}` : '{}';
        return `${objectBody}${descriptionSuffix}`;
      }
    case 'ZodArray':
      {
        const arraySchema = schema;
        const elementType = generateDetailedDescription(arraySchema.element, currentDepth + 1, {
          maxDepth,
          showOptional,
          includeDescriptions,
          indentSpacesNumber,
          singleLine
        });
        return `${elementType}[]${descriptionSuffix}`;
      }
    case 'ZodUnion':
      {
        const unionSchema = schema;
        const unionTypes = unionSchema.options.map(option => generateDetailedDescription(option, currentDepth + 1, {
          maxDepth,
          showOptional,
          includeDescriptions,
          indentSpacesNumber,
          singleLine
        }));
        return `(${unionTypes.join(' | ')})${descriptionSuffix}`;
      }
    case 'ZodOptional':
      {
        const optionalSchema = schema;
        const innerType = generateDetailedDescription(optionalSchema.unwrap(), currentDepth, {
          maxDepth,
          showOptional: false,
          // Don't show optional for inner type since we're handling it here
          includeDescriptions,
          indentSpacesNumber,
          singleLine
        });
        // Only add ? if showOptional is true
        return showOptional ? `${innerType}?` : innerType;
      }
    case 'ZodDiscriminatedUnion':
      {
        const discriminatedSchema = schema;
        const discriminator = discriminatedSchema.discriminator;
        const options = discriminatedSchema.options;
        const unionTypes = options.map(option => generateDetailedDescription(option, currentDepth + 1, {
          maxDepth,
          showOptional,
          includeDescriptions,
          indentSpacesNumber,
          singleLine
        }));
        return `discriminatedUnion<${discriminator}>(${unionTypes.join(' | ')})${descriptionSuffix}`;
      }
    case 'ZodRecord':
      {
        const recordSchema = schema;
        const valueType = generateDetailedDescription(recordSchema.valueSchema || _zod.z.any(), currentDepth + 1, {
          maxDepth,
          showOptional,
          includeDescriptions,
          indentSpacesNumber,
          singleLine
        });
        return `Record<string, ${valueType}>${descriptionSuffix}`;
      }
    default:
      return `${getBasicTypeName(schema)}${descriptionSuffix}`;
  }
}

/**
 * Convert Zod schema to JSON Schema for maximum detail
 */
function getJsonSchemaDescription(schema) {
  return (0, _zodToJsonSchema.zodToJsonSchema)(schema, {
    target: 'jsonSchema7',
    errorMessages: true,
    markdownDescription: true,
    definitions: {}
  });
}

/**
 * Get a TypeScript-like type string from Zod schema
 */
function getTypeScriptLikeDescription(schema) {
  return getDetailedTypeDescription(schema, {
    detailed: true,
    maxDepth: 5,
    showOptional: true,
    includeDescriptions: false
  });
}

/**
 * Get a compact type description suitable for tooltips
 */
function getCompactTypeDescription(schema) {
  return getDetailedTypeDescription(schema, {
    detailed: false,
    maxDepth: 1,
    showOptional: false,
    includeDescriptions: false
  });
}