"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkflowEventLoggerService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _workflow_event_logger = require("./workflow_event_logger");
var _create_index = require("./create_index");
var _index_mappings = require("./index_mappings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class WorkflowEventLoggerService {
  constructor(options) {
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "indexName", void 0);
    (0, _defineProperty2.default)(this, "enableConsoleLogging", void 0);
    (0, _defineProperty2.default)(this, "initialized", false);
    this.esClient = options.esClient;
    this.logger = options.logger;
    this.indexName = options.indexName;
    this.enableConsoleLogging = options.enableConsoleLogging || false;
  }
  async initialize() {
    if (this.initialized) {
      return;
    }
    try {
      // Create the logs index with proper mappings
      await (0, _create_index.createIndexWithMappings)({
        esClient: this.esClient,
        indexName: this.indexName,
        mappings: _index_mappings.WORKFLOW_EXECUTION_LOGS_INDEX_MAPPINGS,
        logger: this.logger
      });
      this.initialized = true;
      this.logger.debug(`WorkflowEventLoggerService initialized with index: ${this.indexName}`);
    } catch (error) {
      this.logger.error('Failed to initialize WorkflowEventLoggerService', error);
      throw error;
    }
  }
  createLogger(context) {
    if (!this.initialized) {
      throw new Error('WorkflowEventLoggerService not initialized. Call initialize() first.');
    }
    return new _workflow_event_logger.WorkflowEventLogger(this.esClient, this.logger, this.indexName, context, {
      enableConsoleLogging: this.enableConsoleLogging
    });
  }
  createWorkflowLogger(workflowId, workflowName) {
    return this.createLogger({
      workflowId,
      workflowName
    });
  }
  createExecutionLogger(workflowId, executionId, workflowName) {
    return this.createLogger({
      workflowId,
      workflowName,
      executionId
    });
  }
  createStepLogger(workflowId, executionId, stepId, stepName, stepType, workflowName) {
    return this.createLogger({
      workflowId,
      workflowName,
      executionId,
      stepId,
      stepName,
      stepType
    });
  }
  async searchLogs(query) {
    if (!this.initialized) {
      throw new Error('WorkflowEventLoggerService not initialized. Call initialize() first.');
    }
    try {
      var _response$hits$total;
      const response = await this.esClient.search({
        index: this.indexName,
        query,
        sort: [{
          '@timestamp': {
            order: 'desc'
          }
        }],
        size: 1000 // Default limit
      });
      return {
        total: typeof response.hits.total === 'number' ? response.hits.total : ((_response$hits$total = response.hits.total) === null || _response$hits$total === void 0 ? void 0 : _response$hits$total.value) || 0,
        logs: response.hits.hits.map(hit => hit._source)
      };
    } catch (error) {
      this.logger.error('Failed to search workflow logs', error);
      throw error;
    }
  }
  async getExecutionLogs(executionId) {
    const query = {
      bool: {
        must: [{
          term: {
            'workflow.execution_id': executionId
          }
        }]
      }
    };
    return this.searchLogs(query);
  }
  async getStepLogs(executionId, stepId) {
    const query = {
      bool: {
        must: [{
          term: {
            'workflow.execution_id': executionId
          }
        }, {
          term: {
            'workflow.step_id': stepId
          }
        }]
      }
    };
    return this.searchLogs(query);
  }
  async getLogsByLevel(level, executionId) {
    const mustClauses = [{
      term: {
        level
      }
    }];
    if (executionId) {
      mustClauses.push({
        term: {
          'workflow.execution_id': executionId
        }
      });
    }
    const query = {
      bool: {
        must: mustClauses
      }
    };
    return this.searchLogs(query);
  }
  async getRecentLogs(limit = 100) {
    const query = {
      match_all: {}
    };
    try {
      var _response$hits$total2;
      const response = await this.esClient.search({
        index: this.indexName,
        query,
        sort: [{
          '@timestamp': {
            order: 'desc'
          }
        }],
        size: limit
      });
      return {
        total: typeof response.hits.total === 'number' ? response.hits.total : ((_response$hits$total2 = response.hits.total) === null || _response$hits$total2 === void 0 ? void 0 : _response$hits$total2.value) || 0,
        logs: response.hits.hits.map(hit => hit._source)
      };
    } catch (error) {
      this.logger.error('Failed to get recent workflow logs', error);
      throw error;
    }
  }
}
exports.WorkflowEventLoggerService = WorkflowEventLoggerService;