"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createOrUpdateIndex = exports.createIndexWithMappings = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const createIndexWithMappings = async ({
  esClient,
  indexName,
  mappings,
  logger
}) => {
  try {
    // Check if index already exists
    const indexExists = await esClient.indices.exists({
      index: indexName
    });
    if (indexExists) {
      logger.debug(`Index ${indexName} already exists`);
      return;
    }
    logger.debug(`Creating index ${indexName} with mappings`);

    // Create the index with proper mappings
    await esClient.indices.create({
      index: indexName,
      mappings
    });
    logger.info(`Successfully created index ${indexName}`);
  } catch (error) {
    var _error$meta, _error$meta$body, _error$meta$body$erro;
    // If the index already exists, we can ignore the error
    if ((error === null || error === void 0 ? void 0 : (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$body = _error$meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : _error$meta$body$erro.type) === 'resource_already_exists_exception') {
      logger.debug(`Index ${indexName} already exists (created by another process)`);
      return;
    }
    logger.error(`Failed to create index ${indexName}: ${error}`);
    throw error;
  }
};
exports.createIndexWithMappings = createIndexWithMappings;
const createOrUpdateIndex = async ({
  esClient,
  indexName,
  mappings,
  logger
}) => {
  try {
    const indexExists = await esClient.indices.exists({
      index: indexName
    });
    if (!indexExists) {
      // Create new index
      await createIndexWithMappings({
        esClient,
        indexName,
        mappings,
        logger
      });
    } else {
      // Index exists, check if we need to update mappings
      try {
        await esClient.indices.putMapping({
          index: indexName,
          ...mappings
        });
        logger.debug(`Updated mappings for existing index ${indexName}`);
      } catch (mappingError) {
        logger.warn(`Failed to update mappings for index ${indexName}: ${mappingError.message}`);
        // Continue - the index exists and can be used
      }
    }
  } catch (error) {
    logger.error(`Failed to create or update index ${indexName}: ${error}`);
    throw error;
  }
};
exports.createOrUpdateIndex = createOrUpdateIndex;