"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkflowExecutionRuntimeManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _workflows = require("@kbn/workflows");
var _apmUtils = require("@kbn/apm-utils");
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _utils = require("../utils");
var _workflow_scope_stack = require("./workflow_scope_stack");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Manages the runtime execution state of a workflow, including step execution, results, and transitions.
 *
 * The `WorkflowExecutionRuntimeManager` class is responsible for orchestrating the execution of workflow steps,
 * tracking their statuses, results, and runtime states, and updating the overall workflow execution status.
 * It maintains the topological order of steps, supports step navigation, and interacts with repositories to persist
 * execution data.
 *
 * Key responsibilities:
 * - Tracks execution status, results, and runtime state for each workflow step.
 * - Navigates between steps according to topological order, skipping steps as needed.
 * - Initiates and finalizes step and workflow executions, updating persistent storage.
 * - Handles workflow completion and error propagation.
 * - Creates APM traces compatible with APM TraceWaterfall embeddable
 *
 * @remarks
 * This class assumes that workflow steps are represented as nodes in a directed acyclic graph (DAG),
 * and uses topological sorting to determine execution order.
 */
class WorkflowExecutionRuntimeManager {
  get topologicalOrder() {
    return this.workflowGraph.topologicalOrder;
  }
  constructor(workflowExecutionRuntimeManagerInit) {
    (0, _defineProperty2.default)(this, "workflowLogger", null);
    (0, _defineProperty2.default)(this, "workflowExecutionState", void 0);
    (0, _defineProperty2.default)(this, "entryTransactionId", void 0);
    (0, _defineProperty2.default)(this, "workflowTransaction", void 0);
    // APM transaction instance
    (0, _defineProperty2.default)(this, "workflowGraph", void 0);
    this.workflowGraph = workflowExecutionRuntimeManagerInit.workflowExecutionGraph;

    // Use workflow execution ID as traceId for APM compatibility
    this.workflowLogger = workflowExecutionRuntimeManagerInit.workflowLogger;
    this.workflowExecutionState = workflowExecutionRuntimeManagerInit.workflowExecutionState;
  }
  get workflowExecution() {
    return this.workflowExecutionState.getWorkflowExecution();
  }

  /**
   * Get the APM trace ID for this workflow execution
   */
  getTraceId() {
    return this.getWorkflowExecution().id;
  }

  /**
   * Get the entry transaction ID (main workflow transaction)
   */
  getEntryTransactionId() {
    return this.entryTransactionId;
  }
  getWorkflowExecutionStatus() {
    return this.workflowExecutionState.getWorkflowExecution().status;
  }
  getWorkflowExecution() {
    return this.workflowExecutionState.getWorkflowExecution();
  }
  getCurrentNode() {
    if (!this.workflowExecution.currentNodeId) {
      return null;
    }
    return this.workflowGraph.getNode(this.workflowExecution.currentNodeId);
  }
  getCurrentStepExecutionId() {
    return (0, _utils.buildStepExecutionId)(this.workflowExecution.id, this.getCurrentNode().stepId, this.workflowExecution.scopeStack);
  }
  navigateToNode(nodeId) {
    if (!this.workflowGraph.getNode(nodeId)) {
      throw new Error(`Node with ID ${nodeId} is not part of the workflow graph`);
    }
    this.workflowExecutionState.updateWorkflowExecution({
      currentNodeId: nodeId
    });
  }
  navigateToNextNode() {
    const currentNodeId = this.workflowExecution.currentNodeId;
    const currentNodeIndex = this.topologicalOrder.findIndex(nodeId => nodeId === currentNodeId);
    if (currentNodeIndex < this.topologicalOrder.length - 1) {
      this.workflowExecutionState.updateWorkflowExecution({
        currentNodeId: this.topologicalOrder[currentNodeIndex + 1]
      });
      return;
    }
    this.workflowExecutionState.updateWorkflowExecution({
      currentNodeId: undefined
    });
  }
  getCurrentNodeScope() {
    return [...this.workflowExecution.scopeStack];
  }
  enterScope(subScopeId) {
    const currentNode = this.getCurrentNode();
    this.workflowExecutionState.updateWorkflowExecution({
      scopeStack: _workflow_scope_stack.WorkflowScopeStack.fromStackFrames(this.workflowExecution.scopeStack).enterScope({
        nodeId: currentNode.id,
        nodeType: currentNode.type,
        stepId: currentNode.stepId,
        scopeId: subScopeId
      }).stackFrames
    });
  }
  exitScope() {
    this.workflowExecutionState.updateWorkflowExecution({
      scopeStack: _workflow_scope_stack.WorkflowScopeStack.fromStackFrames(this.workflowExecution.scopeStack).exitScope().stackFrames
    });
  }
  setWorkflowError(error) {
    this.workflowExecutionState.updateWorkflowExecution({
      error: error ? String(error) : undefined
    });
  }
  getCurrentStepResult() {
    const stepExecution = this.workflowExecutionState.getStepExecution(this.getCurrentStepExecutionId());
    if (!stepExecution) {
      return undefined;
    }
    return {
      input: stepExecution.input || {},
      output: stepExecution.output || {},
      error: stepExecution.error
    };
  }
  async setCurrentStepResult(result) {
    const currentNode = this.getCurrentNode();
    if (result.error) {
      this.setWorkflowError(result.error);
    }
    this.workflowExecutionState.upsertStep({
      id: this.getCurrentStepExecutionId(),
      stepId: currentNode.stepId,
      input: result.input,
      output: result.output,
      error: result.error
    });
  }
  getCurrentStepState() {
    var _this$workflowExecuti;
    return (_this$workflowExecuti = this.workflowExecutionState.getStepExecution(this.getCurrentStepExecutionId())) === null || _this$workflowExecuti === void 0 ? void 0 : _this$workflowExecuti.state;
  }
  async setCurrentStepState(state) {
    const stepId = this.getCurrentNode().stepId;
    this.workflowExecutionState.upsertStep({
      id: this.getCurrentStepExecutionId(),
      stepId,
      state
    });
  }
  async startStep() {
    const currentNode = this.getCurrentNode();
    const stepId = currentNode.stepId;
    return (0, _apmUtils.withSpan)({
      name: `workflow.step.${stepId}`,
      type: 'workflow',
      subtype: 'step',
      labels: {
        workflow_step_id: stepId,
        workflow_execution_id: this.workflowExecution.id,
        workflow_id: this.workflowExecution.workflowId,
        trace_id: this.getTraceId(),
        // Ensure consistent traceId
        service_name: 'workflow-engine'
      }
    }, async () => {
      const stepStartedAt = new Date();
      const stepExecution = {
        id: this.getCurrentStepExecutionId(),
        stepId: currentNode.stepId,
        stepType: currentNode.stepType,
        scopeStack: this.workflowExecution.scopeStack,
        topologicalIndex: this.topologicalOrder.indexOf(currentNode.id),
        status: _workflows.ExecutionStatus.RUNNING,
        startedAt: stepStartedAt.toISOString()
      };
      this.workflowExecutionState.upsertStep(stepExecution);
      this.logStepStart(stepId, stepExecution.id);
      await this.workflowExecutionState.flushStepChanges();
    });
  }
  async finishStep() {
    const stepId = this.getCurrentNode().stepId;
    const stepExecutionId = this.getCurrentStepExecutionId();
    const startedStepExecution = this.workflowExecutionState.getStepExecution(stepExecutionId);
    if (startedStepExecution !== null && startedStepExecution !== void 0 && startedStepExecution.error) {
      await this.failStep(startedStepExecution.error);
      return;
    }
    return (0, _apmUtils.withSpan)({
      name: `workflow.step.${stepId}.complete`,
      type: 'workflow',
      subtype: 'step_completion',
      labels: {
        workflow_step_id: stepId,
        workflow_execution_id: this.workflowExecution.id,
        workflow_id: this.workflowExecution.workflowId,
        trace_id: this.getTraceId(),
        service_name: 'workflow-engine'
      }
    }, async () => {
      const stepExecutionUpdate = {
        id: stepExecutionId,
        status: _workflows.ExecutionStatus.COMPLETED,
        completedAt: new Date().toISOString()
      };
      if (startedStepExecution !== null && startedStepExecution !== void 0 && startedStepExecution.startedAt) {
        stepExecutionUpdate.executionTimeMs = new Date(stepExecutionUpdate.completedAt).getTime() - new Date(startedStepExecution.startedAt).getTime();
      }
      this.workflowExecutionState.upsertStep(stepExecutionUpdate);
      this.logStepComplete(stepExecutionUpdate);
    });
  }
  async failStep(error) {
    const stepId = this.getCurrentNode().stepId;
    return (0, _apmUtils.withSpan)({
      name: `workflow.step.${stepId}.fail`,
      type: 'workflow',
      subtype: 'step_failure',
      labels: {
        workflow_step_id: stepId,
        workflow_execution_id: this.workflowExecution.id,
        workflow_id: this.workflowExecution.workflowId,
        trace_id: this.getTraceId(),
        service_name: 'workflow-engine'
      }
    }, async () => {
      // if there is a last step execution, fail it
      // if not, create a new step execution with fail
      const startedStepExecution = this.workflowExecutionState.getStepExecution(this.getCurrentStepExecutionId());
      const stepExecutionUpdate = {
        id: this.getCurrentStepExecutionId(),
        status: _workflows.ExecutionStatus.FAILED,
        completedAt: new Date().toISOString(),
        output: null,
        error: String(error)
      };
      if (startedStepExecution && startedStepExecution.startedAt) {
        stepExecutionUpdate.executionTimeMs = new Date(stepExecutionUpdate.completedAt).getTime() - new Date(startedStepExecution.startedAt).getTime();
      }
      this.workflowExecutionState.updateWorkflowExecution({
        error: String(error)
      });
      this.workflowExecutionState.upsertStep(stepExecutionUpdate);
      this.logStepFail(stepExecutionUpdate.id, error);
    });
  }
  async setWaitStep() {
    const workflowExecution = this.workflowExecutionState.getWorkflowExecution();
    const stepId = this.getCurrentNode().stepId;
    return (0, _apmUtils.withSpan)({
      name: `workflow.step.${stepId}.delayed`,
      type: 'workflow',
      subtype: 'step_delayed',
      labels: {
        workflow_step_id: stepId,
        workflow_execution_id: workflowExecution.id,
        workflow_id: workflowExecution.workflowId,
        trace_id: this.getTraceId(),
        service_name: 'workflow-engine'
      }
    }, async () => {
      this.workflowExecutionState.upsertStep({
        id: this.getCurrentStepExecutionId(),
        status: _workflows.ExecutionStatus.WAITING
      });
      this.workflowExecutionState.updateWorkflowExecution({
        status: _workflows.ExecutionStatus.WAITING
      });
    });
  }
  markWorkflowTimeouted() {
    const finishedAt = new Date().toISOString();
    this.workflowExecutionState.updateWorkflowExecution({
      status: _workflows.ExecutionStatus.TIMED_OUT,
      finishedAt,
      duration: new Date(finishedAt).getTime() - new Date(this.workflowExecution.startedAt).getTime()
    });
  }
  async start() {
    var _this$workflowLogger;
    (_this$workflowLogger = this.workflowLogger) === null || _this$workflowLogger === void 0 ? void 0 : _this$workflowLogger.logInfo('Starting workflow execution with APM tracing', {
      workflow: {
        execution_id: this.workflowExecution.id
      }
    });
    const existingTransaction = _elasticApmNode.default.currentTransaction;
    if (existingTransaction) {
      var _labels, _this$workflowLogger2, _labels2, _existingTransaction$;
      // Check if this is triggered by alerting (has alerting labels) or task manager directly
      const isTriggeredByAlerting = !!((_labels = existingTransaction._labels) !== null && _labels !== void 0 && _labels.alerting_rule_id);
      (_this$workflowLogger2 = this.workflowLogger) === null || _this$workflowLogger2 === void 0 ? void 0 : _this$workflowLogger2.logDebug('Found existing transaction context', {
        transaction: {
          name: existingTransaction.name,
          type: existingTransaction.type,
          is_triggered_by_alerting: isTriggeredByAlerting,
          alerting_rule_id: (_labels2 = existingTransaction._labels) === null || _labels2 === void 0 ? void 0 : _labels2.alerting_rule_id,
          transaction_id: (_existingTransaction$ = existingTransaction.ids) === null || _existingTransaction$ === void 0 ? void 0 : _existingTransaction$['transaction.id']
        }
      });
      if (isTriggeredByAlerting) {
        var _this$workflowLogger3, _labels3, _workflowTransaction$, _workflowTransaction$2, _trace;
        // For alerting-triggered workflows, create a dedicated workflow transaction
        // This provides a focused view for the embeddable instead of the entire alerting trace
        (_this$workflowLogger3 = this.workflowLogger) === null || _this$workflowLogger3 === void 0 ? void 0 : _this$workflowLogger3.logInfo('Creating dedicated workflow transaction within alerting trace');
        const workflowTransaction = _elasticApmNode.default.startTransaction(`workflow.execution.${this.workflowExecution.workflowId}`, 'workflow_execution');
        this.workflowTransaction = workflowTransaction;

        // Add workflow-specific labels
        workflowTransaction.addLabels({
          workflow_execution_id: this.workflowExecution.id,
          workflow_id: this.workflowExecution.workflowId,
          service_name: 'kibana',
          transaction_hierarchy: 'alerting->workflow->steps',
          triggered_by: 'alerting',
          parent_alerting_rule_id: (_labels3 = existingTransaction._labels) === null || _labels3 === void 0 ? void 0 : _labels3.alerting_rule_id
        });

        // Make the workflow transaction the current transaction for subsequent spans
        _elasticApmNode.default.setCurrentTransaction(workflowTransaction);

        // Store the workflow transaction ID (not the alerting transaction ID)
        const workflowTransactionId = (_workflowTransaction$ = workflowTransaction.ids) === null || _workflowTransaction$ === void 0 ? void 0 : _workflowTransaction$['transaction.id'];
        if (workflowTransactionId) {
          var _this$workflowLogger4, _this$workflowLogger5;
          (_this$workflowLogger4 = this.workflowLogger) === null || _this$workflowLogger4 === void 0 ? void 0 : _this$workflowLogger4.logDebug('Storing workflow transaction ID', {
            transaction: {
              workflow_transaction_id: workflowTransactionId
            }
          });
          this.workflowExecutionState.updateWorkflowExecution({
            entryTransactionId: workflowTransactionId
          });
          (_this$workflowLogger5 = this.workflowLogger) === null || _this$workflowLogger5 === void 0 ? void 0 : _this$workflowLogger5.logDebug('Workflow transaction ID stored in workflow execution');
        }

        // Capture trace ID from the workflow transaction
        let realTraceId;
        if (workflowTransaction !== null && workflowTransaction !== void 0 && workflowTransaction.traceId) {
          realTraceId = workflowTransaction.traceId;
        } else if ((_workflowTransaction$2 = workflowTransaction.ids) !== null && _workflowTransaction$2 !== void 0 && _workflowTransaction$2['trace.id']) {
          realTraceId = workflowTransaction.ids['trace.id'];
        } else if (workflowTransaction !== null && workflowTransaction !== void 0 && (_trace = workflowTransaction.trace) !== null && _trace !== void 0 && _trace.id) {
          realTraceId = workflowTransaction.trace.id;
        }
        if (realTraceId) {
          var _this$workflowLogger6;
          (_this$workflowLogger6 = this.workflowLogger) === null || _this$workflowLogger6 === void 0 ? void 0 : _this$workflowLogger6.logDebug('Captured APM trace ID from workflow transaction', {
            trace: {
              trace_id: realTraceId
            }
          });
          this.workflowExecutionState.updateWorkflowExecution({
            traceId: realTraceId
          });
        }
      } else {
        var _this$workflowLogger7, _existingTransaction$2, _existingTransaction$3, _trace2;
        // For task manager triggered workflows, reuse the existing transaction
        (_this$workflowLogger7 = this.workflowLogger) === null || _this$workflowLogger7 === void 0 ? void 0 : _this$workflowLogger7.logInfo('Reusing task manager transaction for workflow execution');
        this.workflowTransaction = existingTransaction;

        // Add workflow-specific labels to the existing transaction
        existingTransaction.addLabels({
          workflow_execution_id: this.workflowExecution.id,
          workflow_id: this.workflowExecution.workflowId,
          service_name: 'kibana',
          transaction_hierarchy: 'task->steps',
          triggered_by: 'task_manager'
        });

        // Store the task transaction ID in the workflow execution
        const taskTransactionId = (_existingTransaction$2 = existingTransaction.ids) === null || _existingTransaction$2 === void 0 ? void 0 : _existingTransaction$2['transaction.id'];
        if (taskTransactionId) {
          var _this$workflowLogger8, _this$workflowLogger9;
          (_this$workflowLogger8 = this.workflowLogger) === null || _this$workflowLogger8 === void 0 ? void 0 : _this$workflowLogger8.logDebug('Storing task transaction ID', {
            transaction: {
              task_transaction_id: taskTransactionId
            }
          });
          this.workflowExecutionState.updateWorkflowExecution({
            entryTransactionId: taskTransactionId
          });
          (_this$workflowLogger9 = this.workflowLogger) === null || _this$workflowLogger9 === void 0 ? void 0 : _this$workflowLogger9.logDebug('Task transaction ID stored in workflow execution');
        }

        // Capture trace ID from the task transaction
        let realTraceId;
        if (existingTransaction !== null && existingTransaction !== void 0 && existingTransaction.traceId) {
          realTraceId = existingTransaction.traceId;
        } else if ((_existingTransaction$3 = existingTransaction.ids) !== null && _existingTransaction$3 !== void 0 && _existingTransaction$3['trace.id']) {
          realTraceId = existingTransaction.ids['trace.id'];
        } else if (existingTransaction !== null && existingTransaction !== void 0 && (_trace2 = existingTransaction.trace) !== null && _trace2 !== void 0 && _trace2.id) {
          realTraceId = existingTransaction.trace.id;
        }
        if (realTraceId) {
          var _this$workflowLogger10;
          (_this$workflowLogger10 = this.workflowLogger) === null || _this$workflowLogger10 === void 0 ? void 0 : _this$workflowLogger10.logDebug('Captured APM trace ID from task transaction', {
            trace: {
              trace_id: realTraceId
            }
          });
          this.workflowExecutionState.updateWorkflowExecution({
            traceId: realTraceId
          });
        }
      }

      // Set the transaction outcome to success by default
      // It will be overridden if the workflow fails
      existingTransaction.outcome = 'success';
    } else {
      var _this$workflowLogger11;
      // Fallback if no task transaction exists - proceed without tracing
      (_this$workflowLogger11 = this.workflowLogger) === null || _this$workflowLogger11 === void 0 ? void 0 : _this$workflowLogger11.logWarn('No active Task Manager transaction found, proceeding without APM tracing');
    }
    const updatedWorkflowExecution = {
      currentNodeId: this.topologicalOrder[0],
      scopeStack: [],
      status: _workflows.ExecutionStatus.RUNNING,
      startedAt: new Date().toISOString()
    };
    this.workflowExecutionState.updateWorkflowExecution(updatedWorkflowExecution);
    this.logWorkflowStart();
    await this.workflowExecutionState.flush();
  }
  async resume() {
    await this.workflowExecutionState.load();
    const updatedWorkflowExecution = {
      status: _workflows.ExecutionStatus.RUNNING
    };
    this.workflowExecutionState.updateWorkflowExecution(updatedWorkflowExecution);
  }
  async saveState() {
    const workflowExecution = this.workflowExecutionState.getWorkflowExecution();
    const workflowExecutionUpdate = {};
    if (!workflowExecution.currentNodeId) {
      workflowExecutionUpdate.status = _workflows.ExecutionStatus.COMPLETED;
    }
    if (workflowExecution.error) {
      workflowExecutionUpdate.status = _workflows.ExecutionStatus.FAILED;
    }
    if ([_workflows.ExecutionStatus.COMPLETED, _workflows.ExecutionStatus.FAILED].includes(workflowExecutionUpdate.status)) {
      const startedAt = new Date(workflowExecution.startedAt);
      const completeDate = new Date();
      workflowExecutionUpdate.finishedAt = completeDate.toISOString();
      workflowExecutionUpdate.duration = completeDate.getTime() - startedAt.getTime();
      this.logWorkflowComplete(workflowExecutionUpdate.status === _workflows.ExecutionStatus.COMPLETED);

      // Update the workflow transaction outcome when workflow completes
      if (this.workflowTransaction) {
        const isSuccess = workflowExecutionUpdate.status === _workflows.ExecutionStatus.COMPLETED;
        this.workflowTransaction.outcome = isSuccess ? 'success' : 'failure';

        // For alerting-triggered workflows, we created a dedicated transaction and need to end it
        const isTriggeredByAlerting = this.workflowTransaction.type === 'workflow_execution';
        if (isTriggeredByAlerting) {
          var _this$workflowLogger12;
          this.workflowTransaction.end();
          (_this$workflowLogger12 = this.workflowLogger) === null || _this$workflowLogger12 === void 0 ? void 0 : _this$workflowLogger12.logDebug('Workflow transaction ended (alerting-triggered)', {
            transaction: {
              outcome: this.workflowTransaction.outcome
            }
          });
        } else {
          var _this$workflowLogger13;
          // For task manager triggered workflows, Task Manager will handle ending
          (_this$workflowLogger13 = this.workflowLogger) === null || _this$workflowLogger13 === void 0 ? void 0 : _this$workflowLogger13.logDebug('Task transaction outcome updated (task manager will end)', {
            transaction: {
              outcome: this.workflowTransaction.outcome
            }
          });
        }
      }
    }
    this.workflowExecutionState.updateWorkflowExecution(workflowExecutionUpdate);
    await this.workflowExecutionState.flush();
  }
  logWorkflowStart() {
    var _this$workflowLogger14;
    (_this$workflowLogger14 = this.workflowLogger) === null || _this$workflowLogger14 === void 0 ? void 0 : _this$workflowLogger14.logInfo('Workflow execution started', {
      event: {
        action: 'workflow-start',
        category: ['workflow']
      },
      tags: ['workflow', 'execution', 'start']
    });
  }
  logWorkflowComplete(success) {
    var _this$workflowLogger15;
    (_this$workflowLogger15 = this.workflowLogger) === null || _this$workflowLogger15 === void 0 ? void 0 : _this$workflowLogger15.logInfo(`Workflow execution ${success ? 'completed successfully' : 'failed'}`, {
      event: {
        action: 'workflow-complete',
        category: ['workflow'],
        outcome: success ? 'success' : 'failure'
      },
      tags: ['workflow', 'execution', 'complete']
    });
  }
  logStepStart(stepId, stepExecutionId) {
    var _this$workflowLogger16;
    const currentNode = this.getCurrentNode();
    (_this$workflowLogger16 = this.workflowLogger) === null || _this$workflowLogger16 === void 0 ? void 0 : _this$workflowLogger16.logInfo(`Step '${stepId}' started`, {
      workflow: {
        step_id: stepId,
        step_execution_id: stepExecutionId
      },
      event: {
        action: 'step-start',
        category: ['workflow', 'step']
      },
      tags: ['workflow', 'step', 'start'],
      labels: {
        step_type: currentNode.stepType,
        connector_type: currentNode.stepType,
        step_name: currentNode.stepId,
        step_id: stepId
      }
    });
  }
  logStepComplete(step) {
    var _this$workflowLogger17, _step$error, _step$error2, _step$error3;
    const isSuccess = (step === null || step === void 0 ? void 0 : step.status) === _workflows.ExecutionStatus.COMPLETED;
    const currentNode = this.getCurrentNode();
    const stepId = currentNode.stepId;
    (_this$workflowLogger17 = this.workflowLogger) === null || _this$workflowLogger17 === void 0 ? void 0 : _this$workflowLogger17.logInfo(`Step '${stepId}' ${isSuccess ? 'completed' : 'failed'}`, {
      workflow: {
        step_id: stepId,
        step_execution_id: step.id
      },
      event: {
        action: 'step-complete',
        category: ['workflow', 'step'],
        outcome: isSuccess ? 'success' : 'failure'
      },
      tags: ['workflow', 'step', 'complete'],
      labels: {
        step_type: currentNode.stepType,
        connector_type: currentNode.stepType,
        step_name: currentNode.stepId,
        step_id: currentNode.stepId,
        execution_time_ms: step.executionTimeMs
      },
      ...(step.error && {
        error: {
          message: typeof step.error === 'string' ? step.error : ((_step$error = step.error) === null || _step$error === void 0 ? void 0 : _step$error.message) || 'Unknown error',
          type: typeof step.error === 'string' ? 'WorkflowStepError' : ((_step$error2 = step.error) === null || _step$error2 === void 0 ? void 0 : _step$error2.name) || 'Error',
          stack_trace: typeof step.error === 'string' ? undefined : (_step$error3 = step.error) === null || _step$error3 === void 0 ? void 0 : _step$error3.stack
        }
      })
    });
  }
  logStepFail(stepExecutionId, error) {
    var _this$workflowLogger18;
    const currentNode = this.getCurrentNode();
    const stepName = currentNode.stepId;
    const stepType = currentNode.stepType || 'unknown';
    const _error = typeof error === 'string' ? Error(error) : error;

    // Include error message in the log message
    const errorMsg = typeof error === 'string' ? error : (error === null || error === void 0 ? void 0 : error.message) || 'Unknown error';
    const message = `Step '${stepName}' failed: ${errorMsg}`;
    (_this$workflowLogger18 = this.workflowLogger) === null || _this$workflowLogger18 === void 0 ? void 0 : _this$workflowLogger18.logError(message, _error, {
      workflow: {
        step_id: currentNode.stepId,
        step_execution_id: stepExecutionId
      },
      event: {
        action: 'step-fail',
        category: ['workflow', 'step']
      },
      tags: ['workflow', 'step', 'fail'],
      labels: {
        step_type: stepType,
        connector_type: stepType,
        step_name: stepName,
        step_id: currentNode.stepId
      }
    });
  }
}
exports.WorkflowExecutionRuntimeManager = WorkflowExecutionRuntimeManager;