"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkflowContextManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _utils = require("../utils");
var _workflow_scope_stack = require("./workflow_scope_stack");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class WorkflowContextManager {
  get scopeStack() {
    return _workflow_scope_stack.WorkflowScopeStack.fromStackFrames(this.stackFrames);
  }
  constructor(init) {
    (0, _defineProperty2.default)(this, "workflowExecutionGraph", void 0);
    (0, _defineProperty2.default)(this, "workflowExecutionState", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "fakeRequest", void 0);
    (0, _defineProperty2.default)(this, "coreStart", void 0);
    (0, _defineProperty2.default)(this, "stackFrames", void 0);
    (0, _defineProperty2.default)(this, "node", void 0);
    (0, _defineProperty2.default)(this, "stepExecutionId", void 0);
    (0, _defineProperty2.default)(this, "abortController", new AbortController());
    this.workflowExecutionGraph = init.workflowExecutionGraph;
    this.workflowExecutionState = init.workflowExecutionState;
    this.esClient = init.esClient;
    this.fakeRequest = init.fakeRequest;
    this.coreStart = init.coreStart;
    this.node = init.node;
    this.stackFrames = init.stackFrames;
    this.stepExecutionId = (0, _utils.buildStepExecutionId)(this.workflowExecutionState.getWorkflowExecution().id, this.node.stepId, this.stackFrames);
  }

  // Any change here should be reflected in the 'getContextSchemaForPath' function for frontend validation to work
  // src/platform/plugins/shared/workflows_management/public/features/workflow_context/lib/get_context_for_path.ts
  getContext() {
    const stepContext = {
      ...this.buildWorkflowContext(),
      steps: {}
    };
    const currentNode = this.node;
    const currentNodeId = currentNode.id;
    const allPredecessors = this.workflowExecutionGraph.getAllPredecessors(currentNodeId);
    allPredecessors.forEach(node => {
      const stepId = node.stepId;
      const stepData = this.getStepData(stepId);
      if (stepData) {
        stepContext.steps[stepId] = {};
        if (stepData.runStepResult) {
          stepContext.steps[stepId] = {
            ...stepContext.steps[stepId],
            ...stepData.runStepResult
          };
        }
        if (stepData.stepState) {
          stepContext.steps[stepId] = {
            ...stepContext.steps[stepId],
            ...stepData.stepState
          };
        }
      }
    });
    this.enrichStepContextWithMockedData(stepContext);
    this.enrichStepContextAccordingToStepScope(stepContext);
    return stepContext;
  }
  readContextPath(propertyPath) {
    const propertyPathSegments = propertyPath.split('.');
    let result = this.getContext();
    for (const segment of propertyPathSegments) {
      if (!(segment in result)) {
        return {
          pathExists: false,
          value: undefined
        }; // Path not found in context
      }
      result = result[segment];
    }
    return {
      pathExists: true,
      value: result
    };
  }

  /**
   * Get the Elasticsearch client for internal actions
   * This client is already user-scoped if fakeRequest was available during initialization
   */
  getEsClientAsUser() {
    return this.esClient;
  }

  /**
   * Get the fake request from task manager for Kibana API authentication
   */
  getFakeRequest() {
    return this.fakeRequest;
  }

  /**
   * Get CoreStart for accessing Kibana's internal services
   */
  getCoreStart() {
    return this.coreStart;
  }
  buildWorkflowContext() {
    var _workflowExecution$co, _workflowExecution$co2;
    const workflowExecution = this.workflowExecutionState.getWorkflowExecution();
    return {
      execution: {
        id: workflowExecution.id,
        isTestRun: !!workflowExecution.isTestRun,
        startedAt: new Date(workflowExecution.startedAt)
      },
      workflow: {
        id: workflowExecution.workflowId,
        name: workflowExecution.workflowDefinition.name,
        enabled: workflowExecution.workflowDefinition.enabled,
        spaceId: workflowExecution.spaceId
      },
      consts: workflowExecution.workflowDefinition.consts || {},
      event: (_workflowExecution$co = workflowExecution.context) === null || _workflowExecution$co === void 0 ? void 0 : _workflowExecution$co.event,
      inputs: (_workflowExecution$co2 = workflowExecution.context) === null || _workflowExecution$co2 === void 0 ? void 0 : _workflowExecution$co2.inputs
    };
  }
  enrichStepContextWithMockedData(stepContext) {
    var _this$workflowExecuti;
    const contextOverride = (_this$workflowExecuti = this.workflowExecutionState.getWorkflowExecution().context) === null || _this$workflowExecuti === void 0 ? void 0 : _this$workflowExecuti.contextOverride;
    if (contextOverride) {
      stepContext.consts = {
        ...stepContext.consts,
        ...(contextOverride.consts || {})
      };
      stepContext.inputs = {
        ...stepContext.inputs,
        ...(contextOverride.inputs || {})
      };
      stepContext.event = {
        ...stepContext.event,
        ...(contextOverride.event || {})
      };
      stepContext.execution = {
        ...stepContext.execution,
        ...(contextOverride.execution || {})
      };
      stepContext.workflow = {
        ...stepContext.workflow,
        ...(contextOverride.workflow || {})
      };
      if (!stepContext.foreach) {
        stepContext.foreach = contextOverride.foreach;
      }
      Object.entries(contextOverride.steps || {}).forEach(([stepId, stepData]) => {
        if (!stepContext.steps[stepId]) {
          stepContext.steps[stepId] = stepData;
        }
      });
    }
  }
  enrichStepContextAccordingToStepScope(stepContext) {
    let scopeStack = _workflow_scope_stack.WorkflowScopeStack.fromStackFrames(this.workflowExecutionState.getWorkflowExecution().scopeStack);
    while (!scopeStack.isEmpty()) {
      const topFrame = scopeStack.getCurrentScope();
      scopeStack = scopeStack.exitScope();
      const stepExecution = this.workflowExecutionState.getStepExecution((0, _utils.buildStepExecutionId)(this.workflowExecutionState.getWorkflowExecution().id, topFrame.stepId, scopeStack.stackFrames));
      if (stepExecution) {
        switch (stepExecution.stepType) {
          case 'foreach':
            if (!stepContext.foreach) {
              stepContext.foreach = stepExecution.state;
            }
            break;
        }
      }
    }
  }
  getStepData(stepId) {
    const latestStepExecution = this.workflowExecutionState.getLatestStepExecution(stepId);
    if (!latestStepExecution) {
      return;
    }
    return {
      runStepResult: {
        input: latestStepExecution === null || latestStepExecution === void 0 ? void 0 : latestStepExecution.input,
        output: latestStepExecution === null || latestStepExecution === void 0 ? void 0 : latestStepExecution.output,
        error: latestStepExecution === null || latestStepExecution === void 0 ? void 0 : latestStepExecution.error
      },
      stepState: latestStepExecution.state
    };
  }
}
exports.WorkflowContextManager = WorkflowContextManager;