"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NodesFactory = void 0;
var _graph = require("@kbn/workflows/graph");
var _atomic_step_impl = require("./atomic_step/atomic_step_impl");
var _foreach_step = require("./foreach_step");
var _http_step = require("./http_step");
var _if_step = require("./if_step");
var _retry_step = require("./on_failure/retry_step");
var _continue_step = require("./on_failure/continue_step");
var _fallbackStep = require("./on_failure/fallback-step");
var _timeout_zone_step = require("./timeout_zone_step");
var _wait_step = require("./wait_step/wait_step");
var _elasticsearch_action_step = require("./elasticsearch_action_step");
var _kibana_action_step = require("./kibana_action_step");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Import schema and inferred types

class NodesFactory {
  constructor(connectorExecutor,
  // this is temporary, we will remove it when we have a proper connector executor
  workflowRuntime, workflowExecutionState, workflowLogger,
  // Assuming you have a logger interface
  workflowTaskManager, urlValidator, workflowGraph) {
    this.connectorExecutor = connectorExecutor;
    this.workflowRuntime = workflowRuntime;
    this.workflowExecutionState = workflowExecutionState;
    this.workflowLogger = workflowLogger;
    this.workflowTaskManager = workflowTaskManager;
    this.urlValidator = urlValidator;
    this.workflowGraph = workflowGraph;
  }
  create(contextManager) {
    const node = contextManager.node;
    const stepLogger = this.workflowLogger.createStepLogger(this.workflowRuntime.getCurrentStepExecutionId(), node.stepId, node.stepId, node.stepType);

    // Handle elasticsearch.* and kibana.* actions
    if (node.stepType && node.stepType.startsWith('elasticsearch.')) {
      this.workflowLogger.logInfo(`Creating Elasticsearch action step: ${node.stepType}`, {
        event: {
          action: 'internal-action-creation',
          outcome: 'success'
        },
        tags: ['step-factory', 'elasticsearch', 'internal-action']
      });
      return new _elasticsearch_action_step.ElasticsearchActionStepImpl(node, contextManager, this.workflowRuntime, this.workflowLogger);
    }
    if (node.stepType && node.stepType.startsWith('kibana.')) {
      this.workflowLogger.logInfo(`Creating Kibana action step: ${node.stepType}`, {
        event: {
          action: 'internal-action-creation',
          outcome: 'success'
        },
        tags: ['step-factory', 'kibana', 'internal-action']
      });
      return new _kibana_action_step.KibanaActionStepImpl(node, contextManager, this.workflowRuntime, this.workflowLogger);
    }
    switch (node.type) {
      case 'enter-foreach':
        return new _foreach_step.EnterForeachNodeImpl(node, this.workflowRuntime, contextManager, stepLogger);
      case 'exit-foreach':
        return new _foreach_step.ExitForeachNodeImpl(node, this.workflowRuntime, stepLogger);
      case 'enter-retry':
        return new _retry_step.EnterRetryNodeImpl(node, this.workflowRuntime, this.workflowTaskManager, stepLogger);
      case 'exit-retry':
        return new _retry_step.ExitRetryNodeImpl(node, this.workflowRuntime, stepLogger);
      case 'enter-continue':
        return new _continue_step.EnterContinueNodeImpl(node, this.workflowRuntime, stepLogger);
      case 'exit-continue':
        return new _continue_step.ExitContinueNodeImpl(this.workflowRuntime);
      case 'enter-try-block':
        return new _fallbackStep.EnterTryBlockNodeImpl(node, this.workflowRuntime);
      case 'exit-try-block':
        return new _fallbackStep.ExitTryBlockNodeImpl(this.workflowRuntime);
      case 'enter-normal-path':
        return new _fallbackStep.EnterNormalPathNodeImpl(node, this.workflowRuntime, stepLogger);
      case 'enter-fallback-path':
        return new _fallbackStep.EnterFallbackPathNodeImpl(this.workflowRuntime);
      case 'exit-normal-path':
        return new _fallbackStep.ExitNormalPathNodeImpl(node, this.workflowRuntime);
      case 'exit-fallback-path':
        return new _fallbackStep.ExitFallbackPathNodeImpl(node, this.workflowRuntime);
      case 'enter-timeout-zone':
        if ((0, _graph.isEnterWorkflowTimeoutZone)(node)) {
          return new _timeout_zone_step.EnterWorkflowTimeoutZoneNodeImpl(node, this.workflowRuntime, this.workflowExecutionState, contextManager);
        }
        if ((0, _graph.isEnterStepTimeoutZone)(node)) {
          return new _timeout_zone_step.EnterStepTimeoutZoneNodeImpl(node, this.workflowRuntime, this.workflowExecutionState, contextManager);
        }
      case 'exit-timeout-zone':
        if ((0, _graph.isExitWorkflowTimeoutZone)(node)) {
          return new _timeout_zone_step.ExitWorkflowTimeoutZoneNodeImpl(this.workflowRuntime);
        }
        if ((0, _graph.isExitStepTimeoutZone)(node)) {
          return new _timeout_zone_step.ExitStepTimeoutZoneNodeImpl(this.workflowRuntime);
        }
      case 'enter-if':
        return new _if_step.EnterIfNodeImpl(node, this.workflowRuntime, this.workflowGraph, contextManager, stepLogger);
      case 'enter-then-branch':
      case 'enter-else-branch':
        return new _if_step.EnterConditionBranchNodeImpl(node, this.workflowRuntime);
      case 'exit-then-branch':
      case 'exit-else-branch':
        return new _if_step.ExitConditionBranchNodeImpl(node, this.workflowGraph, this.workflowRuntime);
      case 'exit-if':
        return new _if_step.ExitIfNodeImpl(this.workflowRuntime);
      case 'wait':
        return new _wait_step.WaitStepImpl(node, contextManager, this.workflowRuntime, stepLogger, this.workflowTaskManager);
      case 'atomic':
        // Default atomic step (connector-based)
        return new _atomic_step_impl.AtomicStepImpl(node, contextManager, this.connectorExecutor, this.workflowRuntime, stepLogger);
      case 'http':
        return new _http_step.HttpStepImpl(node, contextManager, stepLogger, this.urlValidator, this.workflowRuntime);
      default:
        throw new Error(`Unknown node type: ${node.stepType}`);
    }
  }
}
exports.NodesFactory = NodesFactory;