"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EnterIfNodeImpl = void 0;
var _esQuery = require("@kbn/es-query");
var _eval_kql = require("./eval_kql");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class EnterIfNodeImpl {
  constructor(node, wfExecutionRuntimeManager, workflowGraph, workflowContextManager, workflowContextLogger) {
    this.node = node;
    this.wfExecutionRuntimeManager = wfExecutionRuntimeManager;
    this.workflowGraph = workflowGraph;
    this.workflowContextManager = workflowContextManager;
    this.workflowContextLogger = workflowContextLogger;
  }
  async run() {
    await this.wfExecutionRuntimeManager.startStep();
    this.wfExecutionRuntimeManager.enterScope();
    const successors = this.workflowGraph.getDirectSuccessors(this.node.id);
    if (successors.some(node => !['enter-then-branch', 'enter-else-branch'].includes(node.type))) {
      throw new Error(`EnterIfNode with id ${this.node.id} must have only 'enter-then-branch' or 'enter-else-branch' successors, but found: ${successors.map(node => node.type).join(', ')}.`);
    }
    const thenNode = successors === null || successors === void 0 ? void 0 : successors.find(node => Object.hasOwn(node, 'condition'));
    // multiple else-if could be implemented similarly to thenNode
    const elseNode = successors === null || successors === void 0 ? void 0 : successors.find(node => !Object.hasOwn(node, 'condition'));
    const evaluatedConditionResult = this.evaluateCondition(thenNode.condition);
    if (evaluatedConditionResult) {
      this.goToThenBranch(thenNode);
    } else if (elseNode) {
      this.goToElseBranch(thenNode, elseNode);
    } else {
      // in the case when the condition evaluates to false and no else branch is defined
      // we go straight to the exit node skipping "then" branch
      this.goToExitNode(thenNode);
    }
  }
  goToThenBranch(thenNode) {
    this.workflowContextLogger.logDebug(`Condition "${thenNode.condition}" evaluated to true for step ${this.node.stepId}. Going to then branch.`);
    this.wfExecutionRuntimeManager.navigateToNode(thenNode.id);
  }
  goToElseBranch(thenNode, elseNode) {
    this.workflowContextLogger.logDebug(`Condition "${thenNode.condition}" evaluated to false for step ${this.node.stepId}. Going to else branch.`);
    this.wfExecutionRuntimeManager.navigateToNode(elseNode.id);
  }
  goToExitNode(thenNode) {
    this.workflowContextLogger.logDebug(`Condition "${thenNode.condition}" evaluated to false for step ${this.node.stepId}. No else branch defined. Exiting if condition.`);
    this.wfExecutionRuntimeManager.navigateToNode(this.node.exitNodeId);
  }
  evaluateCondition(condition) {
    if (typeof condition === 'boolean') {
      return condition;
    } else if (typeof condition === 'undefined') {
      return false; // Undefined condition defaults to false
    }
    try {
      return (0, _eval_kql.evaluateKql)(condition, this.workflowContextManager.getContext());
    } catch (error) {
      if (error instanceof _esQuery.KQLSyntaxError) {
        throw new Error(`Syntax error in condition "${condition}" for step ${this.node.stepId}: ${String(error)}`);
      }
      throw error;
    }
  }
}
exports.EnterIfNodeImpl = EnterIfNodeImpl;