"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.HttpStepImpl = void 0;
var _axios = _interopRequireDefault(require("axios"));
var _node_implementation = require("../node_implementation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Extend BaseStep for HTTP-specific properties

class HttpStepImpl extends _node_implementation.BaseAtomicNodeImplementation {
  constructor(node, contextManager, workflowLogger, urlValidator, workflowRuntime) {
    const httpStep = {
      name: node.configuration.name,
      type: node.type,
      spaceId: '',
      // TODO: get from context or node
      with: node.configuration.with
    };
    super(httpStep, contextManager, undefined,
    // no connector executor needed for HTTP
    workflowRuntime);
    this.workflowLogger = workflowLogger;
    this.urlValidator = urlValidator;
  }
  getInput() {
    const context = this.contextManager.getContext();
    const {
      url,
      method = 'GET',
      headers = {},
      body
    } = this.step.with;
    return {
      url: typeof url === 'string' ? this.templatingEngine.render(url, context) : url,
      method,
      headers: this.renderHeaders(headers, context),
      body: this.renderBody(body, context)
    };
  }
  renderHeaders(headers, context) {
    return Object.entries(headers).reduce((acc, [key, value]) => {
      acc[key] = typeof value === 'string' ? this.templatingEngine.render(value, context) : value;
      return acc;
    }, {});
  }
  renderBody(body, context) {
    if (typeof body === 'string') {
      return this.templatingEngine.render(body, context);
    }
    if (body && typeof body === 'object') {
      return this.renderObjectTemplate(body, context);
    }
    return body;
  }

  /**
   * Recursively render the object template.
   * @param obj - The object to render.
   * @param context - The context to use for rendering.
   * @returns The rendered object.
   */
  renderObjectTemplate(obj, context) {
    if (Array.isArray(obj)) {
      return obj.map(item => this.renderObjectTemplate(item, context));
    }
    if (obj && typeof obj === 'object') {
      return Object.entries(obj).reduce((acc, [key, value]) => {
        acc[key] = this.renderObjectTemplate(value, context);
        return acc;
      }, {});
    }
    if (typeof obj === 'string') {
      return this.templatingEngine.render(obj, context);
    }
    return obj;
  }
  async _run(input) {
    try {
      return await this.executeHttpRequest(input);
    } catch (error) {
      return await this.handleFailure(input, error);
    }
  }
  async executeHttpRequest(input) {
    const {
      url,
      method,
      headers,
      body
    } = input;

    // Validate that the URL is allowed based on the allowedHosts configuration
    try {
      this.urlValidator.ensureUrlAllowed(url);
    } catch (error) {
      this.workflowLogger.logError(`HTTP request blocked: ${error.message}`, error instanceof Error ? error : new Error(String(error)), {
        workflow: {
          step_id: this.step.name
        },
        event: {
          action: 'http_request',
          outcome: 'failure'
        },
        tags: ['http', 'security', 'blocked']
      });
      throw error;
    }
    this.workflowLogger.logInfo(`Making HTTP ${method} request to ${url}`, {
      workflow: {
        step_id: this.step.name
      },
      event: {
        action: 'http_request',
        outcome: 'unknown'
      },
      tags: ['http', method.toLowerCase()]
    });
    const config = {
      url,
      method,
      headers,
      signal: this.contextManager.abortController.signal
    };
    if (body && ['POST', 'PUT', 'PATCH'].includes(method)) {
      config.data = body;
    }
    const response = await (0, _axios.default)(config);
    this.workflowLogger.logInfo(`HTTP request completed with status ${response.status}`, {
      workflow: {
        step_id: this.step.name
      },
      event: {
        action: 'http_request',
        outcome: 'success'
      },
      tags: ['http', method.toLowerCase()]
    });
    return {
      input,
      output: {
        status: response.status,
        statusText: response.statusText,
        headers: response.headers,
        data: response.data
      },
      error: undefined
    };
  }
  async handleFailure(input, error) {
    let errorMessage;
    let isAborted = false;
    if (_axios.default.isAxiosError(error)) {
      if (error.code === 'ERR_CANCELED') {
        errorMessage = 'HTTP request was cancelled';
        isAborted = true;
      } else if (error.response) {
        errorMessage = `${error.response.status} ${error.response.statusText}`;
      } else {
        errorMessage = `${error.message ? error.message : error.name}`;
      }
    } else if (error instanceof Error) {
      errorMessage = error.message;
      // Check if this is an AbortError
      if (error.name === 'AbortError') {
        isAborted = true;
      }
    } else {
      errorMessage = String(error);
    }
    this.workflowLogger.logError(`HTTP request failed: ${errorMessage}`, error instanceof Error ? error : new Error(errorMessage), {
      workflow: {
        step_id: this.step.name
      },
      event: {
        action: 'http_request',
        outcome: 'failure'
      },
      tags: isAborted ? ['http', 'cancelled'] : ['http', 'error']
    });
    return {
      input,
      output: undefined,
      error: errorMessage
    };
  }
}
exports.HttpStepImpl = HttpStepImpl;